/*
  This source is part of PCOak, an electronic mailer for DOS based on PCElm.

  PCElm is Copyright (c) 1988-1993 Martin Freiss and Wolfgang Siebeck
           Copyright (c) 1992-1999 Demon Internet
  PCOak is Copyright (c) 2000-2001 Simon Turner, Pete Disdale and dispc members

  Thanks to an agreement between the original PCElm authors and Demon Internet
  made in late 1999:

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License, version 1, as
	published by the Free Software Foundation.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	See the file COPYING, which contains a copy of the GNU General
	Public License.
*/

/*
 * expand aliases for PCOak
 */

#include <stdio.h>
#include <conio.h>
#include <time.h>
#include <ctype.h>
#include <dos.h>
#include <sys/stat.h>
#include <string.h>
#include "pcoak.h"
#include "filename.h"
#include "ustring.h"
#include "chars.h"

/*
 * Global variables defined elsewhere and used here
 */
extern const char cant_create[];	/* "Can't create " */
extern BOOL slash_ed;		/* Use / on editor filename instead of \ */

/*
 * Functions defined elsewhere and used in this module
 */
char *next_address(char *into, size_t intosize, const char *from,
		   char **addr_ptr, size_t *addr_len);
char rip(char *s);
void fwdslash(char *s);
FILE *invoke_editor(FILE *tfile, const char *to, const char *subject,
		    BOOL mailing, BOOL reopen);
void errordsp(const char *txt);


/*
 * aliasfile()
 *
 * Fill in the full path name of the alias file in <fullpath>
 */
static void aliasfile(char *fullpath, size_t size)
{
    PUSTRLCPY3(fullpath, size, homedir, "\\", ALIASFILE);
}


/*
 * expand_alias()
 *
 * Input to this function is a line containing (multiple) recipients; we split
 * them up into invididual addresses by iteratively calling next_address() to
 * copy each address in turn into a local buffer.  Addresses are expected to
 * be in one of the following forms:
 *
 *   (a)  simon@twoplaces.co.uk (Simon Turner)
 *   (a1) simon@twoplaces.co.uk				[subset of (a)]
 *   (b)  Simon Turner <simon@twoplaces.co.uk>
 *   (b1) "Simon Turner" <simon@twoplaces.co.uk>	[alternative to (b)]
 *   (b2) <simon@twoplaces.co.uk>			[subset of (b)]
 *   (c)  simon
 *
 * We do alias expansion on the entire address, considered as a single string.
 * Since it is illegal to have whitespace in an alias, an address which
 * contains whitespace cannot be an alias.  That will take care of (a), (b)
 * and (b1) in the examples above; and it's OK to apply alias checks to (a1)
 * and (b2), since they are legal aliases -- and if the user has decided to
 * alias "simon@twoplaces.co.uk" to "spammer@juno.com", that's up to him/her.
 * Therefore, we apply alias checks to an address if it contains no
 * whitespace; if it contains whitespace, we just pass it through to the final
 * list verbatim.
 *
 * Alias expansion for a particular address is done by running through all
 * aliases defined in the ALIASFILE, looking for a match: if a match is found,
 * the full text for that alias is copied to the result list instead of the
 * address extracted from <st>; we don't currently support recursive aliases
 * directly (and we never have).
 *
 * Once we've processed the entire address list, we copy the result back into
 * <st> to that the calling function gets the expanded version, returning 0 if
 * there was enough room (so the list is complete), or 1 if we ran out of
 * room.
 */
int expand_alias(char *st, size_t stsize)
{
    const char *sep = ", ";		/* Address separator string */
    FILE *afile;
    char line[LLINELEN];
    char workstr[LLINELEN], returnstr[2 * LLINELEN + 2], *cp, *aliasptr;
    int foundalias;
    BOOL ok = 1;

    aliasfile(line, sizeof(line));
    if ((afile = fopen(line, "r")) == NULL)
	return 0;		/* no aliases found */

    if (stsize >= sizeof(returnstr))	/* Reduce size to match our array */
	stsize = sizeof(returnstr) - 1;

    *returnstr = '\0';
    cp = st;
    while (ok && (cp = next_address(workstr, sizeof(workstr), cp,
				    NULL, NULL)) != NULL)
    {
	foundalias = FALSE;
	if (strpbrk(workstr, chs_lwsp) == NULL)	/* may be an alias */
	{
	    rewind(afile);		/* goto BOF */
	    while (fgets(line, sizeof(line), afile) != NULL)
	    {
		if (line[0] == '#')	/* comment */
		    continue;
		if (foundalias && !(isspace(line[0])))
		    break;	/* not continuation line: stop now */

		/*
		 * If <foundalias> is true, this is a continuation line of a
		 * matching alias: there's nothing to check.  Otherwise, we
		 * need to find the first word (delimited by whitespace) and
		 * check that against the current address in <workstr>
		 */
		(void) rip(line);
		if ((aliasptr = strpbrk(line, chs_lwsp)) != NULL)
		{
		    if (!foundalias)		/* Examine alias name */
			*aliasptr++ = '\0';
		    if (foundalias || stricmp(line, workstr) == 0)
		    {		/* Match: skip spaces, add to list */
			while (*aliasptr && isspace(*aliasptr))
			    aliasptr++;
			if (*returnstr)		/* not the first entry */
			{
			    /* ensure we've ONE ", " sep: kill trailing seps */
			    kill_trail_sep(returnstr);
			    AUSTRLCAT2(returnstr, sep, aliasptr);
			}
			else			/* first entry */
			    AUSTRCPY(returnstr, aliasptr);
			foundalias = TRUE;
			if (strlen(returnstr) >= stsize)	/* Too long! */
			{
			    ok = FALSE;
			    break;	/* out of the while (fgets) */
			}
		    }
		}
	    }
	}
	if (ok && !foundalias)	/* No alias found: copy address verbatim */
	{
	    if (*returnstr)		/* not the first entry */
	    {
		/* ensure we've ONE ", " separator: kill trailing sep stuff */
		kill_trail_sep(returnstr);
		AUSTRLCAT2(returnstr, sep, workstr);
	    }
	    else			/* first entry */
		AUSTRCPY(returnstr, workstr);
	    if (strlen(returnstr) >= stsize)	/* Too long! */
		ok = FALSE;
	}
    }

    if (!ok)		/* Ran out of room */
    {
	errordsp("Alias expansion fails, not sent.");
	(void) fclose(afile);				/* OK: read only */
	sleep(2);
	return 1;
    }

    /*
     * Remove trailing separator characters (blanks and commas), skip leading
     * separator debris, copy into <st> ready to return to calling function
     */
    kill_trail_sep(returnstr);
    PUSTRCPY(st, stsize, skip_sep(returnstr));
    (void) fclose(afile);				/* OK: read only */
    return 0;
}


/*
 * edit_aliases()
 *
 * Fire up the user's editor on the alias file; if it doesn't exist, try to
 * create it; if that fails, write an appropriate error message into <errstr>
 * (which is <errsize> bytes long) and return immediately.
 */
void edit_aliases(char *errstr, size_t errsize)
{
    FILE *afile;
    char afname[LLINELEN];
    time_t secsnow;
    struct stat sb;

    aliasfile(afname, sizeof(afname));
    if (stat(afname, &sb) || sb.st_size == 0L)
    {
	BOOL fail = TRUE;	/* Assume the worst for now */

	if ((afile = fopen(afname, "w")) != NULL)
	{
	    secsnow = time(0L);
	    if (fprintf(afile, "#\n# aliases created by %s, %s#\n",
			myversion, ctime(&secsnow)) != EOF)
		fail = FALSE;		/* Wa-hey! */
	    if (fclose(afile) != 0)
		fail = TRUE;
	}
	if (fail)
	{
	    PUSTRLCPY2(errstr, errsize, cant_create, afname);
	    return;
	}
    }
    if (slash_ed)
	fwdslash(afname);
    (void) invoke_editor(NULL, afname, NULL, FALSE, FALSE);
}
