/*
  This source is part of PCOak, an electronic mailer for DOS based on PCElm.

  PCElm is Copyright (c) 1988-1993 Martin Freiss and Wolfgang Siebeck
           Copyright (c) 1992-1999 Demon Internet
  PCOak is Copyright (c) 2000-2003 Simon Turner, Pete Disdale and dispc members

  Thanks to an agreement between the original PCElm authors and Demon Internet
  made in late 1999:

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License, version 1, as
	published by the Free Software Foundation.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	See the file COPYING, which contains a copy of the GNU General
	Public License.
*/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <conio.h>
#include <dir.h>
#include <dos.h>
#include <string.h>
#include <mem.h>
#include "pcoak.h"
#include "header.h"
#include "keyboard.h"
#include "ustring.h"
#include "chars.h"
#include "macros.h"
#include "display.h"

#define TABSIZE		8	/* expand tabs to 8 blanks */

#define BOUNDLEN	72	/* Max. length of a multipart boundary is 70 */
#define MAX_HDRS	5	/* Max. no. of MIME header lines to store */
#define MAX_MULTI	5	/* Max. depth of MIME multiparts supported */

#define FIXED_TOPLINES	2	/* No. of non-scrolling lines at top */
#define PROMPT_BOTLINES	1	/* No. of prompt-only lines at bottom */

#define NO_ACTION	0	/* "Action" codes from handle_entity() */
#define STOP_NOW	1
#define SKIP_START	2
#define SHOW_FULL	3
#define SHOW_REMAINS	4
#define REDRAW_REMAINS	5
#define END_MESSAGE	6
#define HANDLE_KEY	7

struct multipart {	/* Describe a MIME multipart entity */
    CTTYPE type;		/* CT_MULTI_MIXED/ALTERN/DIGEST/MISC */
    char boundary[BOUNDLEN];	/* boundary delimiter string */
    unsigned nparts;		/* no. of parts found so far */
};

typedef struct slineinfo {	/* Describe a line within the viewer */
    long fpos;
    unsigned char type;
} LINEINFO;

#define LT_UNKNOWN	0	/* Unknown line type */
#define LT_NORMAL	1	/* Normal body line */
#define LT_QPTEXT	2	/* Quoted-printable text body line */
#define LT_QPBIN	3	/* Quoted-printable binary body line */
#define LT_64TEXT	4	/* Base64 text body line */
#define LT_64BIN	5	/* Base64 binary body line */
#define LT_ENDMSG	255	/* End of the message */

/*
 * Global variables local to this module
 */
static int local_tabsize;	/* 4 or 8 */
static struct multipart multipart[MAX_MULTI]; /* store for MIME multiparts */
static int multidepth;			      /* multipart nesting depth */

static const char decode_err[]     = "Decode error";
static const char atend_prompt[]   = "Command? (You have seen 100%)";
static const char nomore_search[]  = "No more searching: already at end";
static const char illegal_qp[]     =
    "Illegal =XX q-p sequences were found in text";
static const char blank_to_abort[] =
    "Blank the field (press ESC twice) to abort";

/*
 * A system for describing what lines are currently drawn on in the view
 * window
 */
static int winlines[50];	/* Index value for all possible window lines */
static int winlen;		/* No. of lines in the window */
static int winbot;		/* Index of bottom line in the window */
static size_t winscrollsize;	/* No. of bytes in (window - 1 line) */
static int wincur;		/* Line that cursor is currently on */

#define WL_BLANK	-1	/* winlines[] idx: line is blank */
#define WL_STATUS	-2	/* winlines[] idx: line has status info */

#define TOPLNDR		winlines[0]	/* index value for the top line */

static void win_setlen(int nlines);
static void win_init(void);
static void win_curidx(int val);
static void win_newline(void);
static void win_clrline(int y);
static void win_scrollup(void);
static void win_scrolldown(void);
static void win_gotoline(int y);
static void win_defline(int y, int val);
static void win_clrscr(void);
static int win_last(void);

/*
 * External global variables used in this module
 */
extern const char tempfile_err[];	/* "Error opening tempfile" */
extern const char no_tmpfile[];		/* "Couldn't create temp file" */
extern const char no_mail[];		/* "No mail" */
extern const char no_mem[];		/* "Not enough memory" */
extern const char search_prompt[];	/* "Search for what? " */
extern const char somebody[];		/* "Somebody" */
extern const char bracket_none[];	/* "(none)" */
extern const char copying_msg[];	/* "copying message" */
extern int weedlevel;		/* Level of weedout to apply (0-3) (Chgable) */
extern char *quotestr;		/* for quoting include msgs in R)eply */
extern char *pager;		/* user's favourite pager */
extern char *gifview;		/* user's favourite gif viewer */
extern char *jpegview;		/* user's favourite jpeg viewer */
extern char *mpegview;		/* user's favourite mpeg viewer */
extern char *audioview;		/* user's favourite audio player */
extern char *miscview;		/* user's favourite miscellaneous viewer */
extern char *multimedia;	/* user's favourite multimedia program */
extern int extpager;		/* External pager usage (0-2) (Changeable) */
extern char *sequencename;	/* name of sequence file */
extern BOOL retendquit;		/* Quit the pager with <Return> at end? */

/*
 * Functions defined and used in this module
 */
static long decode(FILE *mmfile, long end, CETYPE code_type,
		   const char *filename);
int savemsg(PHTL cmsg, FILE *tfile, int weed, unsigned options);
int check_status(char *buf, size_t bufsize, FILE *fp, BOOL quote);
void verify_status_line(char *buf, size_t bufsize);
int write_status_line(FILE *fp, BOOL quote);
int mlock(const char *dir, const char *id);
void rmlock(const char *dir, const char *id);
static int uu_begin(const char *linestr, char *namebuf, size_t namebufsize);
static int extprog(const char *cmdstr, const char *filename,
		   const char *action, const char *desc);
FILE *opentemp(char *name, const char *mode);
static int decode_qpline(char *line, BOOL hardbreak);
static void get_content_info(PCHTL cmsg, FILE *fp, char *type,
			     size_t typesize, char *enc, size_t encsize);
static int multi_check(const char *line, CTTYPE type);
static int isboundary(const char *line, int *isend);
static int handle_entity(FILE *efp, char *temp, size_t tempsize, long end,
			 unsigned char *linetype, int *chp);
static unsigned char mime_linetype(CTTYPE type, CETYPE encoding);
static long redraw_view(PCHTL cmsg, int screen_lines, int eohlndr,
			const LINEINFO *dr, BOOL *inheadptr,
			char *tstring, size_t tstring_size);
static int last_line(const LINEINFO *dr, long end, FILE *fp);
static int file_percent(const LINEINFO *dr, long end, long totalsize,
			int idx, long fpos, int endidx);
static void draw_viewscreen(PCHTL cmsg);

/*
 * Functions defined elsewhere and used in this module
 */
char maxgets(char *s, int maxlen);
int getwin(int x, int y, int maxlen, int winattr, char *input,
	   const char *prompt, ...);
void winscrl(int nolines);
void shellcmd(void);
char rip(char *s);
char eoltest(const char *s);
void get_cr(const char *msg);
BOOL y_n(const char *msg);
int view_cr(const char *msg);
int ignore_alt(const char *msg);
char *buildcmd(const char *cmdline, const char *rcvr, const char *subject,
	       const char *fname);
HEADERTYPE htype(const char *p);
BOOL htype_match(const char *p, HEADERTYPE type);
const char *hdr_body(const char *field, HEADERTYPE type);
void writef(int x, int y, int attr, const char *line);
char *seqstr(size_t n, char ch);
void quickcenter(int line, int attr, const char *txt);
void errordsp(const char *txt);
void errorbox(const char *title, ...);
void fopen_error(const char *filename, const char *descrip);
void write_error(const char *filename, const char *descrip);
CTTYPE get_content_type(const char *tstring);
CETYPE get_content_encoding(const char *tstring);
unsigned int getkey(void);
int get_addr_filename(char *into, size_t intosize, const char *addr);
char *stristr(const char *s1, const char *s2);
int getstring(char *dst, size_t dstsize, const char *src, int equals);
int wildmat(const char *text, const char *p, int ignore_case);
const char *select_max(int maxnum);
void weedset(int lvl);
int weedit(const char *line, long fpos);
BOOL weedlineisnew(void);
BOOL weedlevelok(int lvl);


/*
 * decode()
 *
 * Decode the encoded attachment on <mmfile> to <filename>; the attachment is
 * ended either by reaching the end-of-file, reaching file position <end>, or
 * by the presence of a known multipart boundary "--<boundary>" at the start
 * of a line (use isboundary() to check).  <code_type> is the type of encoding
 * used for the attachment.  Return the position in <mmfile> at which the
 * final boundary line started, or < 0 on error.
 *
 * Remember that, for plain text and quoted-printable, the newline on the end
 * of the line is perceptually regarded as belonging to the *next* line of the
 * body: the boundary delimiter is actually "CRLF--<boundary>", and if there
 * is no blank line between the last text line and the boundary, the last text
 * line DOES NOT END WITH A NEWLINE.  To this end, for the q-p and identity
 * transforms, we don't write out the terminating newline, but actually
 * precede the *next* line with a newline, once we're sure that the boundary
 * hasn't turned up.  This doesn't affect the base64 (or uuencode) encodings,
 * because they don't use newlines as such anyway -- their output is more
 * likely to be binary than text.
 */
static long decode(FILE *mmfile, long end, CETYPE code_type,
		   const char *filename)
{
    char tstring[LINELEN], *p;
    const char *mode;
    unsigned char vals[4];
    int j, found_start = FALSE;
    long dr;
    FILE *wfile;
    BOOL b64stop = FALSE;
    unsigned qperrs = 0;
    BOOL newline = FALSE;	/* don't prepend a newline yet */
    char last_eol = 1;
    BOOL writerr = FALSE;

    if (code_type == CE_BASE64 || code_type == CE_UUENCODE ||
	code_type == CE_8BIT || code_type == CE_BINARY)	/* binary output */
	mode = "wb";
    else						/* text output */
	mode = "wt";

    if ((wfile = fopen(filename, mode)) == NULL)
    {
	fopen_error(filename, "decoded file");
	return -1L;
    }

    for (dr = ftell(mmfile);
	 dr < end && fgets(tstring, sizeof(tstring), mmfile) != NULL;
	 dr = ftell(mmfile))
    {
	/*
	 * We no longer rip the line immediately, so that we can spot soft
	 * line breaks in quoted-printable stuff
	 */
	if (last_eol)		/* This is a new line */
	{
	    if (code_type == CE_UUENCODE && strcmp(tstring, "end\n") == 0)
	    {
		dr = ftell(mmfile);	/* note start of line *after* "end" */
		break;
	    }
	    if (isboundary(tstring, NULL) >= 0)	/* a multipart boundary */
	    {
		/* dr already set to the file pos at the start of this line */
		break;
	    }
	}

	if (code_type == CE_BASE64)		/* MIME base64 decode */
	{
	    char ch;

	    last_eol = rip(tstring);
	    if (b64stop)		/* stop now anyway */
		break;
	    if (!*tstring)		/* blank line */
		if (found_start)
		    break;
		else
		    continue;    /* skip past blank lines at the start */
	    found_start = TRUE;
	    p = tstring;
	    while (*p && !b64stop)
	    {
		for (j = 0; j < 4; j++, p++)
		{
		    vals[j] = 0;
		    if (*p == '=')	/* pad character: stop decoding */
		    {
			b64stop = TRUE;
			break;
		    }
		    else if (*p >= 'A' && *p <= 'Z')
			vals[j] = *p - 'A';
		    else if (*p >= 'a' && *p <= 'z')
			vals[j] = *p - 'a' + 26;
		    else if (*p >= '0' && *p <= '9')
			vals[j] = *p - '0' + 52;
		    else if (*p == '+')
			vals[j] = 62;
		    else if (*p == '/')
			vals[j] = 63;
		    else if (isspace(*p))	/* ignore unexpected space */
			j--;
		    else		/* illegal character: stop! */
		    {
			char errbuf[32];
			sprintf(errbuf, "  Char %d (%c)", (int) *p, *p); /*OK*/
			errorbox(decode_err, "Illegal base64 character found:",
				 errbuf, "Abandoning decode", NULL);
			(void) fclose(wfile);		/* OK: giving up */
			return dr;
		    }
		}
		/* got 4 bytes - now extract values and write 1-3 bytes out */
		ch = (char) ((vals[0] << 2) | (vals[1] >> 4));
		(void) fputc(ch, wfile);	/* 1st byte always written */
		if (j > 2)
		{
		    ch = (char) ((vals[1] << 4) | (vals[2] >> 2));
		    (void) fputc(ch, wfile);	/* 2nd byte if <= 1 '=' char */
		    if (j > 3)
		    {
			ch = (char) ((vals[2] << 6) | (vals[3]));
			(void) fputc(ch, wfile); /* 3rd byte if no '=' chars */
		    }
		}
	    }
	}
	else if (code_type == CE_UUENCODE)		/* UU decode */
	{
	    int n;
	    char ch;

#define DEC(c)	(((c) - ' ') & 077)		/* single character decode */
	    last_eol = rip(tstring);
	    p = tstring;
	    /*
	     * The first character in the line tells us how many bytes to
	     * write for this line (it is encoded itself, of course); we
	     * expect the last encoded line to be a single character which
	     * decodes to 0.
	     */
	    if ((n = (int) DEC(*p)) == 0)	/* No chars to write out */
		continue;
	    for (++p; n > 0; p += 4, n -= 3)
	    {
		ch = (char) ((DEC(p[0]) << 2) | (DEC(p[1]) >> 4));
		(void) fputc(ch, wfile);
		if (n > 1)
		{
		    ch = (char) ((DEC(p[1]) << 4) | (DEC(p[2]) >> 2));
		    (void) fputc(ch, wfile);
		    if (n > 2)
		    {
			ch = (char) ((DEC(p[2]) << 6) | (DEC(p[3])));
			(void) fputc(ch, wfile);
		    }
		}	
	    }
	}
	else				/* quoted-printable, or no decoding */
	{
	    last_eol = eoltest(tstring);
	    if (code_type == CE_QUOTED)		/* quoted-printable */
	    {
		if (decode_qpline(tstring, FALSE))
		    qperrs++;
	    }
	    if (newline)		/* prepend a newline to this line */
		(void) fputs("\n", wfile);
	    if ((p = tstring + strlen(tstring) - 1) >= tstring && *p == '\n')
	    {
		newline = TRUE;		/* need a newline next time around */
		*p = '\0';
	    }
	    else
		newline = FALSE;	/* no newline before next line */
	    (void) fputs(tstring, wfile);
	}
	/* check for write errors every 4 bytes or line */
	if (ferror(wfile))
	{
	    writerr = TRUE;
	    break;	/* out of the for (dr) */
	}
    }

    if (fclose(wfile) != 0)
	writerr = TRUE;

    if (writerr)		/* Error while writing or closing file */
    {
	write_error(filename, "decoding");
	return -1L;
    }

    if (qperrs > 0)	/* Some illegal quoted-printable sequences seen */
	errorbox(decode_err, illegal_qp, NULL);

    return dr;
}


static int hex_to_dec(char p)
{
    if (isdigit(p))
	return p - '0';
    p = (char) tolower(p);
    if (p >= 'a' && p <= 'f')
	return p - 'a' + 10;
    return 0;
}


/*
 * print_string()
 *
 * Display string <tstring> at the current screen position, which involves (a)
 * expanding TABs to use spaces instead (to get character spacing right), and
 * (b) decoding quoted-printable bits (if <qpdecode> is true); this means
 * changing the contents of the buffer, whose size <tstringsize> we are told
 * so we can make sure we don't overrun it in our enthusiasm.
 *
 * <here> is the position in the main file stream <mfile> for the beginning of
 * this line; if the line is q-p and ends in "=\n", we may need to append
 * one or more extra line(s) in order to get a full display line.
 *
 * If the text runs past the end of the screen line, we note the fact and
 * ensure that the file pointer is positioned such that the next line read
 * will begin with the appropriately wrapped character -- when this function
 * returns, the stream pointer will always be at the correct position to read
 * the next line to display.
 *
 * If <pattern> is non-NULL, compare the text of this line (and any other that
 * get appended during q-p decoding) with the pattern, and display the entire
 * line in a different colour if it matches; return TRUE if we get a match,
 * FALSE if not (including if the pattern is NULL).
 */
static BOOL print_string(char *tstring, size_t tstringsize, FILE *fp,
			 long here, const char *pattern, BOOL qpdecode)
{
    char *p;
    int n, n_wrap;
    const char *end_ptr = tstring + tstringsize - 2 * local_tabsize;
    const char *wrap_ptr = tstring + display.screenwidth;
    char *match = NULL;
    long here2 = -1L, here3 = -1L;
    int n2 = -1, n3 = -1;
    BOOL is_qp;
    BOOL do_bell;
    size_t l;

    /*
     * Line wrapping is handled by noting how many bytes into the original
     * string we were when we reached the end of the screen line, then
     * positioning the file pointer to that point (forcing the "lost" text to
     * be re-read next time around).  We keep track of this with <n> (no. of
     * characters processed so far) and <n_wrap> (no. processed at wrap
     * point).
     *
     * Making sure we wrap properly is a bit of a sod.  An 80-char line
     * followed by a newline obviously shouldn't wrap before the newline; but
     * what if the 81st char is actually a q-p "=\n" (soft line break)
     * followed by a real one?  Or a single whitespace character followed by a
     * newline?  We normally remove trailing whitespace from a line before
     * printing it, so a line which goes "word| \n" (where the | marks the end
     * of the screen line) should probably have the single trailing space
     * ignored, in which case it fits into the 80-char screen line.  What
     * about a q-p soft-linebreak line which is followed by a line consisting
     * of a single space, e.g. "word|=\n" followed by "=20\n", which -- once
     * the q-p stuff has been decoded -- is "word| \n" as above?
     *
     * Even worse is what to do if we come across "unusual" control chars;
     * I've seen a raw CR, as well as backspace(!).  These have to be handled
     * carefully.
     */
    p = tstring;
    n = n_wrap = 0;
    do_bell = FALSE;

    while (*p && p < end_ptr)    /* Safe to carry on expanding (for now) */
    {
	/*
	 * Do any q-p decoding of <p> first, so that we're working with real
	 * characters later on
	 */
	is_qp = FALSE;
	if (qpdecode && *p == '=')		/* q-p "special" */
	{
	    if (strcmp(p, "=\n") == 0)	/* end of line: append new one */
	    {
		int nx = n;
		long herex = ftell(fp);
		if (fgets(p, (int) tstringsize - (int) (p - tstring), fp) !=
		    NULL)		/* got next line OK */
		{
		    if (n2 < 0)		/* store values in n2, here2 */
		    {
			n2 = nx;
			here2 = herex;
		    }
		    else if (n3 < 0)	/* store values in n3, here3 */
		    {
			n3 = nx;
			here3 = herex;
		    }
		    else		/* copy *3 -> *2; store in n3, here3 */
		    {
			n2 = n3;
			here2 = here3;
			n3 = nx;
			here3 = herex;
		    }
		}
		else			/* no next line: stop here */
		{
		    *p = '\0';
		    break;	/* out of the while (*p) */
		}
	    }
	    if (*p == '=' && isxdigit(p[1]) && isxdigit(p[2]))	/* =AA code */
	    {
		char ch = (char) (hex_to_dec(p[1]) * 16 + hex_to_dec(p[2]));
		if (ch == 163)	/* convert u-acute to pound: assumes */
		    ch = 156;	/* ISO-8859-1 (US-ASCII shouldn't be > 127) */
		/* ST-FIXME: this support for ISO-8859-1 is dreadful! */
		*p = ch;
		memmove(p + 1, p + 3, strlen(p + 3) + 1);	/* OK */
		n += 2;		/* we've skipped two bytes in the file */
		is_qp = TRUE;
	    }
	}

	/*
	 * We're about the display <p>; it may need careful handling, or we
	 * may need to wrap the line at this point.  The following characters
	 * need special handling rather than just being displayed "raw":
	 *	0x07	\a	Bell; no printed character
	 *	0x08	\b	Backspace; overwrite previous char
	 *	0x09	\t	Tab; expand to spaces internally
	 *	0x0a	\n	Newline; stop line (may continue after?)
	 *	0x0d	\r	Carriage Return; back to start of line
	 * We actually treat \r the same as a newline, because mail generated
	 * on a maggotbox may expect that to happen.
	 */
	if (*p == '\b')			/* Backspace */
	{
	    if (p > tstring)	/* Overwrite <p - 1> */
	    {
		memmove(p - 1, p + 1, strlen(p + 1) + 1);	/* OK */
		p--;
	    }
	    else		/* Just ignore this backspace */
		memmove(p, p + 1, strlen(p + 1) + 1);		/* OK */
	    n++;		/* we've skipped one byte in the file */
	    continue;	/* with the while (*p) -- restart at this new char */
	}
	if (*p == '\a')			/* Bell: ignore it, but sound bell */
	{
	    do_bell = TRUE;
	    memmove(p, p + 1, strlen(p + 1) + 1);		/* OK */
	    n++;		/* we've skipped one byte in the file */
	    continue;	/* with the while (*p) -- restart at this new char */
	}
	if (*p == '\n' || *p == '\r')	/* End of line: stop here */
	{
	    if (p[1] != '\0')		/* It carries on beyond the newline */
		n_wrap = n + 1;	/* note no. of file chars before wrap */
	    *p ='\0';
	    break;	/* out of the while (*p) */
	}
	if (n_wrap == 0 && p >= wrap_ptr && (p > wrap_ptr || strcmp(p, " \n")))
	{				/* Wrapping: note posn. */
	    /* <p> should be the first character of the next line */
	    n_wrap = n;		/* note no. of file chars before wrap */
	    if (is_qp)		/* back to start of =AB (currently at 'B') */
		n_wrap -= 2;
	    break;	/* out of the while (*p) */
	}

	/*
	 * This character goes on this line: expand tabs and move on to the
	 * next one
	 */
	if (*p == '\t')			/* expand this TAB */
	{
	    int len = local_tabsize - ((int) (p - tstring) % local_tabsize);
	    /* len is always > 0 */
	    memmove(p + len, p + 1, strlen(p) + 1);	/* OK */
	    memset(p, ' ', (size_t) len);
	    p += len;
	}
	else
	    p++;
	n++;
    }

    if (pattern && (match = stristr(tstring, pattern)) != NULL)
	if ((match - tstring) > display.screenwidth)	/* past end of line */
	    match = NULL;

    /*
     * We want to end up at the start of the next line after displaying this
     * one.  If this string is too long to fit on one line (even after
     * removing trailing whitespace), truncate it after the last character on
     * the line, and seek the file pointer to the first character to display
     * on the next line (<n_wrap> tells us where this is) -- the resulting
     * line will exactly fill a line, and will cause the cursor to wrap on to
     * the start of the next line.  If, however, the string is too short to
     * fill the entire line, we force a CRLF on to the end to make the cursor
     * end up on the next line after displaying this one.  It is assumed that
     * the underlying screen line is already blank, so we don't bother with
     * time consuming clreol() calls or anything.
     */
    kill_trail_lwsp(tstring);		/* remove trailing spaces */
    if ((l = strlen(tstring)) > display.screenwidth ||	/* too long for line */
	n_wrap > 0)					/* forced wrap */
    {
	tstring[display.screenwidth] = '\0';
	if (here3 >= 0L && n_wrap > n3)		/* wrapped in 3rd string */
	    fseek(fp, here3 + n_wrap - n3, SEEK_SET);
	else if (here2 >= 0L && n_wrap > n2)	/* wrapped in 2nd string */
	    fseek(fp, here2 + n_wrap - n2, SEEK_SET);
	else					/* wrapped in 1st string */
	    fseek(fp, here + n_wrap, SEEK_SET);
    }
    if (l < display.screenwidth)			/* too short to wrap */
	PUSTRCAT(tstring, tstringsize, "\r\n");
    textattr(match ? col_hilite : col_normal);
    if (do_bell)
	cputs("\a");
    cputs(tstring);
    textattr(col_normal);
    return (match != NULL);
}


/*
 * doviewmsg()
 *
 * Main function for viewing a message from the current open file; it's entry
 * <cmsg> in the linked list.
 */
static int doviewmsg(PHTL cmsg)
{
    char *tstring, *t1;
    size_t tstring_size, t1_size;
    static char search_item[LINELEN];
    long end, totalsize, percent;
    int screen_lines, bottom_line, going_down;
    int ch;			/* ch must be int for KF1 etc. */
    BOOL searching, found, inheader, atend;
    BOOL checked_uuencode;
    int loc_weedlevel = weedlevel;	/* Local weedout level to apply */
    int opp_weedlevel;			/* Opposive level to <weedlevel> */
    unsigned char linetype;	/* Type of data in current message/entity */
    LINEINFO *dr;	/* Info (file pos, type) about each screen line */
    long here;		/* File position at start of current line */
    int lndr, eohlndr, endlndr, maxlndr;
    const LINEINFO *cdr;
    int i;		/* generic loop variables; nothing special */
    int found_boundary;
    int init_multidepth = multidepth;
    BOOL mpskipping, mpcommand;

    if (mfile == NULL)
    {
	textcolor(col_enhanc);
	cprintf("%s\n", no_mail);
	return 0;
    }
    /*
     * Make <tstring> large enough to allow tab expansion, i.e. make it
     * TABSIZE times bigger than the no. of characters we're going to read
     * into it.
     */
    if ((tstring = malloc(tstring_size = TABSIZE * LINELEN)) == NULL)
    {
	cprintf("%s\n", no_mem);
	sleep(3);
	return 0;
    }
    /*
     * <t1>, on the other hand, is just the right size for the number of raw
     * characters to be read; it's only used for holding filenames, so 256
     * characters should be ample
     */
    if ((t1 = malloc(t1_size = LINELEN)) == NULL)
    {
	cprintf("%s\n", no_mem);
	sleep(3);
	free(tstring);
	return 0;
    }

    local_tabsize = TABSIZE;

    /*
     * Set an appropriate and available "opposite" value for <weedlevel>
     */
    opp_weedlevel = (weedlevel == HW_FULL) ? HW_WEEDED : HW_FULL;
    while (!weedlevelok(opp_weedlevel))
	opp_weedlevel = (opp_weedlevel + 1) % NUM_WEEDLEVEL;

    /*
     * The view screen has FIXED_TOPLINES fixed lines at the top (a From /
     * Subject summary line and a divider), and PROMPT_BOTLINES lines at the
     * bottom which are used for prompts, status messages etc. (and which
     * means that we can display the bottom "text" line, immediately above
     * these status lines, without worrying about line wrapping causing the
     * screen to scroll).  The actual "text" display area is therefore
     * (FIXED_TOPLINES + PROMPT_BOTLINES) lines smaller than the screen; to
     * ensure that the fixed top lines stay in place, we define a scrolling
     * window which starts at screen row (1 + FIXED_TOPLINES) and runs to the
     * bottom (and is therefore FIXED_TOPLINES lines smaller than the screen
     * display).  This window is in use throughout; row 1 is the first row of
     * the text area, and we set <screen_lines> to the number of lines
     * available to us for message display, and <bottom_line> to be the status
     * line at the very bottom of the screen.
     */
    screen_lines = display.screenheight - (FIXED_TOPLINES + PROMPT_BOTLINES);
    bottom_line = display.screenheight - FIXED_TOPLINES;

    /*
     * We keep track of displayed lines in <dr>, a dynamically allocated (and
     * expanding) array of LINEINFO structs, which describe the starting file
     * position and line type for each line that gets displayed on the screen.
     * (We need the line type to ensure that q-p lines get correctly decoded
     * when displayed, amongst other things.)
     *
     * Since we won't always be drawing the last line on the screen last (e.g.
     * if we scroll up by one line), we also keep track of which line (a <dr>
     * index value) is drawn at which screen line, via the win_*() functions
     * and associated variables.  This enables us to find the bottom line of
     * text, or to discover whether we actually have a full screen of text or
     * not.  The first entry in the array of <dr> indexes represents the index
     * for the top screen line: this used to be stored in a variable called
     * <toplndr>.  We get at it via macro TOPLNDR.
     *
     * At any given time, the current state is defined by various variables:
     *
     *   <lndr>		<dr> index of the line most recently printed to the
     *			screen: the cursor will be at the start of the line
     *			below this one.  This is not necessarily the bottom
     *			line of the screen, or even the bottom displayed line.
     *
     *   <going_down>	Describes the direction in which we're moving through
     *			the file.  It has three values:
     *			   1	Forwards: moving down a line at a time
     *			  -1	Back 1 line: display a new top line, which is
     *				immediately before the "current" top line
     *			   0	Completed the "back 1 line" stage
     *
     * We need to ensure that <lndr>, <going_down> and the actual current file
     * position are always correctly in sync (which is more than PCElm ever
     * did).  The win_*() routines should ensure that TOPLNDR is always the
     * correct index for the top display line, and that <wincur> is the
     * current line for the cursor (just to be safe, there is an explicit
     * check against the actual screen cursor).
     *
     * The .type field of the LINEINFO struct is used for noting what sort of
     * line this is, and therefore how we should display it: this mostly
     * relates to quoted-printable decoding, which is done on the fly by the
     * print_line() function.  When running through the entire message from
     * start to finish, you always know how a particular line should be
     * displayed; however, if you move back to an earlier point in the
     * display, the current "are we in a q-p entity" flag doesn't tell you
     * whether the earlier line should be displayed au naturel, or q-p
     * decoded; so we keep the line type in this field.  Note that the main
     * message headers are treated specially; their .type is the type of the
     * message body, even though header lines themselves are always displayed
     * in raw form.  We keep track of the file offset of the End Of Header
     * line (the blank line after the first message headers) to tell us
     * whether a given line is in the headers or not.
     *			
     * Try to allocate the <dr> array of LINEINFO structs a screenful at a
     * time; <maxlndr> contains the number of lines we can store at any given
     * point.  Wimp out now if we can't allocate the first screenful.
     */
    maxlndr = screen_lines;
    if ((dr = malloc(sizeof(*dr) * (size_t) maxlndr)) == NULL)
    {
	cprintf("%s\n", no_mem);
	sleep(3);
	free(tstring);
	free(t1);
	return 0;
    }

showagain:
    /*
     * Set all the local "control" variables to their start-of-message states;
     * they must all be set identically each time the message is viewed.
     *
     * <found_boundary> is somewhat bizarre; it is normally set to -1, to
     * indicate that we haven't found any boundaries yet; but if the entire
     * message is encoded with base64 or uuencode, we set it to -2, which
     * forces the decode to be called immediately.  Most odd.
     */
    found_boundary = -1;
    linetype = LT_NORMAL;
    if (enablemime)
    {
	linetype = mime_linetype(cmsg->content_type, cmsg->content_encoding);
	if (cmsg->content_encoding == CE_BASE64 ||
	    cmsg->content_encoding == CE_UUENCODE)
	    found_boundary = -2;	/* decode the entire message */
    }

    lndr = -1;
    eohlndr = -1;
    endlndr = -1;
    going_down = 1;
    inheader = TRUE;
    weedset(loc_weedlevel);		/* Reset weedout checking system */
    atend = FALSE;
    totalsize = cmsg->size;
    fseek(mfile, cmsg->position, SEEK_SET);
    end = cmsg->position + cmsg->size - 1L;
    checked_uuencode = FALSE;
    searching = found = FALSE;
    dr[0].fpos = -1L;
    dr[0].type = LT_ENDMSG;

    search_item[0] = '\0';		/* do we *really* want this? -ST */

    multidepth = init_multidepth;
    for (i = 0; i < multidepth; i++)	/* reset multipart body counts */
	multipart[i].nparts = 0;

    mpskipping = FALSE;			/* not skipping multipart yet */
    mpcommand = FALSE;			/* not a multipart prompt command */

    if (!(cmsg->status & S_READ))	/* not yet read */
    {
	cmsg->status |= S_READ;
	mbchanged = TRUE;		/* note the change to Status: */
    }

    /*
     * Draw the basic view screen, which sets our scrolling window, then set
     * our "window" line-tracking system to the correct length, and initialise
     * it to all blank lines
     */
    draw_viewscreen(cmsg);
    win_setlen(display.screenheight - FIXED_TOPLINES);
    win_init();

    /*
     * The code will allow you to scroll up several lines at time by setting
     * going_down to -2 or more, but this is slow and looks bad when doing a
     * complete screen; we therefore don't use it for Page Up.
     *
     * If we haven't yet reached the end of this message, or we're scrolling
     * back up, process a line.
     */
    while ((!feof(mfile) && ftell(mfile) < end) || going_down <= 0)
    {
	if (going_down <= 0)		/* scroll back up 1 line */
	{
	    /*
	     * We're supposed to reprint the line immediately before <lndr>,
	     * at the top of the screen; make absolutely certain we don't run
	     * back before the start of the message (as PCElm was prone to
	     * doing)
	     */
	    if (lndr <= 0)
	    {
		errorbox("Pager Error", "Line numbering screwed (-ve line)!",
			 "Resetting...", NULL);
		lndr = 1;	/* ready to be decremented back to 0 */
	    }
	    cdr = &dr[--lndr];
	    inheader = (eohlndr < 0 || lndr < eohlndr);
	    linetype = cdr->type;
	    fseek(mfile, here = cdr->fpos, SEEK_SET);
	}
	else
	    here = ftell(mfile);

	/*
	 * <here> is now the file position at the start of this line, which
	 * will be stored in index <lndr + 1>
	 */

	if (fgets(tstring, LINELEN, mfile) == NULL)	/* only read LINELEN */
	    break;	/* hit EOF -- break out to end-of-file handler */

	/*
	 * If we're past the message headers, going down (rather than
	 * scrolling back up) and (a) <found_boundary> is -2 (indicating that
	 * the whole message body is a single base64/uu encoded entity to be
	 * decoded), or (b) we're in a multipart entity, looking for boundary
	 * delimiter strings, we have a quick look here to see if we can find
	 * anything interesting to do; if not, or we fall out of the end, we
	 * simply display the line.
	 *
	 * ST-FIXME: this doesn't check we're at the start of a line before
	 * looking for MIME boundaries etc., but with the going up and down in
	 * the message it would be too painful to do.  Wait until The Grand
	 * Pager Rewrite.
	 */
	if (!inheader && going_down == 1 &&
	    (found_boundary == -2 || multidepth > 0))
	{
	    int action = NO_ACTION;
	    int boundidx, isend;

	    if (found_boundary == -2)		/* decode whole message */
	    {
		fseek(mfile, cmsg->position, SEEK_SET);
		action = handle_entity(mfile, tstring, tstring_size, end,
				       &linetype, &ch);
		if (action == HANDLE_KEY)	/* handle keypress */
		{
		    mpcommand = TRUE;
		    goto gotakey;
		}
		if (action == SHOW_REMAINS || action == REDRAW_REMAINS)
		{				/* make message look normal */
		    found_boundary = -1;
		    linetype = LT_NORMAL;
		}
		else if (action == SKIP_START)
		{
		    fseek(mfile, end, SEEK_SET);   /* jump to end of message */
		    continue;
		}
	    }
	    else if ((boundidx = isboundary(tstring, &isend)) >= 0)
	    {				/* <tstring> is a multipart boundary */
		/*
		 * The CRLF immediately before a multipart boundary is part of
		 * the boundary, and shouldn't be displayed: if we've just
		 * displayed a blank line, we need to undo it.  ST-FIXME
		 */
		if (mpskipping)	/* clear "Skipping..." display line */
		{
		    quickcenter(display.screenheight, col_normal, "");
		    mpskipping = FALSE;
		}
		if (isend)	/* it's the end of a multipart entity */
		{
		    if (boundidx == 0)	/* That's all folks! */
		    {
			/* get_cr("End of Multipart Mime Message"); */
			action = END_MESSAGE;
		    }
		    else		/* Back to m/p index <boundidx - 1> */
		    {
			multidepth = boundidx;	/* Note new m/p depth */
			action = SKIP_START;
		    }
		}
		else		/* it's the next entity of the current m/p */
		{
		    found_boundary++;
		    multipart[multidepth - 1].nparts++;
		    action = handle_entity(mfile, tstring, tstring_size, end,
					   &linetype, &ch);
		}
	    }
	    else
	    {
		if (found_boundary == -1)  /* before 1st boundary: ignore */
		    continue;  /* with the main line loop: while (!feof) */
	    }

	    switch (action)
	    {
	    case STOP_NOW:	/* Stop display immediately */
		multidepth = 0;
		free(tstring);
		free(t1);
		free(dr);
		return 0;

	    case SKIP_START:	/* Restart boundary search; skip this line */
		found_boundary = -1;
		quickcenter(display.screenheight, col_hilite, "Skipping...");
		mpskipping = TRUE;
		continue;

	    case REDRAW_REMAINS:  /* Redraw screen, then as SHOW_REMAINS */
		here = redraw_view(cmsg, screen_lines, eohlndr, dr, &inheader,
				   tstring, tstring_size);
		continue;

	    case SHOW_REMAINS:	/* Reset position; skip current line */
		here = ftell(mfile);
		continue;

	    case SHOW_FULL:	/* Skip current line */
		continue;

	    case END_MESSAGE:	/* End of the message */
		fseek(mfile, end, SEEK_SET);	/* jump to end of message */
		continue;

	    case HANDLE_KEY:	/* Handle keypress stored in <ch> */
		mpcommand = TRUE;
		goto gotakey;

	    case NO_ACTION:	/* Do nothing special: display this line */
	    default:
		break;
	    }
	}

	/*
	 * If we get to here, we're actually displaying the text of the
	 * message line by line as it appears (with possible q-p decoding).
	 * Note that we no longer unwrap quoted-printable soft linebreaks
	 * here; that is now done by print_string().
	 *
	 * We need special processing for header lines while displaying them
	 * on the way down: we need to spot the end of the headers, and we may
	 * need to weed some of them out as well.
	 */
	if (inheader && going_down == 1)
	{
	    int weedstat = weedit(tstring, here);

	    if (weedstat < 0)		/* End of header */
	    {
		inheader = FALSE;
		eohlndr = lndr + 1;	/* note end-of-header line */
		if (loc_weedlevel == HW_NONE)	/* weed ALL headers */
		    continue;	/* with the main line loop : skip this line */
	    }
	    else if (weedstat > 0)	/* Weed this line out */
		continue;	/* with the main line loop : skip this line */
	}

	/*
	 * This is a valid line: if going down, we need to note this line's
	 * file position and line type, and we may need to allocate more space
	 * to store file positions.  Note that we don't do this if we're
	 * redisplaying a line (in which case, <going_down> will be <= 0):
	 * <lndr> is already set for the line we're doing, and the line's dr[]
	 * contents are already correct.
	 */
	if (going_down == 1)	/* Normal forwards passage through msg */
	{
	    if (lndr == maxlndr - 1)	/* already using last allocated line */
	    {
		/* allocate more memory: another screenful of lines */
		LINEINFO *ndr;

		maxlndr += screen_lines;
		ndr = realloc(dr, sizeof(*dr) * (size_t) maxlndr);
		if (ndr == NULL)
		{
		    cprintf("%s\n", no_mem);
		    sleep(3);
		    free(tstring);
		    free(t1);
		    free(dr);				/*lint !e644*/ /* OK */
		    return 0;
		}
		dr = ndr;
	    }
	    dr[++lndr].fpos = here;
	    dr[lndr].type = linetype;
	}

	/* <lndr> is now the index for the line we're about to print */

	found = print_string(tstring, tstring_size, mfile, here,
			     (searching) ? search_item : NULL,
			     (BOOL) (!inheader && linetype == LT_QPTEXT));
	win_curidx(lndr);	/* note what line is displayed here */
	win_newline();		/* note that cursor has wrapped to next line */

	/*
	 * If (a) we're in the message body, (b) we're going down, (c) the
	 * user hasn't already refused a uuencoded file decode, (d) the
	 * message isn't quoted-printable, and (e) it's not a MIME message,
	 * look to see if the current line matches the uuencode "begin" line
	 * format EXACTLY.  If it does, offer to decode the file: if the user
	 * refuses, we turn off uuencode checking for this file; if he accept,
	 * run the decode and then immediately exit!
	 *
	 * This "exit if decoded" concept is broken: there's nothing stopping
	 * files having more than one uuencoded part (I've seen it lots of
	 * times; I've even done it myself).  ST-FIXME
	 */
	if (!inheader && going_down == 1 && !checked_uuencode &&
	    linetype != LT_QPTEXT && linetype != LT_QPBIN &&
	    !(cmsg->status & S_MIME))
	{
	    if (uu_begin(tstring, t1, t1_size) == 0)
	    {		/* Got uuencode "begin " line: decode file <t1> */
		if (wincur > 12 && wincur < 23)    /* Box will hide the line */
		    get_cr("Found a uuencoded file");
		if (getwin(3, 18, 50, col_select, t1,
			   "This message contains a uuencoded file",
			   t1,
			   "",
			   "Enter the filename to decode it to",
			   blank_to_abort,
			   "",
			   NULL))
		{
		    /* TODO: keep going after the decode() */
		    (void) decode(mfile, end, CE_UUENCODE, t1);
		    free(tstring);
		    free(t1);
		    free(dr);
		    return 0;
		}
		checked_uuencode = TRUE;
	    }
	}

	/*
	 * print_string() always (a) writes out 80 chars, or (b) ends with a
	 * CRLF -- so the cursor will be at the start of the line *after* the
	 * one just printed.  If this line is the bottom line of the screen
	 * (the "status" line), the screen will scroll up by one line next
	 * time we call print_string(), and we'll be on the bottom line again
	 * next time around (but with the screen scrolled up by one line since
	 * then).
	 */
	if (wincur != wherey() - 1)	/* Aargh! */
	{
	    char errbuf[81];
	    int y = wherey();
	    sprintf(errbuf, "Real cursor at %d, internal cursor at %d",	/*OK*/
		    y, wincur + 1);
	    errorbox("Pager error",
		     "Mismatch between real and internal cursor rows:",
		     errbuf, NULL);
	    win_gotoline(y);
	}
	
	if (going_down < 0)	/* we've re-done up 1 line: reduce up-count */
	    going_down++;

	/*
	 * The line has been displayed.  <TOPLNDR> is the index of the line
	 * displayed at the top of the screen; <lndr> is the index of the line
	 * we have just displayed; <wincur> is the row immediately below the
	 * line we've just displayed; <ftell(mfile)> is the current file
	 * position, ready to read in the next line of the message.
	 * <going_down> is either 1 (it was 1 throughout the pass) or 0 (it
	 * was -1 at the start of the pass).  If <going_down> is 0, we've just
	 * printed the very top line; if not, we could be anywhere (we will
	 * usually have just printed the last line, but that's not necessarily
	 * certain).
	 *
	 * If we were searching for a pattern match, <search> is true: this is
	 * only the case if <going_down> == 1.  If the line just displayed
	 * matched the pattern, <found> will be true; it will be false if we
	 * didn't get a match.
	 *
	 * We want to show the bottom status line, and go into the keypress
	 * handler, if (a) we've just displayed the bottom text line (<wincur>
	 * will be <bottom_line - 1>), (b) we've just seen a match for the
	 * pattern (<found> is true), or (c) we've just scrolled up
	 * (<going_down> == 0), in which case we must have been at the
	 * keypress stage last time: we should return there now.  If none of
	 * this is true, we carry on to look at the next line: if there is
	 * more to come, we will display the next line and end up here again;
	 * if not, we will fall out of the main line loop and end up at the
	 * "end of file" handler below; this displays a "you've seen it all"
	 * bottom status line, then jumps into the keypress handler at the
	 * "getakey" point (bypassing the stuff which normally does the bottom
	 * status line).
	 */

	if (found || going_down == 0 || wincur >= bottom_line - 1)
	{
	    /*
	     * Note out CURRENT file position: it's more likely to be of
	     * interest than the position at the start of the line we've just
	     * displayed.
	     */
	    here = ftell(mfile);

	    if (searching && !found)	/* searching, but not found yet */
	    {
		if (!feof(mfile) && here < end)
		{		/* OK to continue searching */
#if 0
		    /* manually scroll the screen up by 1 line to make room */
		    win_clrline(bottom_line);	/* clear status line */
		    win_scrollup();		/* scroll up 1 line */
		    win_gotoline(bottom_line - 1);  /* bottom text line */
#endif
		    continue;	/* with the main line loop: on to next line */
		}
		else		/* reached end of msg without a match */
		{
		    get_cr("No match found");
		    searching = FALSE;
		    break;	/* out of the main line loop: 100% seen */
		}
	    }

	getakey_help:
	    win_defline(bottom_line, WL_STATUS);
	    percent = file_percent(dr, end, totalsize, lndr, here, endlndr);
	    textattr(col_select);
	    if (!found && percent >= 100)	/* end of the road */
	    {
		cputs(atend_prompt);
		atend = TRUE;
	    }
	    else				/* match, or more to come */
	    {
		cprintf("%s (%2d%%) ", (found) ? "MATCH!" : "More ", percent);
		atend = FALSE;
	    }
	    textattr(col_normal);
	    clreol();
	    /* cursor is now on bottom line */
	getakey:
	    mpcommand = FALSE;		/* not a multipart prompt command */
	    while ((ch = (int) getkey()) == 0)
		;
	gotakey:
	    found = 0;
	    switch (ch)					/*lint !e644*/	/*OK*/
	    {
	    case 'i':		/* Quit immediately */
	    case 'q':
	    case ESC:
		free(tstring);
		free(t1);
		free(dr);
		return 0;		/* leave function altogether */

	    case '?':		/* Some help please */
	    case KF1:
		win_gotoline(bottom_line);
		textattr(col_hilite);
		clreol();
		cputs("/=Search h=header toggle q=quit a=show again, "
		      "4/8 tabs, mrRgGfpdunstw");
		while (getkey() == 0)	/* Wait for any keypress */
		    ;
		win_clrline(bottom_line);
		found = FALSE;		/* turn this off now */
		here = ftell(mfile);	/* ensure this is right before goto */
		goto getakey_help;	/* redo last line & reenter switch */

	    case '/':		/* Initialize search */
		searching = FALSE;
		if (lndr != (i = TOPLNDR + screen_lines - 1))
		{	/* not currently at bottom of screen: move there */
		    int nlndr = last_line(dr, end, mfile);
		    if (nlndr < 0)	/* haven't got a full screen */
		    {
			get_cr(nomore_search);
			goto getakey_help;
		    }
		    lndr = nlndr;
		    inheader = (eohlndr < 0 || lndr < eohlndr);
		    linetype = dr[lndr].type;
		    going_down = 1;
		}
		if (ftell(mfile) >= end)
		{
		    get_cr(nomore_search);
		    goto getakey_help;
		}
		win_gotoline(bottom_line);
		textattr(col_hilite);
		cputs(search_prompt);
		textattr(col_normal);
		clreol();
		{
		    char tmp[80];
		    tmp[0] = '\0';
		    maxgets(tmp, 60);
		    if (tmp[0] != '\0')
			AUSTRCPY(search_item, tmp);
		}
		win_clrline(bottom_line);
		if ((searching = (BOOL) (search_item[0] != '\0')) == FALSE)
		    goto getakey_help;		/* nothing to do */
		/* we're at the bottom line, and we've got a full screen: */
		/* scroll up 1 line */
		win_clrline(bottom_line);
		win_scrollup();
		win_gotoline(bottom_line - 1);
		going_down = 1;
		break;		/* leave switch; back to main line loop */

	    case KKP2:		/* Scroll down 1 line */
	    case KDN:
	    case KCR:
		if (atend)			/* already at end */
		{
		    if (ch == KCR && retendquit)	/* exit now */
		    {
			free(tstring);
			free(t1);
			free(dr);
			return 0;
		    }
		    goto getakey;		/* else ignore key */
		}
		if (lndr != TOPLNDR + screen_lines - 1)
		{	/* not currently at bottom of screen: move there */
		    int nlndr = last_line(dr, end, mfile);
		    if (nlndr < 0)	/* haven't got a full screen */
			goto getakey;
		    lndr = nlndr;
		    inheader = (eohlndr < 0 || lndr < eohlndr);
		    linetype = dr[lndr].type;
		}
		going_down = 1;
		if (ftell(mfile) < end)
		{
		    /* clear bottom screen line, scroll window up 1 line */
		    win_clrline(bottom_line);
		    win_scrollup();
		    win_gotoline(bottom_line - 1);
		    searching = FALSE;
		}
		break;

	    case KKP8:		/* Scroll up one line */
	    case KUP:
		if (lndr <= 0 || TOPLNDR <= 0)	/* can't go back: ignore key */
		{
		    if (mpcommand)	/* Was at multipart prompt! */
		    {
			/*
			 * Was at a multipart prompt: reset the file position
			 * to the start of the prompt line and go again,
			 * re-reading the line so we can re-prompt
			 */
			fseek(mfile, here, SEEK_SET);
			continue;	/* with the while (!feof(mfile)) */
		    }
		    goto getakey;
		}
		lndr = TOPLNDR;		/* jump to top line */
		inheader = (eohlndr < 0 || lndr < eohlndr);
		linetype = dr[lndr].type;
		/* scroll down 1 line, move cursor to (empty) top line */
		win_scrolldown();
		win_gotoline(1);
		searching = FALSE;
		atend = FALSE;
		going_down = -1;
		break;

	    case ' ':		/* Down one page, or exit if already at end */
	    case KKP3:
	    case KPGDN:
	    case KTAB:
		if (atend)		/* already at end: wimp out now */
		{
		    free(tstring);
		    free(t1);
		    free(dr);
		    return 0;
		}
		going_down = 1;
		if (lndr != (i = TOPLNDR + screen_lines - 1))
		{	/* not currently at bottom of screen: move there */
		    int nlndr = last_line(dr, end, mfile);
		    if (nlndr < 0)	/* haven't got a full screen */
		    {
			/* Fudge end of file, back into loop (will fall out) */
			if ((lndr = win_last()) < 0)
			    lndr = 0;
			inheader = (eohlndr < 0 || lndr < eohlndr);
			linetype = dr[lndr].type;
			win_gotoline(lndr - TOPLNDR + 1);
			fseek(mfile, here = end, SEEK_SET);
			break;		/* out of switch, back to line loop */
		    }
		    lndr = nlndr;
		    inheader = (eohlndr < 0 || lndr < eohlndr);
		    linetype = dr[lndr].type;
		}
		if (ftell(mfile) < end)		/* more to come */
		{
		    win_clrscr();
		    searching = FALSE;
		}
		break;

	    case KKP9:		/* Up one page */
	    case KPGUP:
		if (lndr <= 0 || TOPLNDR <= 0)	/* can't go back: ignore key */
		{
		    if (mpcommand)	/* Was at multipart prompt! */
		    {
			/*
			 * Was at a multipart prompt: reset the file position
			 * to the start of the prompt line and go again,
			 * re-reading the line so we can re-prompt
			 */
			fseek(mfile, here, SEEK_SET);
			continue;	/* with the while (!feof(mfile)) */
		    }
		    goto getakey;
		}
		/* set lndr 1 page up from top, clear screen, redraw all */
		lndr = TOPLNDR - screen_lines;
		if (lndr < 0)
		    lndr = 0;
		win_clrscr();
		for (i = 0, cdr = &dr[lndr]; i < screen_lines;
		     i++, lndr++, cdr++)
		{
		    inheader = (eohlndr < 0 || lndr < eohlndr);
		    fseek(mfile, here = cdr->fpos, SEEK_SET);
		    if (fgets(tstring, LINELEN, mfile) == NULL)	/* LINELEN */
			break;
		    (void) print_string(tstring, tstring_size, mfile, here,
					NULL, (BOOL) (!inheader &&
						      cdr->type == LT_QPTEXT));
		    win_curidx(lndr);	/* note what line is displayed here */
		    win_newline();	/* note that cursor has wrapped */
		}
		/* make it look like we just drew the top line, going up */
		searching = FALSE;
		atend = FALSE;
		cdr = &dr[lndr = TOPLNDR];
		inheader = (eohlndr < 0 || lndr < eohlndr);
		linetype = cdr->type;
		win_gotoline(2);
		fseek(mfile, here = cdr[1].fpos, SEEK_SET);
		going_down = 0;
		goto getakey_help;

	    case 'h':		/* Toggle header weeding on/off */
		if (loc_weedlevel == weedlevel)	  /* Set opposite level */
		    loc_weedlevel = opp_weedlevel;
		else				  /* Restore original level */
		    loc_weedlevel = weedlevel;
		goto showagain;

	    case 'a':		/* Show entire message again */
		goto showagain;

	    case '8':		/* Change tab size to 8 chars */
		local_tabsize = 8;
		goto showagain;

	    case '4':		/* Change tab size to 4 chars */
		local_tabsize = 4;
		goto showagain;

	    case '!':		/* Execute shellcmd */
		shellcmd();
		if (mpcommand)		/* Shelled from multipart prompt! */
		{
		    /*
		     * Shelled out from a multipart prompt: redraw the screen,
		     * then reset the file position to the start of the prompt
		     * line and go again, re-reading the line so we can
		     * re-prompt
		     */
		    (void) redraw_view(cmsg, screen_lines, eohlndr, dr,
				       &inheader, tstring, tstring_size);
		    fseek(mfile, here, SEEK_SET);
		    continue;	/* with the while (!feof(mfile)) */
		}
		here = redraw_view(cmsg, screen_lines, eohlndr, dr, &inheader,
				   tstring, tstring_size);
		goto getakey_help;

		/*
		 * Finally, a block of hard-wired "index" command keys; exit
		 * and return the key to calling function for action.  These
		 * are mostly historical, and really shouldn't be hard-wired
		 */
	    case 'r':		/* CMD_REPLY */
	    case 'R':		/* CMD_REPLY_INCL */
	    case 'm':		/* CMD_MAILTO */
	    case 'g':		/* CMD_GROUP */
	    case 'G':		/* CMD_GROUP_INCL */
	    case 'f':		/* CMD_FORWARD */
	    case 'p':		/* CMD_REPAINT (?!) */
	    case 'd':		/* CMD_DELETE */
	    case 'u':		/* CMD_UNDELETE */
	    case 'n':		/* CMD_NEXTUNREAD */
	    case 's':		/* CMD_SAVE */
	    case 't':		/* CMD_TRANSMIT */
	    case 'w':		/* CMD_WRITE */
	    case 'e':		/* will be mapped to CMD_EXTPAGREAD */
		free(tstring);
		free(t1);
		free(dr);
		return ch;

	    default:
		goto getakey;
	    }
	}
    }

    /*
     * Finished the main line loop: we've reached the end of the message.
     * Display a helpful prompt and jump back into the keypress handler.
     * (I *hate* these gotos! -ST)
     */
    here = ftell(mfile);
    endlndr = lndr + 1;
    atend = TRUE;
    win_defline(bottom_line, WL_STATUS);
    textattr(col_select);
    cputs(atend_prompt);
    textattr(col_normal);
    clreol();
    goto getakey;
    /*NOTREACHED*/
}						/*lint !e429 */ /*OK*/


/*
 * viewmsg()
 *
 * Prepare a message for display. Either shove it to a multimedia mail
 * program, to an external pager or display using internal pager.
 *
 * We run through the message's headers, looking for clues; this does not
 * involve looking at any MIME bits that occur during the body.  We only look
 * at the message headers.
 *
 * If the whole message is an attachment, this doesn't get handled properly;
 * in particular, if it's a text attachment, it just gets displayed by the
 * pager without any chance to decode it to a file.  ST-FIXME!
 */
int viewmsg(PHTL cmsg)
{
    multidepth = 0;		/* No MIME multipart entities known yet */

    if (enablemime && (cmsg->headers & H_MIME_VERSION))
    {
	/* MIME */
	const char *cmd;
	char buf1[81], buf2[81], buf3[81], input[5];
	char contenttype[LINELEN], contentencoding[LINELEN];
	char tmpname[FNAMELEN];
	FILE *filetmp;
	int have_mm, err;

	/*
	 * If this MIME message is easy to deal with, get on with it: we only
	 * put up the "what do you want to do" dialog if it's not obvious what
	 * the user would want.  In particular, plain text and q-p text tend
	 * to get the pager immediately; q-p may produce an offer of using the
	 * internal if the user has defined an external, unless he's set
	 * <extpager> to EP_EXTONLY to stop being asked this.
	 */
	if (cmsg->content_type == CT_NONE)	/* treat as plain ASCII */
	    goto pager;

	if (cmsg->content_type == CT_TEXT_PLNASCII ||	/* plain ASCII */
	    cmsg->content_type == CT_TEXT_PLN88591 ||	/* plain iso-8859-1 */
	    cmsg->content_type == CT_TEXT_RICH)		/* enriched */
	{
	    if (cmsg->content_encoding == CE_7BIT ||	/* 7bit encoding */
		cmsg->content_encoding == CE_8BIT ||	/* 8bit encoding */
		cmsg->content_encoding == CE_NONE)	/* not encoded */
		goto pager;
	    if (cmsg->content_encoding == CE_QUOTED)	/* quoted-printable */
	    {
		if (extpager == EP_EXTONLY)  /* always use external: OK here */
		    goto pager;
		else if (extpager == EP_INTONLY ||	/* always internal */
			 y_n("MIME message (q-p).  The internal pager "
			     "can read it for you. Use it?"))
		    goto use_builtin;	/* internal pager can handle this */
	    }
	}

	if (cmsg->content_encoding == CE_BASE64 &&	/* base64 encoded */
	    extpager != EP_EXTONLY &&	/* allow choice of internal pager */
	    (extpager == EP_INTONLY ||
	     y_n("MIME message (BASE64).  "
		 "The internal pager can decode it for you. Use it?")))
	    goto use_builtin;	/* internal pager can handle this */

	/*
	 * Get the FULL Content-Type and Content-Transfer-Encoding lines,
	 * unfolded if necessary, so that we can use them for displaying
	 * in the prompt window
	 */
	get_content_info(cmsg, mfile, contenttype, sizeof(contenttype),
			 contentencoding, sizeof(contentencoding));

	/*
	 * If this message body is a multipart entity, try to find the
	 * boundary string from the Content-Type line; store it in the
	 * <multipart[]> array, incrementing the <multidepth> counter to
	 * indicate that we've found one.
	 */
	if (multi_check(contenttype, cmsg->content_type) < 0)
	    errorbox("MIME error", "Bad multipart Content-Type line",
		     contenttype, NULL);

	/*
	 * Pop up a window to tell user that this is a MIME mail.
	 * Prompt for 1) show using regular pager
	 *	      2) skip it
	 * 	      3) pipe to multimedia mail program (if defined)
	 */
	have_mm = (multimedia != NULL);
	sprintf(buf1, "  %.60s ", contenttype);				/*OK*/
	sprintf(buf2, "  %.60s - encoding.", contentencoding);		/*OK*/
	sprintf(buf3, "    '%.40s' as multimedia command line.",	/*OK*/
		(multimedia) ? multimedia : "");
	if (enablemime && extpager == EP_INTONLY)
	    strcpy(input, "1");
	else if (have_mm)
	    strcpy(input, "3");
	else
	    strcpy(input, "2");
	while (TRUE)
	{
	    if (getwin(3, 3, 1, col_select, input,
		       "Multimedia",
		       "This mail is probably a multimedia mail. Its Content-Type is",
		       buf1,	/* "  %.60s ", contenttype */
		       "and it is encoded using",
		       buf2,	/* "  %.60s - encoding.", contentencoding */
		       "",
		       " 1. Show it using your regular pager",
		       " 2. Skip, don't show the mail",
		       have_mm ?
		       " 3. Pipe it to a multimedia mail program; you have defined" :
		       "  You have not defined a multimedia mail program, so you won't",
		       have_mm ? buf3 :
		       "  be able to benefit from the multimedia extensions.",
		       "",
		       select_max(have_mm ? 3 : 2),	/* "Select your choice (1 - X):  " */
		       NULL) == 0 || input[0] == '2')
		return 0;	/* Skip this message */
	    if (input[0] == '1')
		goto pager;	/* Use the normal pager */
	    if (input[0] == '3' && have_mm)
		break;		/* Break out to calling the multimedia prog */
	}

	/*
	 * Option '3' chosen: use the user's defined "multimedia" program;
	 * copy the mail (with full headers) to a temporary file, and fire up
	 * the program on that temp file.  (Why do we do it in binary mode,
	 * when <mfile> was probably opened in text mode? -ST)
	 */
	if ((filetmp = opentemp(tmpname, "w+b")) == NULL)
	    return 0;		/* error message already shown */
	err = savemsg(cmsg, filetmp, HW_FULL, SM_DECODE);
	(void) fclose(filetmp);		/* OK: savemsg() does fflush() */
	if (err > 0)		/* write error during savemsg() */
	    write_error(tmpname, copying_msg);
	else if (err == 0)	/* savemsg() worked OK */
	{
	    cmd = buildcmd(multimedia, "", "", tmpname);
	    (void) disp_system(cmd, FALSE);
	}
	unlink(tmpname);
	if (multimedia[0] != NPCMD)	/* Prompt before resuming */
	    get_cr(NULL);
    }
    else		/* Not a MIME message */
    {
    pager:
	if (extpager != EP_INTONLY)	/* pipe through user's pager */
	{
	    const char *cmd;
	    char tmpname[FNAMELEN];
	    FILE *filetmp;
	    int err;

	    /* make a temp. file */
	    if ((filetmp = opentemp(tmpname, "w+")) == NULL)
		return 0;	/* error message already shown */
	    err = savemsg(cmsg, filetmp, weedlevel, SM_DECODE);
	    (void) fclose(filetmp);	/* OK: savemsg() does fflush() */
	    if (err > 0)	/* write error during savemsg() */
		write_error(tmpname, copying_msg);
	    if (err == 0)	/* savemsg() worked OK */
	    {
		cmd = buildcmd(pager, "", "", tmpname);
		(void) disp_system(cmd, FALSE);
		/* We never have a prompt after the external pager */
	    }
	    unlink(tmpname);
	}
	else
	{
	    int zz;

	use_builtin:
	    /* use builtin pager */
	    zz = doviewmsg(cmsg);
	    disp_fullscreen();
	    return zz;
	}
    }
    return 0;
}


/*
 * savemsg()
 *
 * Save message <cmsg> to stream <tfile>.  <weed> describes the level of
 * header weeding to apply during the copy; <options> adds further options,
 * which may be any or all of:
 *	SM_DECODE		Decode any q-p MIME encoding as we're writing
 *	SM_QUOTE		Prepend the quote string to each line
 *	SM_FORWARD		Is this message being forwarded?
 *
 * Return 0 if it's written OK, > 0 if we have a write error, or < 0 on
 * internal error (e.g. no messages, empty message); display a one-line gripe
 * if we have an internal error, but leave it to the calling function to
 * complain if we get a write error.  There may be some quoted-printable
 * decoding errors, which we warn the user about if all else goes OK, but they
 * don't stop us writing the whole message to <tfile>.
 *
 * IMPORTANT: we must call fflush() before returning OK, since there may be
 * buffered output and we only return 0 if it has *DEFINITELY* all been
 * written out to disk successfully.
 *
 * Not used for console output!
 */
int savemsg(PHTL cmsg, FILE *tfile, int weed, unsigned options)
{
    char tstring[LLINELEN];	/* now stack, not malloc(); should be OK -ST */
    long end;
    BOOL inheader = TRUE;
    BOOL done_status = FALSE;
    BOOL qpdecode;
    unsigned qperrs = 0;
    int chk, weedstat;

    if (mfile == NULL)
    {
	textcolor(col_enhanc);
	cprintf("%s\n", no_mail);
	return -1;
    }
    fseek(mfile, cmsg->position, SEEK_SET);
    end = cmsg->position + cmsg->size - 1L;
    if (!(cmsg->status & S_READ))	/* not yet read */
    {
	cmsg->status |= S_READ;
	mbchanged = TRUE;		/* note the change to Status: */
    }

    /*
     * Set up the header weeding to use for this message, then check whether
     * we should weed out a "Status:" line; if so, set <done_status> to TRUE
     * to prevent us trying to do anything with it
     */
    weedset(weed);	/* Set up the header weeding for this message */
    if (weedit("Status: R\n", -1L) != 0)	/* Don't write Status: line */
	done_status = TRUE;

    /* TODO: allow different forwarding styles */
    if (options & SM_FORWARD)	/* Forwarding: announce what we did ... */
	if (fputs("Forwarded message follows:\n\n", tfile) == EOF)
	    return 1;		/* write error */

    /* Note whether we're doing quoted-printable decoding or not */
    qpdecode = ((options & SM_DECODE) && enablemime &&
		mime_linetype(cmsg->content_type, cmsg->content_encoding) ==
		LT_QPTEXT);

    while (ftell(mfile) < end &&
	   fgets(tstring, sizeof(tstring), mfile) != NULL)
    {
	if (inheader)	/* Check for end of header and Status: line */
	{
	    weedstat = weedit(tstring, -1L);
	    if (done_status == FALSE && weedlineisnew() &&
		(chk = check_status(tstring, sizeof(tstring), tfile,
				    (options & SM_FORWARD))) != 0)
	    {
		if (chk < 0)		/* Write error adding Status: line */
		    return 2;
		done_status = TRUE;
	    }
	    if (weedstat < 0)		/* End of header (blank line) */
	    {
		inheader = FALSE;
		if (weed == HW_NONE)		/* weed ALL headers */
		    continue;	/* with the main line loop : skip this line */
	    }
	    else if (weedstat > 0)	/* Weed this line out */
		continue;	/* with the main line loop : skip this line */
	}

	/*
	 * There was a long-standing bug when putting a quote string at the
	 * start of each line: if the previous line written didn't end in a
	 * newline (e.g. an exceptionally long line which was too long for the
	 * buffer, or a quoted-printable one with a soft line break), the
	 * quote string would appear in the middle of the final output line.
	 * There are two approaches to fixing this (at least for the
	 * soft-break q-p case): (a) only write the quote string when the
	 * previous line ended in a newline (keep a check on this as we write
	 * each line out), or (b) force a line break at q-p soft line wrapping
	 * anyway, to avoid getting a whole paragraph as a single line in the
	 * output file.  This second version has the benefit of giving us
	 * plausible quoting of reasonably short lines, rather than a whole
	 * paragraph quoted as a single line, so this is what we do for now.
	 * Of course, such long lines *should* be wrapped by us before
	 * quoting, but let's leave that for another day.  ST-FIXME
	 */
	if (options & SM_QUOTE)
	    (void) fputs(quotestr, tfile);	/* check next fputs() */
	if (!inheader && qpdecode)	/* Do q-p decoding on message body */
	    if (decode_qpline(tstring, (options & SM_QUOTE)))
		qperrs++;
	if (fputs(tstring, tfile) == EOF)	/* Write error! */
	    return 3;
    }

    if (fflush(tfile) != 0)		/* write error */
	return 4;

    /*
     * If we're supposed to be weeding all headers, but we never found the end
     * of the header, this message had no body and we have written nothing;
     * complain
     */
    if (inheader && weed == HW_NONE)
    {
	cputs("Message has no body to save!\r\n");
	return -2;
    }

    if (qperrs > 0)		/* Warn user about q-p decoding errors */
	errorbox(decode_err, illegal_qp, NULL);
    return 0;
}


/*
 * check_status()
 *
 * <buf> is a new header line (i.e. one preceded by LF).  Check to see if it
 * contains a Status: header line; if it does, check to see if it contains
 * 'R', and append one before the newline if not.  If this is a blank line, it
 * marks the end of the header; write a "Status: R" line to file <fp>.  Return
 * 0 if we did nothing, 1 if <buf> contains a valid Status: string containing
 * 'R', 2 if this was the end of the header and we wrote a "Status: R" line to
 * the file <fp>, or < 0 if we had a write error.
 */
int check_status(char *buf, size_t bufsize, FILE *fp, BOOL quote)
{
    if (htype_match(buf, HT_STATUS))	/* Got Status: header field */
    {
	verify_status_line(buf, bufsize);	/* always succeeds */
	return 1;
    }
    else if (ISBLANKLINE(buf))		/* End-of-header blank; add Status: */
    {
	return (write_status_line(fp, quote) == 0) ? 2 : -2;
    }
    return 0;
}


/*
 * verify_status_line()
 *
 * Given string <buf> (size <bufsize>) which starts with a Status: header
 * field name, check that it contains 'R' in the field body somewhere; if not,
 * insert one before the newline (which is expected to be present).
 */
void verify_status_line(char *buf, size_t bufsize)
{
    if (strchr(hdr_body(buf, HT_STATUS), 'R') == NULL)
    {
	(void) rip(buf);
	PUSTRCAT(buf, bufsize, "R\n");
    }
}


/*
 * write_status_line()
 *
 * Write a "Status: R" line to <fp>; if <quote> is true, prefix it with a
 * quote string (global <quotestr>).  Return 0 on success, > 0 on write error.
 */
int write_status_line(FILE *fp, BOOL quote)
{
    if (quote)
	(void) fputs(quotestr, fp);	/* check fprintf() below */
    if (fprintf(fp, "%s R\n", headername[HT_STATUS]) == EOF)
	return 1;
    return 0;
}


/*
 * delmsg()
 *
 * Delete message <cmsg> in current notesfile
 */
int delmsg(PHTL cmsg)
{
    if (mfile == NULL)
    {
	errordsp(no_mail);
	return 0;
    }
    if (!(cmsg->status & S_READ)) 		/* don't delete unread msgs! */
	return 0;
    if (!(cmsg->status & S_DELETE))		/* not yet deleted */
    {
	cmsg->status |= S_DELETE;
	mbchanged = TRUE;			/* note the change */
    }
    return 1;
}


/*
 * get_msgid()
 *
 * Try to get the next sequence number from file <sequencename> in the mail
 * queue directory; we always return a long number between 0L and 99999999L
 * inclusive, even if we had a problem with the sequence file (in wgucg case
 * we return 0L).
 */
long get_msgid(void)
{
    char sfilename[LINELEN];
    char s[20];
    long sequence = 0L;
    FILE *sfile;

    AUSTRLCPY3(sfilename, mailqdir, "\\", sequencename);

    /* if sequence file exists, get the value, otherwise set it */
    if ((sfile = fopen(sfilename, "r")) != NULL)
    {
	if (fgets(s, sizeof(s), sfile))
	    sequence = atol(s);
	/* Keep it in range of an 8 digit number to use for dos name prefix */
	if (sequence < 0L || sequence > 99999999L)
	    sequence = 0L;
	(void) fclose(sfile);				/* OK: read only */
    }

    /* increment sequence number, and write to sequence file */
    if ((sfile = fopen(sfilename, "w")) == NULL)
	fopen_error(sfilename, "sequence file");
    else
    {
	BOOL writerr = FALSE;

	if (fprintf(sfile, "%ld\n", ++sequence) == EOF)
	    writerr = TRUE;
	if (fclose(sfile) != 0)
	    writerr = TRUE;
	if (writerr)
	    write_error(sfilename, "updating sequence number");
    }

    return sequence;
}


#pragma argsused
/*
 * create mail lockfile
 */
/*ARGSUSED*/
int mlock(const char *dir, const char *id)
{
#if 0
    char lockname[PATHLEN];
    FILE *fp;

    if (strlen(dir) > sizeof(lockname) - FNAMELEN - 1)
    {
	errordsp("Directory too long for mlock()");
	sleep(2);
	return -1;
    }
    sprintf(lockname, "%s\\%.8s%.3s", dir, id, lckfileext);		/*OK*/
    if (!access(lockname, 0))
    {
	char errtxt[LINELEN];
	AUSTRLCPY2(errtxt, cant_create, lockname);
	errordsp(errtxt);
	sleep(2);
	return -1;
    }
    if ((fp = fopen(lockname, "w")) != NULL)
    {
	(void) fclose(fp);			/* OK: unused */
	return chmod(lockname, S_IREAD);   /* protect lockfile */
    }
    return -1;
#else
    return 0;
#endif
}


#pragma argsused
/*
 * remove mail lockfile
 */
/*ARGSUSED*/
void rmlock(const char *dir, const char *id)
{
#if 0
    char lockname[PATHLEN];

    if (strlen(dir) > sizeof(lockname) - FNAMELEN - 1)
    {
	errordsp("Directory too long for rmlock()");
	sleep(2);
	return -1;
    }
    sprintf(lockname, "%s\\%.8s%.3s", dir, id, lckfileext);		/*OK*/
#ifdef __TURBOC__
    chmod(lockname, S_IWRITE | S_IREAD);   /* unprotect lockfile */
    /* strangely enough, this does not seem to be necessary in aztec.
       i have never seen an aztec compiler in my life, but an unlink
       operation seems to unlink all & everything there... */
#endif
    if (unlink(lockname))
	unlink_error(lockname, "lock file");
#endif
}


/*
 * get_save_file()
 *
 * Find the "Reply-To" address (or failing that, the "From" address) for
 * message <cmsg>, and create a valid DOS name from the SMTP address; copy the
 * filename into <into>, which is of size <intosize>, and return the length of
 * filename thus produced.
 */
int get_save_file(char *into, size_t intosize, PCHTL cmsg)
{
    char tstring[LLINELEN];
    char *p;
    long end;
    char last_eol = '\0';

    *into = '\0';
    fseek(mfile, cmsg->position, SEEK_SET);	/* go to start of msg */
    end = cmsg->position + cmsg->size - 1L;
    while (ftell(mfile) < end &&
	   fgets(tstring, sizeof(tstring), mfile) != NULL)
    {
	if (last_eol && ISBLANKLINE(tstring))	/* end of header */
	    break;	/* out of the while (ftell) */
	last_eol = rip(tstring);
	if ((*into == '\0' && htype_match(tstring, HT_FROM)) ||
	    htype_match(tstring, HT_REPLYTO))
	{
	    if ((p = strchr(tstring, ':')) != NULL)	/* past the header */
		(void) get_addr_filename(into, intosize, p + 1);
	}
    }
    return (int) strlen(into);
}


/*
 * fgets_msg()
 *
 * A ghastly crock which purports to read lines from the message, unfolding
 * them as they come in if <unfold> is true (we assume this will only be true
 * if we're currently reading the header), and copying complete lines --
 * possibly unfolded -- into the <into> buffer, which is of size <intosize>.
 *
 * Return TRUE if we copy something into <into>, or FALSE if we've reached the
 * end of the file.
 *
 * If <into> is NULL, we simply reset our internal stored string and return
 * FALSE.
 */
BOOL fgets_msg(char *into, size_t intosize, FILE *fp, BOOL unfold)
{
    static char got_string[LLINELEN] = "";
    static char last_eol = 1;	/* Did last line returned end with EOL? */
    char into_eol = 0;		/* Does <into> end with EOL? */
    int maxread;

    if (into == NULL)		/* Reset */
    {
	*got_string = '\0';
	last_eol = 1;
	return FALSE;
    }

    /*
     * Work out how many to try and read at once: the smaller of
     * sizeof(got_string) and <intosize>.
     */
    maxread = (int) min(sizeof(got_string), intosize);

    if (*got_string)		/* Got a string left from last time */
    {
	PUSTRCPY(into, intosize, got_string);
	*got_string = '\0';
	if (!unfold)		/* Not unfolding: just return this line */
	    return TRUE;
    }
    else			/* No pending string */
	*into = '\0';

    /*
     * Start reading new lines from the file.  If we're supposed to be
     * unfolding, there MAY be something already in <into> which was left over
     * from last time; if we're not unfolding, <into> is currently empty, and
     * we simply copy the first line we get into it and return TRUE.
     */
    while (fgets(got_string, maxread, fp) != NULL)
    {
	if (*into == '\0')	/* <into> is empty so far */
	{
	    strcpy(into, got_string);	/* OK : <= <maxread> chars total */
	    *got_string = '\0';
	    if (!unfold)		/* No unfolding: return this line */
		return TRUE;
	    continue;	/* with the while (fgets) : grab the next line */
	}

	/*
	 * The only way we can get to here is if <into> already contains the
	 * previous line (which may or may not be a complete line ending with
	 * LF), and <unfold> is true.  We are therefore trying to unfold the
	 * (presumed header) lines into the provided buffer <into>; we've got
	 * a line in <into> already, and we've just read another one into
	 * <got_string>.  If the following conditions are all true:
	 *   (a) this new line begins with an LWSP-char, AND
	 *   (b) the line in <into> ends with LF, AND
	 *   (c) the line in <into> is *not* the end-of-header blank line,
	 * then this new line a folded continuation of the line in <into>.
	 * We can test for (a) and (b) simply enough; (c) is true if (1) the
	 * first character of <into> is not the newline, or (2) <last_eol> is
	 * zero (i.e. the last line returned didn't end with LF).
	 *
	 * If so, we need to try to unfold this new line into the buffer --
	 * IF, AND ONLY IF, THERE IS ROOM TO DO SO!  If there isn't room, we
	 * fudge our way around it by putting up an overrun warning and
	 * returning TRUE now, leaving the existing contents of <into> intact,
	 * and the new line waiting in <got_string> for the next time this
	 * function is called.  This is gruesome, but it's better than a
	 * gazillion overrun warnings from repeated calls to PUSTRLCAT2() with
	 * an already-full buffer, which is what otherwise happens.
	 *
	 * We do the unfolding by appending the new line to <into>, separated
	 * by a single space (which involves ditching any trailing whitespace
	 * in <into>, and skipping over any leading whitespace in
	 * <got_string>).  If <got_string> consists solely of whitespace,
	 * ignore it (continue to the next line).
	 */
	into_eol = eoltest(into);
	if (islwsp(*got_string)	&& into_eol &&	/* conditions (a) and (b) */
	    (!ISBLANKLINE(into) || last_eol == 0))	/* condition (c) */
	{			/* It's a folded header continuation line */
	    char *p = skip_lwsp(got_string);

	    if (*p)		/* got something to append to <into> */
	    {
		size_t il, pl, rl;

		kill_trail_wsp(into);		/* zap trailing WS */
		il = strlen(into);
		pl = strlen(p);
		if ((rl = il + pl + 2) > intosize)	/* Aargh! */
		{
#ifdef REPORT_OVERRUNS
		    warn_overrun(into, intosize, rl, __FILE__, __LINE__,
				 "into");
#endif
		    if (il + 1 < intosize)	/* room to replace '\n' */
			PUSTRCAT(into, intosize, "\n");
		    last_eol = into_eol;
		    return TRUE;
		}
		PUSTRLCAT2(into, intosize, " ", p);
		*got_string = '\0';
	    }
	    continue;	/* with the while (fgets) : grab the next line */
	}

	/*
	 * <got_string> isn't a continuation line after all, but the start of
	 * a brand new header field: return TRUE to pass the current contents
	 * of <into> back to the calling function, and keep the new header
	 * line in <got_string> for the next call.
	 */
	last_eol = into_eol;
	return TRUE;
    }

    /*
     * We've reached end-of-file; there may be something already in <into>
     */
    *got_string = '\0';
    last_eol = into_eol;
    if (*into)		/* there's something in <into> */
	return TRUE;
    return FALSE;	/* ran off end of file?? */
}


/*
 * uu_begin()
 *
 * Test for line <linestr> being the start line of a uuencoded section.  In
 * order to be considered for this, it must start "begin "; if it doesn't
 * start this way, return -1.  It *should* go on to have a 3-digit octal
 * number, whitespace, and a filename; if it has all these, copy the filename
 * into <namebuf> (which is <namebufsize> bytes long) and return 0; if it
 * doesn't match this pattern exactly, return something > 0 (1-5).
 */
static int uu_begin(const char *linestr, char *namebuf, size_t namebufsize)
{
    const char *cp;

    if (strncmp(linestr, "begin ", 6))	/* not even the "begin " */
	return -1;

    cp = linestr + 6;
    if (cp[0] < '1' || cp[0] > '7')	/* invalid 1st octal digit */
	return 1;
    if (cp[1] < '0' || cp[1] > '7')	/* invalid 2nd octal digit */
	return 2;
    if (cp[2] < '0' || cp[2] > '7')	/* invalid 3rd octal digit */
	return 3;
    if (!islwsp(cp[3]))			/* no space after octal no. */
	return 4;

    cp = skip_lwsp(cp + 3);
    if (!*cp)				/* no filename */
	return 5;

    PUSTRCPY(namebuf, namebufsize, cp);
    kill_trail_wsp(namebuf);		/* zap trailing whitespace */
    return 0;
}


/*
 * extprog()
 *
 * Run external attachment-handling command <cmdstr> (which should contain '$'
 * where we want the filename to appear) with file <filename>, and with
 * <action> and <desc> as descriptive bits to use when asking the user what to
 * do.  Return 0 if we didn't mangle the screen, 1 if we did.
 */
static int extprog(const char *cmdstr, const char *filename,
		   const char *action, const char *desc)
{
    char cmdline[LINELEN];
    char promptline[LINELEN];
    BOOL predef = (strchr(cmdstr, '$') != NULL) ? TRUE : FALSE;

    if (predef)			/* Full user-defined command: use buildcmd() */
    {
	if (!action)			/* No supplied action description */
	    action = "view";
	if (!desc)			/* No supplied file description */
	    desc = "file";
	AUSTRCPY(cmdline, buildcmd(cmdstr, "", "", filename));	/* Local copy*/
	AUSTRLCPY4(promptline,
		   "Enter the command to use to ", action, " this ", desc);
    }
    else			/* Basic command: just append filename */
    {
	AUSTRLCPY3(cmdline, cmdstr, " ", filename);
	AUSTRLCPY3(promptline,
		   "Replace '", cmdstr, "' with an application name to use");
    }
    if (getwin(15, 17, 50, col_select, cmdline,
	       promptline,
	       blank_to_abort,
	       "", NULL) > 0)
    {
	(void) disp_system(cmdline, FALSE);
	/* We probably need to clear up the screen now */
	if (!predef && cmdline[0] != NPCMD)	/* Prompt before returning */
	    get_cr("Finished");
	disp_clear();
	return 1;		/* screen has been mangled */
    }
    return 0;			/* screen probably OK */
}


/*
 * opentemp()
 *
 * Open a temporary file, copying its name into <name> (which must be big
 * enough to hold an 8.3 filename); use mode <mode> for the fopen().  Return
 * the opened file pointer (which may be NULL if things went awry).
 */
FILE *opentemp(char *name, const char *mode)
{
    FILE *fp;

    strcpy(name, "PCOAKXXXXXX");	/* OK: room for 8 + '.' + 3 + null */
    if (mktemp(name) == NULL || (fp = fopen(name, mode)) == NULL)
    {
	errorbox(tempfile_err, name, no_tmpfile, NULL);
	return NULL;
    }
    return fp;
}


/*
 * decode_qpline()
 *
 * Given quoted-printable line <line>, decode any quoted-printable stuff we
 * find.  This will either be "=XX" ('=' followed by two hex digits) or '='
 * immediately before the newline (a "soft" line break).  Anything else is
 * invalid, and should be ignored as suggested by RFC 2045.  If <hardbreak> is
 * true, we force a hard line break in place of a soft one anyway (replace a
 * trailing "=\n" with "\n"); if not, we simply remove the soft line break
 * altogether (truncate the line at the "=\n") and let the next line flow on
 * from the end of this one.
 *
 * We don't need to be told the size of the line buffer, since decoding
 * quoted-printable always makes the line smaller.  Return 0 if the decode
 * goes well, 1 if we encounter an illegal =* sequence.
 */
static int decode_qpline(char *line, BOOL hardbreak)
{
    unsigned char uc;
    char *p = line;

    while ((p = strchr(p, '=')) != NULL)
    {
	if (strcmp(p, "=\n") == 0)	/* soft line break */
	{
	    if (hardbreak)	/* force a plain newline here */
		strcpy(p, "\n");			/* OK -ST */
	    else		/* truncate before the newline */
		*p = '\0';
	    break;	/* out of the while (p) */
	}
	if (!isxdigit(p[1]) || !isxdigit(p[2]))
	    return 1;	/* stop decoding at this bogus =* sequence */
	uc = (unsigned char) (hex_to_dec(p[1]) * 16 + hex_to_dec(p[2]));
	if (uc == 163)	/* convert u-acute to pound for GBP: assumes */
	    uc = 156;	/* ISO-8859-1 (US-ASCII shouldn't be > 127) */
	*p = uc;
	memmove(p + 1, p + 3, strlen(p + 3) + 1);	/* OK */
	p++;
    }
    return 0;
}


/*
 * get_content_info()
 *
 * Find the full text of the "Content-Type:" and "Content-Transfer-Encoding:"
 * header field bodies for message <cmsg>, which is available on stream <fp>,
 * and copy them into <type> and <enc> respectively; the sizes of the buffers
 * are <typesize> and <encsize>.  If the headers aren't present, copy "(none)"
 * into the buffers; if they are supposed to be there but we can't find them,
 * just make them empty strings.
 */
static void get_content_info(PCHTL cmsg, FILE *fp, char *type,
			     size_t typesize, char *enc, size_t encsize)
{
#if 0
    char tstring[LINELEN];
    char got_eol, last_eol = 1;
    BOOL got_type = FALSE;
    BOOL got_enc = FALSE;
    char *buf = NULL;
    size_t bufsize = 0;

    if (!(cmsg->headers & H_CONTENT_TYPE))
    {
	PUSTRCPY(type, typesize, bracket_none);
	got_type = TRUE;
    }
    else
	*type = '\0';
    if (!(cmsg->headers & H_CONTENT_XFER_ENCODING))
    {
	PUSTRCPY(enc, encsize, bracket_none);
	got_enc = TRUE;
    }
    else
	*enc = '\0';

    if (got_type && got_enc)
	return;

    if (fseek(fp, cmsg->position, SEEK_SET) != 0)
	return;

    while (fgets(tstring, sizeof(tstring), fp) != NULL)
    {
	if (last_eol && ISBLANKLINE(tstring))	/* end of header */
	    break;	/* out of the while (fgets) */
	got_eol = rip(tstring);
	kill_trail_lwsp(tstring);
	if (buf != NULL)	/* Look for folded continuation lines */
	{
	    if (!last_eol)		/* Direct continuation of last line */
		PUSTRCAT(buf, bufsize, tstring);
	    else if (islwsp(*tstring))	/* Folded continuation of last line */
		PUSTRLCAT2(buf, bufsize, " ", skip_lwsp(tstring));
	    else			/* Not a continuation line at all */
		buf = NULL;
	}
	if (buf == NULL && last_eol)	/* New line; look for match */
	{
	    if (got_type && got_enc)
		break;	/* out of the while (fgets) */
	    if (!got_type && htype_match(tstring, HT_CONTENT_TYPE))
	    {
		buf = type;
		bufsize = typesize;
		PUSTRCPY(buf, bufsize,
			 skip_lwsp(tstring +
				   strlen(headername[HT_CONTENT_TYPE])));
		got_type = TRUE;
	    }
	    else if (!got_enc &&
		     htype_match(tstring, HT_CONTENT_XFER_ENCODING))
	    {
		buf = enc;
		bufsize = encsize;
		PUSTRCPY(buf, bufsize,
			 skip_lwsp(tstring +
				   strlen(headername[HT_CONTENT_XFER_ENCODING])));
		got_enc = TRUE;
	    }
	}
	last_eol = got_eol;
    }

    while ((buf = strchr(type, '\t')) != NULL)
	*buf = ' ';	/* convert tabs to spaces for neatness of display */

    /* That's it! */
#else
    int get_hdrfield_body(HEADERTYPE hdrtype, FILE *fp, long pos, char *buf,
			  size_t bufsize, BOOL unfold);

    if (!(cmsg->headers & H_CONTENT_TYPE))
	PUSTRCPY(type, typesize, bracket_none);
    else if (get_hdrfield_body(HT_CONTENT_TYPE, fp, cmsg->position,
			       type, typesize, TRUE) == 0)
    {
	char *ptr;
	while ((ptr = strchr(type, '\t')) != NULL)
	    *ptr = ' '; /* convert tabs to spaces for neatness of display */
    }
    else
	*type = '\0';

    if (!(cmsg->headers & H_CONTENT_XFER_ENCODING))
	PUSTRCPY(enc, encsize, bracket_none);
    else if (get_hdrfield_body(HT_CONTENT_XFER_ENCODING, fp, cmsg->position,
			       enc, encsize, TRUE) != 0)
        *enc = '\0';

    /* That's it! */
#endif
}


/*
 * multi_check()
 *
 * Given the Content-Type line <line>, which has type <type>, check it for
 * multipart information, which is stored in the global array <multipart>; the
 * nested depth thus far in this message is in global <multidepth>.  If this
 * header does not introduce a multipart entity, return 0; if it is multipart,
 * check that we haven't exceeded the maximum depth of multipart entities
 * MAX_MULTI, and return -1 if we have.  If not, try to find the boundary
 * delimiter string from the line given.  If we can't find the string, return
 * -2; if we find it OK, copy it into the .boundary entry and return 1 to
 * indicate success, having incremented the <multidepth> to note our current
 * level.
 */
static int multi_check(const char *line, CTTYPE type)
{
    const char *cp;
    struct multipart *mp;

    if (type != CT_MULTI_MIXED && type != CT_MULTI_ALTERN &&
	type != CT_MULTI_DIGEST && type != CT_MULTI_MISC)
	return 0;			/* not multipart */

    if (multidepth >= MAX_MULTI)	/* multiparts nested too deeply */
	return -1;

    if ((cp = stristr(line, "boundary")) == NULL)	/* oh dear... */
	return -2;

    mp = &multipart[multidepth];	/* pointer to next multipart store */
    if (getstring(mp->boundary, sizeof(mp->boundary), cp + 8, TRUE) <= 0)
	return -3;			/* bad format */

    mp->type = type;
    mp->nparts = 0;
    multidepth++;		/* increment multipart nesting depth */
    return 1;
}


/*
 * isboundary()
 *
 * Look to see if <line> is one of our multipart boundaries, starting at the
 * current depth and moving back to wards the first/outer multipart (it is not
 * unknown for the ending boundary delimiter to be missing from multipart
 * entities).  To be a valid RFC 2046 boundary, <line> must start with
 * "--<boundary>", followed by an optional "--" (indicating that this is the
 * ending boundary), then optional linear whitespace and a [CR]LF.  If it is
 * one of the boundaries, return the index of boundary found (+ve number,
 * between 0 and <multidepth - 1>); if not, return -1.  If we find a match and
 * <isend> is not a NULL pointer, set it to 1 if the boundary is an "ending"
 * one ("--<boundary>--"), or 0 if not.
 */
static int isboundary(const char *line, int *isend)
{
    int i;
    size_t l, ll, bll;
    const char *end;

    if (line[0] != '-' || line[1] != '-' || multidepth <= 0)
	return -1;
    line += 2;
    end = line + strlen(line);
    if (end <= line || end[-1] != '\n')		/* doesn't end with LF */
	return -2;
    while (end > line && isspace(end[-1]))
	end--;
    /*
     * <end> now points just after the last non-whitespace char in <line>;
     * set <ll> to the length of this line, and if the last two characters
     * of the line are "--", set <bll> to <ll - 2>: this is the potential
     * length of a boundary match which ends in "--".
     */
    ll = (size_t) (end - line);
    bll = (ll > 2 && end[-2] == '-' && end[-1] == '-') ? (ll - 2) : 0;
    for (i = multidepth - 1; i >= 0; i--)	/* search from last to first */
    {
	if ((l = strlen(multipart[i].boundary)) > 0 &&
	    (l == ll || l == bll) &&
	    strncmp(line, multipart[i].boundary, l) == 0)
	{
	    if (isend)
		*isend = (l == bll) ? 1 : 0;
	    return i;
	}
    }
    return -2;
}


/*
 * handle_entity()
 *
 * Handle a MIME entity: this may be the entire message, or a part of a
 * multipart entity; we don't really care, as all entities should be treated
 * the same.  Upon entry to this function, we're at the start of the first
 * line of the entity's headers: scan them to find out all the information we
 * can about the entity, then decide what to do with it.  Lines are read from
 * <efp>; we're given a scratch buffer <temp> and its size <tempsize> to play
 * with.  We mustn't read beyond position <end>.
 *
 * <*linetype> should be filled in with the type of MIME data we expect from
 * the body lines of the entity: this will be one of the LT_* constants.
 *
 * We return one of the following "action to take" codes:
 *	STOP_NOW	Stop displaying the message; exit the viewer
 *	SKIP_START	Skip this line; restart looking for boundary
 *	SHOW_FULL	Skip this line; show this entity as plaintext
 *	SHOW_REMAINS	Reset "here" position, skip line; display what's left
 *	REDRAW_REMAINS	As SHOW_REMAINS, but redraw whole screen first
 *	HANDLE_KEY	Handle the keypress stored in <*chp>
 */
static int handle_entity(FILE *efp, char *temp, size_t tempsize, long end,
			 unsigned char *linetype, int *chp)
{
    const struct multipart *mp = NULL;
    char headers[MAX_HDRS][LINELEN], gash[LINELEN], input[5], name[81];
    const char *ptr;
    CTTYPE ct_type = CT_NONE;
    CETYPE ct_enc = CE_NONE;
    HEADERTYPE hdrtype = HT_UNKNOWN;
    int i, nhdrs = 0;
    BOOL has_name, new_multi, eoh;
    BOOL text, plaintext, binary, warn_multi;
    char last_eol = 1;		/* We begin at the start of the 1st line */

    gash[0] = name[0] = '\0';
    has_name = new_multi = eoh = FALSE;
    text = plaintext = binary = FALSE;
    warn_multi = TRUE;
    for (i = 0; i < MAX_HDRS; i++)	/* Make all headers null strings */
	*headers[i] = '\0';

    *linetype = LT_UNKNOWN;	/* Play safe until we know better */

    while (ftell(efp) < end && fgets(temp, (int) tempsize, efp) != NULL)
    {
	if (last_eol && ISBLANKLINE(temp))	/* end of header */
	    eoh = TRUE;
	last_eol = rip(temp);
	if (eoh || !islwsp(*temp))	/* NOT a folded continuation line */
	{
	    /*
	     * Either at end of header, or start of a new header field: either
	     * way, the previous header line (which has type <hdrtype>) is now
	     * complete and unfolded in <gash>
	     */
	    if (hdrtype == HT_CONTENT_TYPE)
	    {
		ct_type = get_content_type(gash);
		if (multipart[multidepth - 1].type == CT_MULTI_ALTERN ||
		    ct_type == CT_MULTI_MISC)
		{	/* try to copy the actual "type/subtype" text */
		    ptr = hdr_body(gash, hdrtype);
		    if (sscanf(ptr, "%70[^ \t\n;]", name + 2) == 1)
			name[0] = name[1] = ' ';	/* 2 leading spaces */
		}
		if (multi_check(gash, ct_type) > 0)	/* New multipart */
		    new_multi = TRUE;
	    }
	    else if (hdrtype == HT_CONTENT_XFER_ENCODING)
	    {
		ct_enc = get_content_encoding(gash);
	    }
	    if (eoh)			/* end of headers: that's all folks! */
		break;	/* out of the while (ftell) */
	    hdrtype = htype(temp);
	    gash[0] = '\0';
	}
	if (hdrtype == HT_CONTENT_TYPE || hdrtype == HT_CONTENT_XFER_ENCODING)
	    AUSTRCAT(gash, temp);	/* unfold into <gash> */

	switch (hdrtype)
	{
	case HT_CONTENT_TYPE:
	case HT_CONTENT_DESCRIPTION:
	case HT_CONTENT_DISPOSITION:
	    if (!has_name &&
		(stristr(temp, "name=") != NULL ||
		 ((ptr = stristr(temp, "name")) != NULL &&
		  wildmat(ptr + 4, "[ \t]*=*", 0))))
		has_name = TRUE;	/* seen "name=" in MIME headers */
	    /* FALLTHROUGH */

	case HT_CONTENT_XFER_ENCODING:
	case HT_CONTENT_ID:
	case HT_CONTENT_UNKNOWN:
	    if (nhdrs < MAX_HDRS)	/* copy interesting headers */
	    {
		AUSTRCPY(headers[nhdrs], temp);
		nhdrs++;	/* don't put this in the above macro! */
	    }
	    break;

	default:	/* Ignore non-MIME or unrecognised headers */
	    break;
	}						/*lint !e788 */ /*OK*/
    }

    /*
     * If the new entity we are just starting is a new multipart entity, ask
     * the user what to do with it (skip it or process it).
     */
    if (new_multi)
    {
	const char *typestr;

	switch (ct_type)
	{
	case CT_MULTI_MIXED:
	    typestr = "  multipart/mixed";
	    break;
	case CT_MULTI_ALTERN:
	    typestr = "  multipart/alternative";
	    break;
	case CT_MULTI_DIGEST:
	    typestr = "  multipart/digest";
	    break;
	case CT_MULTI_MISC:
	    typestr = (name[0]) ? name : "  multipart/? (unknown)";
	    break;
	default:		/* shouldn't happen */
	    typestr = "  (bad multipart type)";
	    break;
	}						/*lint !e788 */ /*OK*/

	strcpy(input, "1");
	do
	    if (getwin(5, 10, 1, col_select, input,
		       "MIME multipart message",
		       "A new multipart section has been found, of type",
		       typestr,
		       "",
		       " 1. Process it as usual",
		       " 2. Skip over it",
		       "",
		       select_max(2),	/* "Select your choice (1 - 2):  " */
		       NULL) == 0)
		input[0] = '2';
	while (input[0] != '1' && input[0] != '2');

	if (input[0] == '2')	/* Skip it: pretend we haven't seen it */
	    multidepth--;

	return SKIP_START;	/* Skip line, (re)start looking for boundary */
    }

    /*
     * We've (a) run out of file, or (b) found the end of the headers for this
     * entity.  <nhdrs> is the number of headers stored in the headers[]
     * array.  One thing we can be quite certain of is that the current
     * contents of <temp> are unimportant, so we can use it as temporary
     * storage for any other devious purpose.
     *
     * If the current entity is multipart/alternative, and this is the second
     * or later part, ask the user whether to skip it or not.  Leave <mp> set
     * for future reference.
     */
    if (multidepth > 0)
    {
	mp = &multipart[multidepth - 1];
	if (mp->type == CT_MULTI_ALTERN && mp->nparts > 1)
	{
	    char buf[128];
	    int ch;

	    sprintf(buf,						/*OK*/
		    "Alternative %u (%.*s) next: ESC/ret/spc=ignore, v=view",
		    mp->nparts,
		    (mp->nparts < 10) ? 30 : (mp->nparts < 100) ? 29 : 27,
		    (name[0]) ? skip_lwsp(name) : "unknown");
	    ch = ignore_alt(buf);
	    if (ch == 1)	/* Ignore, don't view */
		return SKIP_START;  /* Skip line, start looking for boundary */
	    if (ch > 1)		/* Keypress that should be obeyed now */
	    {
		/* We haven't acted on this boundary: pretend it's unseen */
		((struct multipart *) mp)->nparts--;
		*chp = ch;
		return HANDLE_KEY;
	    }
	    /* else ch == 0, view the alternative version */
	    warn_multi = FALSE;		/* Don't warn user about m/p bndry */
	}
    }

    /*
     * Find out some useful information about this entity's style, so we can
     * decide how to deal with it; we regard text/enriched, text/html, other
     * text/??? and message/rfc822 as "noteworthy" text types that the user
     * should be asked about, whereas text/plain is generally not worth
     * disturbing the user over (neither is message/delivery-status, come to
     * that).
     */
    if (ct_type == CT_TEXT_RICH || ct_type == CT_TEXT_HTML ||
	ct_type == CT_TEXT_MISC || ct_type == CT_MSG_RFC822 ||
	(mp && mp->type == CT_MULTI_DIGEST && ct_type == CT_NONE))
	text = TRUE;
    else if (ct_type == CT_NONE || ct_type == CT_TEXT_PLNASCII ||
	     ct_type == CT_TEXT_PLN88591 || ct_type == CT_MSG_DELIVSTAT)
	plaintext = TRUE;

    if (ct_enc == CE_BASE64 || ct_enc == CE_UUENCODE)
	binary = TRUE;		/* treat it as binary */

    *linetype = mime_linetype(ct_type, ct_enc);

    /*
     * If the entity is (a) plain text, (b) doesn't appear to have an
     * associated filename, and (c) is not binary encoded, just display it
     * without further ado.  (Is this *really* what we want to do? -ST)
     */
    if (plaintext && !has_name && !binary)
	return SHOW_FULL;		/* Show this entity in full */

    /*
     * If it's a real multipart entity, let the user know he's moving into a
     * new entity (unless he's already said he wants to view the next
     * multipart/alternative entity, in which case <multi_warn> will be FALSE)
     */
    if (multidepth > 0 && warn_multi)
    {
	/* TODO: improve prompt to describe the forthcoming entity type */
	int ch = view_cr("Multipart message - more to come");
	if (ch != 0)		/* Keypress that should be obeyed now */
	{
	    *chp = ch;
	    return HANDLE_KEY;
	}
    }

    if (nhdrs == 0)		/* No headers! */
	strcpy(headers[0], "** No headers (i.e. text/plain) **");

    /*
     * Don't automatically display it; ask the user what he wants done with
     * the data.  If we've got something we can handle, offer to decode it --
     * if it doesn't appear to have a name, and is a basically textual entity,
     * default to "see it as text" rather than "let me decode it".
     */
    if (!has_name && (plaintext || text) && !binary) /* default to "see it" */
	strcpy(input, "1");
    else					/* default to "decode it" */
	strcpy(input, "2");

    do
	if (getwin(3, 12, 1, col_select, input, headers[0],
		   headers[1], headers[2], headers[3], headers[4], "",
		   " 1. See it as text",
		   (binary) ?
		   " 2. Let me decode it" :
		   " 2. Write it to a file",
		   " 3. Skip this item",
		   "",
		   select_max(3),	/* "Select your choice (1 - 3):  " */
		   NULL) == 0)
	    return STOP_NOW;		/* Stop the display immediately */
    while (input[0] != '1' && input[0] != '2' && input[0] != '3');
    if (input[0] == '1')
	return SHOW_REMAINS;	/* Display remainder of entity as text */
    if (input[0] == '3')
	return SKIP_START;	/* Skip line, (re)start looking for boundary */

    /*
     * Decode the entity body to a file.  Get a file name for the attachment:
     * we set some rather basic defaults, then try and extract a "real" name
     * from (a) the stored MIME headers for this section, or (b) the "begin"
     * line of a uuencoded section.
     */
    if (ct_type == CT_IMAGE_GIF)
	strcpy(name, "OUTPUT.GIF");
    else if (ct_type == CT_IMAGE_JPEG)
	strcpy(name, "OUTPUT.JPG");
    else if (ct_type == CT_VIDEO_MPEG)
	strcpy(name, "OUTPUT.MPG");
    else if (ct_type == CT_AUDIO_AU)
	strcpy(name, "OUTPUT.AU");
    else if (ct_enc == CE_UUENCODE)
	strcpy(name, "OUTPUT.UUE");
    else
	strcpy(name, "OUTPUT.ENC");

    /*
     * Now look through *ALL* the saved headers (not just the first one) for a
     * file name to use
     */
    for (i = 0; i < MAX_HDRS; i++)
	if (((ptr = stristr(headers[i], "name=")) != NULL ||
	     (ptr = stristr(headers[i], "name")) != NULL) &&
	    getstring(temp, tempsize, ptr + 4, TRUE) > 0)
	{
	    AUSTRCPY(name, leafname(temp));
	    break;	/* out of the for (i): got a name */
	}

    if (ct_enc == CE_UUENCODE)
    {
	/*
	 * uuencoded attachment: run forwards to find the "begin " line, and
	 * (we hope) get the filename
	 */
	BOOL gotit = FALSE;
	gash[0] = '\0';		/* use <gash> for filename */
	while (ftell(efp) < end && fgets(temp, (int) tempsize, efp) != NULL)
	{
	    if (last_eol && uu_begin(temp, gash, sizeof(gash)) >= 0)
	    {
		gotit = TRUE;
		break;	/* out of the local while (ftell/fgets) */
	    }
	    last_eol = eoltest(temp);
	}
	if (!gotit)
	{
	    errorbox("UUDECODE Error",
		     "Can't find uuencode \"begin\" line!"
		     "Not a valid uuencoded attachment", NULL);
	    return STOP_NOW;		/* stop displaying immediately */
	}
	/* Got the "begin" line OK; did we get a name? */
	if (gash[0])
	    AUSTRCPY(name, leafname(gash));
    }

    /*
     * We've now got the name to use (which may be the default name if we
     * couldn't find one from the headers), and we're in the right place to
     * start decoding the attachment; ask the user to choose which name to
     * REALLY use, based on our suggestion, and decode the attachment if he
     * chooses a non-blank name.
     */
    if (getwin(15, 17, 50, col_select, name,
	       "Enter Filename (including full path name if desired)", NULL)
	> 0)
    {
	/*
	 * Got a non-blank name: decode the data to that file.  Once done,
	 * move to the line containing the ending boundary and run a program
	 * of the user's choice if he so chooses
	 */
	int redraw;
	long here = decode(efp, end, ct_enc, name);

	if (here < 0L)		/* it's all gone horribly wrong */
	    return SHOW_REMAINS;

	fseek(efp, here, SEEK_SET);	/* seek to terminating line */
	if (ct_enc == CE_BASE64 && ct_type == CT_IMAGE_GIF && gifview)
	    redraw = extprog(gifview, name, "view", "gif image");
	else if (ct_enc == CE_BASE64 && ct_type == CT_IMAGE_JPEG && jpegview)
	    redraw = extprog(jpegview, name, "view", "jpeg image");
	else if (ct_enc == CE_BASE64 && ct_type == CT_VIDEO_MPEG && mpegview)
	    redraw = extprog(mpegview, name, "view", "MPEG video");
	else if (ct_enc == CE_BASE64 && ct_type == CT_AUDIO_AU && audioview)
	    redraw = extprog(audioview, name, "listen to", "audio file");
	else if (miscview && *miscview)
	    redraw = extprog(miscview, name, NULL, NULL);
	else
	    redraw = SHOW_REMAINS;

	return (redraw) ? REDRAW_REMAINS : SHOW_REMAINS;
    }

    return SHOW_REMAINS;	/* display anything remaining as text */
}


/*
 * mime_linetype()
 *
 * Given the MIME <type> and <encoding> for a particular line, return one of
 * the following which described the type of data described by that line:
 *	LT_NORMAL	Nothing special: plain unencoded text
 *	LT_QPTEXT	Quoted-printable encoding of text (can be displayed)
 *	LT_QPBIN	Quoted-printable encoding of binary/non-text data
 *	LT_64TEXT	Base64 encoding of text (can be displayed)
 *	LT_64BIN	Base64 encoding of binary/non-text data
 */
static unsigned char mime_linetype(CTTYPE type, CETYPE encoding)
{
    BOOL istext;

    istext = (type == CT_NONE || type == CT_TEXT_PLNASCII ||
	      type == CT_TEXT_PLN88591 || type == CT_TEXT_RICH ||
	      type == CT_TEXT_HTML || type == CT_TEXT_MISC ||
	      type == CT_MSG_RFC822);
    if (encoding == CE_QUOTED)
	return (istext) ? LT_QPTEXT : LT_QPBIN;
    if (encoding == CE_BASE64)
	return (istext) ? LT_64TEXT : LT_64BIN;
    return LT_NORMAL;
}


/*
 * redraw_view()
 *
 * Redraw the current view screen as it was before something changed it
 * (e.g. running an external program); we're viewing message <cmsg> on a
 * screen with <screen_lines> lines; <dr> is an array of LINEINFO structs
 * describing the lines on the screen; <inheadptr> is a pointer to the
 * parent's "in header" variable, which should be filled in with the final
 * "are we in the header?" state once we're done.  <tstring> and
 * <tstring_size> describe a temp. buffer we can use for reading in new lines
 * from the file (global <mfile>) and to pass to print() to get the lines
 * displayed.
 *
 * Note the <mfile> file position when we start redrawing (we will change it
 * while redrawing) and restore it to that position when we're done; return
 * this file position.
 */
static long redraw_view(PCHTL cmsg, int screen_lines, int eohlndr,
			const LINEINFO *dr, BOOL *inheadptr,
			char *tstring, size_t tstring_size)
{
    int i;
    long here = ftell(mfile);	/* Note current file position */
    BOOL inheader = *inheadptr;

    draw_viewscreen(cmsg);	/* Recreate base view screen */
    for (i = 0; i < screen_lines; i++)
    {
	int n = winlines[i];

	if (n >= 0)		/* Redraw this screen line */
	{
	    long pos = dr[n].fpos;

	    inheader = (eohlndr < 0 || n < eohlndr);
	    fseek(mfile, pos, SEEK_SET);
	    if (fgets(tstring, LINELEN, mfile) == NULL)
		break;
	    gotoxy(1, i + 1);
	    (void) print_string(tstring, tstring_size, mfile, pos, NULL,
				(BOOL) (!inheader && dr[n].type == LT_QPTEXT));
	}
    }
    gotoxy(1, wincur + 1);
    fseek(mfile, here, SEEK_SET);
    *inheadptr = inheader;
    return here;
}


/*
 * last_line()
 *
 * Given the array of LINEINFO structs <dr>, the end-of-message file position
 * <end> and the file stream <fp>, find the last line of the screen text area
 * (if we have a complete screenful of displayed lines) and the start of the
 * next line after that; return the index of this last line, having set the
 * file stream pointer to the start of the next line (if we have any problems
 * with out-of-range lines, force it to <end> for safety).  If we don't have a
 * complete screenful of displayed lines, return -1 and leave the file stream
 * unchanged.
 */
static int last_line(const LINEINFO *dr, long end, FILE *fp)
{
    int target, n;
    long fpos;

    target = TOPLNDR + winlen - 2;	/* expected index for last line */
    n = win_last();			/* last displayed line */
    if (n < target)			/* don't have a screenful */
	return -1;

    /* seek to posn. of next line (after last on screen) */
    fpos = dr[n + 1].fpos;
    if (fpos <= dr[n].fpos || fpos > end)	/* ?!? */
    {
	char errbuf[81];
	sprintf(errbuf, "Last line %ld, end %ld, fpos %ld",		/*OK*/
		dr[n].fpos, end, dr[n + 1].fpos);
	errorbox("File position error",
		 "Found bogus file position past end of screen:",
		 errbuf,
		 "Moving to end of file",
		 NULL);
	fpos = end;
    }
    fseek(fp, fpos, SEEK_SET);
    return n;
}


/*
 * file_percent()
 *
 * Calculate how far through the message we are at the end of the current
 * displayed screen, and return that figure as a percentage: don't return 100%
 * unless we're actually right at the end of the message.  This message is
 * <totalsize> long, and the end is at position <end>.  <endidx>, if not < 0,
 * is the line index value at which we have previously established the end of
 * the message to be -- i.e. line <endidx - 1> is the last line which can be
 * displayed.
 */
static int file_percent(const LINEINFO *dr, long end, long totalsize,
			int idx, long fpos, int endidx)
{
    int n;
    long percent;

    n = win_last();		/* last displayed line */
    if (n > idx)		/* can't use <fpos>: try to find real pos */
    {
	if (n > 0 && endidx == n + 1)	/* last possible line */
	    return 100;
	fpos = dr[n + 1].fpos;	/* Posn. of next line (after last on screen) */
	if (fpos <= dr[n].fpos || fpos >= end)	/* ?!? assume it's the end */
	    fpos = end;
    }
    if (fpos >= end)
	return 100;
    percent = 100L - ((end - fpos) * 100L) / totalsize;
    if (percent > 99L)
	return 99;
    return (int) percent;	/* If it's -ve, we need to show this! */
}


/*
 * win_setlen()
 *
 * Set the length of the view window in lines
 */
static void win_setlen(int nlines)
{
    winlen = nlines;
    winbot = winlen - 1;
    winscrollsize = (size_t) winbot * sizeof(*winlines);
}


/*
 * win_init()
 *
 * Initialise all the line values to WL_BLANK
 */
static void win_init(void)
{
    int i;

    for (i = 0; i < winlen; i++)
	winlines[i] = WL_BLANK;
    wincur = 0;
}


/*
 * win_curidx()
 *
 * Set the index value for the current line to <val>
 */
static void win_curidx(int val)
{
    winlines[wincur] = val;
}


/*
 * win_newline()
 *
 * Move the cursor on to the next line (we've just printed one that caused
 * this to happen to the screen); if the cursor is now below the bottom line,
 * scroll the screen accordingly.
 */
static void win_newline(void)
{
    if (++wincur >= winlen)
    {
	/* window will have scrolled up by one line */
	memmove(winlines, winlines + 1, winscrollsize);
	winlines[wincur = winbot] = WL_BLANK;	/* bottom line is now blank */
    }
}


/*
 * win_clrline()
 *
 * Move to the line <y> and clear it with <col_normal>
 */
static void win_clrline(int y)
{
    win_gotoline(y);
    textattr(col_normal);
    clreol();
    winlines[y - 1] = WL_BLANK;
}


/*
 * win_scrollup()
 *
 * Scroll the screen up by one line, putting a blank line in at the bottom
 */
static void win_scrollup(void)
{
    winscrl(1);
    memmove(winlines, winlines + 1, winscrollsize);
    winlines[winbot] = WL_BLANK;	/* bottom line is now blank */
    /* cursor doesn't move */
}


/*
 * win_scrolldown()
 *
 * Scroll the screen down by one line, putting a blank line in at the top
 */
static void win_scrolldown(void)
{
    winscrl(-1);
    memmove(winlines + 1, winlines, winscrollsize);
    winlines[0] = WL_BLANK;		/* topline is now blank */
    /* cursor doesn't move */
}


/*
 * win_gotoline()
 *
 * Move to line <y> (both our cursor and the real screen one)
 */
static void win_gotoline(int y)
{
    gotoxy(1, y);
    wincur = y - 1;
}


/*
 * win_defline()
 *
 * Move to line <y> (both our cursor and the real screen one), and set its
 * index value to <val>
 */
static void win_defline(int y, int val)
{
    win_gotoline(y);
    winlines[wincur] = val;
}


/*
 * win_clrscr()
 *
 * Clear the screen, and set all our lines to blank
 */
static void win_clrscr(void)
{
    clrscr();
    win_init();
}


/*
 * win_last()
 *
 * Return the index value of the last "normal" non-blank line on the screen
 */
static int win_last(void)
{
    int i;

    for (i = 0; i < winlen; i++)
	if (winlines[i] < 0)		/* this line is blank or abnormal */
	    break;
    i--;
    return (i < 0) ? -1 : winlines[i];
}


/*
 * draw_viewscreen()
 *
 * Create a blank viewmsg screen: the top two lines contain info about the
 * message and a separator, then the area below that is a scrolling window.
 * Get the entire screen, clear it, draw the top two (fixed) lines, then set
 * the smaller scrolling window that will be used for text.
 */
static void draw_viewscreen(PCHTL cmsg)
{
    disp_clear();		/* Full screen, cleared */
    /* Line 1: title line */
    cprintf("Msg %d from %s, Subject %s",
	    cmsg->listno,
	    (cmsg->headers & H_FROM) ? cmsg->from : somebody,
	    (cmsg->headers & H_SUBJECT) ? cmsg->subject : "");
    /* Line 2: dividing line */
    writef(1, 2, col_hilite, seqstr(display.screenwidth, '\315'));
    /* Now fix the top FIXED_TOPLINES lines; scroll from here to bottom */
    disp_fixtop(FIXED_TOPLINES);
}
