/*
  This source is part of PCOak, an electronic mailer for DOS based on PCElm.

  PCElm is Copyright (c) 1988-1993 Martin Freiss and Wolfgang Siebeck
           Copyright (c) 1992-1999 Demon Internet
  PCOak is Copyright (c) 2000-2002 Simon Turner, Pete Disdale and dispc members

  Thanks to an agreement between the original PCElm authors and Demon Internet
  made in late 1999:

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License, version 1, as
	published by the Free Software Foundation.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	See the file COPYING, which contains a copy of the GNU General
	Public License.
*/

/*
 * config.c -- program and global variable configuration
 */

#include <stdio.h>
#include <alloc.h>
#include <conio.h>
#include <stdlib.h>
#include <dir.h>
#include <dos.h>
#include <ctype.h>
#include <string.h>
#include <time.h>
#include "pcoak.h"
#include "display.h"
#include "notepad.h"
#include "ustring.h"
#include "chars.h"
#include "filename.h"
#include "macros.h"

/*
 * Global variables defined elsewhere and used here
 */
extern const char str_builtin[];	/* "builtin+" */
extern const char wrkfileext[];		/* ".wrk" */

/*
 * Program configuration: there are squillions of global variables which need
 * to be correctly set up before we can start.  The entry point to all this
 * lot is setup_globals() below; it does various stuff, calls read_rc() to
 * read in the RC file, then tidies up.  If we get out of setup_globals()
 * without crashing out of the program with a fatal error, we should have most
 * of a working setup.
 */

/************************************************************************/
/* Global vars defined once at startup/config, and thereafter unchanged	*/
/************************************************************************/

/* Directly configurable by the user */
//extern BOOL enablemime;	/* Only try to handle MIME if this is set */
extern BOOL nosigdashes;	/* Supress the --_ before .signatures */
extern BOOL slash_ed;		/* Use / on editor filename instead of \ */
extern BOOL use_ed;		/* Drop into editor when creating messages */
extern BOOL asktoggle;		/* Ask for new host/fullname on file change */
extern BOOL dosify;		/* Change files to DOS EOL */
extern int saveall;		/* Save all outgoing messages to =sent (0-3) */
extern BOOL saverecd;		/* Default save messages to =received */
extern BOOL autosave;		/* Don't ask whether to save changes */
extern BOOL dotmenu;		/* If true then use . .q .e .? etc. */
extern BOOL autodel;		/* Don't auto flag as delete after save */
extern BOOL betterread;		/* Use Better Read it First msg? (eughh!) */
extern BOOL askdelqueue;	/* Ask before deleting from mail queue */
extern BOOL exitconfirm;	/* Ask on Exit? Default to FALSE */
extern BOOL sortdescend;	/* TRUE for descending, FALSE for ascending */
extern SORTBYTYPE sortby;	/* Sort order (SORT_INCOMING / SORT_DATE) */
extern char escape_char;	/* "Dot menu" escape character (default '.') */
extern char movedirection;	/* Current move direction ('D', 'A' or 'U') */
extern char *quotestr;		/* for quoting include msgs in R)eply */
extern char *username;		/* name of this user from rc file */
extern char *useraddr;		/* local address of this user from rc file */
extern char *replyto;		/* address for Reply-To: header field */
extern char *uucpreplyto;	/* addr. for Reply-To: header (uucp mode) */
extern char *maildir;		/* defined mail directory */
extern char *editor;		/* user's favourite text editor */
extern char *pager;		/* user's favourite pager */
extern char *gifview;		/* user's favourite gif viewer */
extern char *jpegview;		/* user's favourite jpeg viewer */
extern char *mpegview;		/* user's favourite mpeg viewer */
extern char *audioview;		/* user's favourite audio player */
extern char *miscview;		/* user's favourite miscellaneous viewer */
extern char *multimedia;	/* user's favourite multimedia program */
extern char *mboxext_store[NUM_SYSTEMS]; /* mailbox file extensions */
extern char *uucp_command;	/* cmd to use when spawning rmail */
extern char *sequencename;	/* name of sequence file */
//extern char *mailqdir;	/* dir for spooling outgoing mail */
extern char *printdevice;	/* Used print device, if not spooling */
extern char **weedlist1;	/* Headers to weed out for weedlevel 1 */
extern char **weedlist2;	/* Headers to display for weedlevel 2 */
extern char *opthdr[MAXOPTHDR];	/* List of optional header lines */
extern int smarthost;		/* 0 = never, 1 = always, 2 = you_know */
extern char *mailhost;		/* name of our smart host */
extern int weedstyle;		/* Style of weedout to use (0, 1 or 2) */
extern int printweed;		/* Level of weedout for printing (0-3) */
extern BOOL cnfmsendabort;	/* Confirm aborting message to be sent? */
extern BOOL cnfmfoldcreate;	/* Confirm creating new folder when saving? */
extern BOOL viewdelread;	/* Auto-read next msg when 'd' from viewer? */
extern BOOL viewnextunread;	/* Auto-read next msg when 'n' from viewer? */
extern BOOL retendquit;		/* Quit the pager with <Return> at end? */
extern int enclong;		/* 0 = never, 1 = always, 2 = prompt */

/* Not directly configurable by the user */
extern char *cwd;		/* current directory */
//extern char *homedir;		/* user's home directory */
extern char *firstname;		/* name of FIRST (main) user; used by "c!" */
extern char *mailqueue;		/* workjob filename wildcard, outgoing mail */
extern char *tfname;		/* file for mailtext */
extern char *mfname;		/* file for mailtext w/ header */
extern int printernumber;	/* printer port to use */
extern time_t yeartime;		/* UTC time before which we display year */

/************************************************************************/
/* Global vars defined at startup/config, but may be changed later	*/
/************************************************************************/

/* Directly configurable by the user */
//extern int used_mailsystem;	/* 1: use smtp (ham radio); 0: spawn 'rmail'*/
//extern int col_normal;	/* Normal colour attribute */
//extern int col_enhanc;	/* Enhanced, errors etc. */
//extern int col_select;	/* Selected message in menu */
//extern int col_system;	/* Attribute at startup */
//extern int col_hilite;	/* Highlighted Letters etc. */
//extern int col_heading;	/* Heading colour attribute */
extern int extpager;		/* External pager usage (0-2) (Changeable) */
extern int weedlevel;		/* Level of weedout to apply (0-3) (Chgable) */
extern BOOL insertmode;		/* Insert or overwrite in maxgets()? */
extern BOOL show_help;		/* Do we show the main screen help? */
//extern char *fullname;	/* full name of this user from rc file */
extern char *hostname_store[NUM_SYSTEMS]; /* names of this host */
extern char *startofhdr_store[NUM_SYSTEMS]; /* SOH strings for all systems*/
//extern struct notepad_parms npsetup;	/* Builtin "notepad" editor settings */

/* Not directly configurable by the user */
extern BOOL mmdf_store[NUM_SYSTEMS]; /* <mmdf> storage for all systems */
//extern char notename[];		/* Name of current mailbox */
extern char *hdr_fullname;	/* User's full name, for use in header */
extern BOOL color_card;		/* Colour display (TRUE) or mono (FALSE)? */

/************************************************************************/

/*
 * The RC file configuration of variables is done with an array of configentry
 * structs (defined below); we then hammer through the file, comparing each
 * line in the file with the entries in the array and setting up the variable
 * described by a matching entry according to its configuration type, which is
 * one of the configtype enums below.
 *
 * Once we've read the RC file in and processed all the entries, there is
 * still some work to be done to make sure that defaults are applied for
 * variables that weren't set up in the RC file, and to do some range checking
 * (with resetting to defaults if out of range) etc.  One-off checking is done
 * longhand, but if there's more than one variable which needs the same type
 * of checking, it is done with arrays of strdefault or intdefault structs.
 */

enum configtype {	/* All the possible types of RC entry */
    /* Path-type strings */
    C_PATH,			/* File path string (char **) */
    C_CMD,			/* Command string (char **) */
    C_ECMD,			/* May-be-empty command string (char **) */
    /* Normal strings */
    C_STR,			/* Generic string (char **) */
    C_FSTR,			/* First-definition-only string (char **) */
    C_STR5,			/* 5-char string, padded (char [6]) */
    /* Booleans, ints etc. */
    C_BOOL,			/* Boolean (BOOL *) */
    C_RBL,			/* Reversed boolean [1 -> FALSE] (BOOL *) */
    C_INT,			/* Integer (int *) */
    C_CHAR,			/* Single char (char *) */
    /* Specials */
    C_WEED,			/* Header weeding list (char ***) */
    C_SYS,			/* Type of system (SYS_HAM or SYS_UUCP) */
    C_COLR,			/* Specify colours to use */
    C_SORT,			/* Define message sort order */
    C_CRSR,			/* Specify shapes for all cursors */
    C_YRTM			/* The <yeartime> timespan in months or days */
};

#define LAST_PATH	C_ECMD	/* Last path-using configtype enum */

struct configentry {	/* Describe a config file entry */
    const char *label;		/* RC entry label */
    enum configtype type;	/* Type of variable */
    void *varptr;		/* Generic pointer to variable */
};

struct strdefault {	/* Describe a string's default settings */
    char **ptr;			/* Address of string pointer variable */
    const char *envvar;		/* User environment variable */
    const char *defstr;		/* Fixed last-ditch default string */
    const char *errstr;		/* Error string if not defined by user */
};

struct intdefault {	/* Describe a range of acceptable integers */
    int *ptr;			/* Pointer to integer variable */
    int minval;			/* Minimum acceptable value */
    int maxval;			/* Maximum acceptable value */
    int defval;			/* Default value if out of bounds */
};

/*
 * The main list of all known RC file entries, with what to do with them; the
 * order is basically the same as the old PCElm order (which probably has
 * historical significance, as new entries were tacked on to the end of the
 * enum), but modified slightly to group related items together better
 *
 * IMPORTANT!  For backwards compatibility reasons, matching of labels only
 * checks for minimal matches: so the RC file string "username" would match
 * the label "user", for instance.  It is *IMPERATIVE* that no label is ever
 * added to this list which is a superstring of a previous label, for proper
 * backwards compatibility with PCElm if nothing else.  (This is why the RC
 * label for <useraddr> is <usraddr>, for example.)
 *
 * For the same reason of needing a new PCOak RC file to work with old PCTree
 * executables,, we must also keep track of every configuration entry that has
 * ever been used in a released version of PCTree, and make sure we don't
 * clash with those.
 */
static struct configentry cfglist[] = {
    /* label			type	varptr			*/
    { "smtp",			C_PATH,	&maildir		},
    { "host",			C_STR,	&hostname_store[SYS_HAM]  },
    { "uucphost",		C_STR,	&hostname_store[SYS_UUCP] },
    { "user",			C_FSTR,	&username		},
    { "fullname",		C_STR,	&fullname		},
    { "reply",			C_STR,	&replyto		},// NB!
    { "hamreply",		C_STR,	&replyto		},// NB!
    { "uucpreply",		C_STR,	&uucpreplyto		},
    { "video",			C_INT,	&directvideo		},
    { "SOH-ham",		C_STR,	&startofhdr_store[SYS_HAM]  },
    { "SOH-UUCP",		C_STR,	&startofhdr_store[SYS_UUCP] },
    { "start",			C_SYS,	NULL			},
    { "ham-ext",		C_STR,	&mboxext_store[SYS_HAM]  },
    { "uucp-ext",		C_STR,	&mboxext_store[SYS_UUCP] },
    { "colors",			C_COLR,	NULL			},
    { "queuedir",		C_PATH,	&mailqdir		},
    { "sequence",		C_STR,	&sequencename		},
    { "print",			C_STR,	&printdevice		},
    { "weedout",		C_WEED,	&weedlist1		},
    { "escape",			C_CHAR,	&escape_char		},
    { "use-ed",			C_BOOL,	&use_ed			},
    { "help",			C_RBL,	&show_help		},
    { "opthdr0",		C_STR,	&opthdr[0]		},
    { "opthdr1",		C_STR,	&opthdr[1]		},
    { "opthdr2",		C_STR,	&opthdr[2]		},
    { "opthdr3",		C_STR,	&opthdr[3]		},
    { "opthdr4",		C_STR,	&opthdr[4]		},
    { "opthdr5",		C_STR,	&opthdr[5]		},
    { "opthdr6",		C_STR,	&opthdr[6]		},
    { "opthdr7",		C_STR,	&opthdr[7]		},
    { "opthdr8",		C_STR,	&opthdr[8]		},
    { "opthdr9",		C_STR,	&opthdr[9]		},
    { "quote",			C_STR,	&quotestr		},
    { "edit",			C_CMD,	&editor			},
    { "uucpcall",		C_PATH,	&uucp_command		},//C_CMD?
    { "pager",			C_CMD,	&pager			},
    { "multimedia",		C_PATH,	&multimedia		},//C_CMD?
    { "ask-toggle",		C_BOOL,	&asktoggle		},
    { "dosify",			C_BOOL,	&dosify			},
    { "exitconfirm",		C_BOOL,	&exitconfirm		},
    { "saveall",		C_INT,	&saveall		},
    { "defaultreceived",	C_BOOL, &saverecd		},
    { "enablemime",		C_BOOL,	&enablemime		},
    { "nosigdashes",		C_BOOL,	&nosigdashes		},
    { "autosave",		C_BOOL,	&autosave		},
    { "slash_ed",		C_BOOL,	&slash_ed		},
    { "dotmenu",		C_BOOL,	&dotmenu		},
    { "autodel",		C_BOOL,	&autodel		},
    { "betterread",		C_BOOL,	&betterread		},
    { "askdelqueue",		C_BOOL,	&askdelqueue		},
    { "builtin-tabsize",	C_INT,	&npsetup.edit_tabsize	},
    { "builtin-leftmargin",	C_INT,	&npsetup.edit_lira	},
    { "builtin-rightmargin",	C_INT,	&npsetup.edit_rera	},
    { "builtin-wordwrap",	C_BOOL, &npsetup.wrap		},
    { "builtin-insert",		C_BOOL,	&npsetup.edit_insrt	},
    { "builtin-key-end",	C_STR5,	npsetup.key_end		},
    { "builtin-confirm-abort",	C_BOOL, &npsetup.confirm_abort	},
    { "gifview",		C_CMD,	&gifview		},
    { "jpegview",		C_CMD,	&jpegview		},
    { "mpegview",		C_CMD,	&mpegview		},
    { "audioview",		C_CMD,	&audioview		},
    { "miscview",		C_ECMD,	&miscview		},
    { "sortdirection",		C_BOOL,	&sortdescend		},
    { "sortby",			C_SORT,	NULL			},
    { "movedirection",		C_CHAR,	&movedirection		},
    { "smarthost",		C_INT,	&smarthost		},
    { "mailhost",		C_STR,	&mailhost		},
    { "extpager",		C_INT,	&extpager		},
    { "weedlevel",		C_INT,	&weedlevel		},
    { "weedstyle",		C_INT,	&weedstyle		},
    { "weedlist1",		C_WEED,	&weedlist1		},
    { "weedlist2",		C_WEED,	&weedlist2		},
    { "prnweedlevel",		C_INT,	&printweed		},
    { "insertmode",		C_BOOL,	&insertmode		},
    { "cursors",		C_CRSR,	NULL			},
    { "confirmsendabort",	C_BOOL,	&cnfmsendabort		},
    { "confirmfoldcreate",	C_BOOL,	&cnfmfoldcreate		},
    { "usraddr",		C_STR,	&useraddr		},
    { "viewdelread",		C_BOOL,	&viewdelread		},
    { "viewnextunread",		C_BOOL,	&viewnextunread		},
    { "retendquit",		C_BOOL,	&retendquit		},
    { "yeardispage",		C_YRTM,	NULL			},
    { "enclong",		C_INT,	&enclong		},
    /* Obsolete RC file entries which we must avoid clashing with: */
    /*	"delete-unread"		*/
    /*	"fido"			*/
    /*	"hamhide"		*/
    /*	"maxlet"		*/
    /*	"uucphide"		*/
    /*	"zone"			*/
};

/*
 * Defaults for some strings, both old-style (deprecated) environment
 * variables and fixed strings, with error message codes and "required" status
 * for what we do if the variable isn't defined: variables which the user is
 * required to specify MUST NOT have a fixed default string; those which the
 * user doesn't have to set, but which are required to be valid, MUST have a
 * fixed default string, which is set (after the user has been warned) if the
 * variable isn't specified by the RC file or an env. variable
 *
 * N.B. >>> If defstr is NULL, you *MUST* provide a valid errstr <<<
 */
static struct strdefault strdefs[] = {
    /* ptr			envvar		defstr		errstr	*/
    { &hostname_store[SYS_HAM],	"HAMNODENAME",	NULL,		"Hostname for Internet undefined"	},
    { &hostname_store[SYS_UUCP],"NODENAME",	NULL,		"Hostname for uucp undefined"		},
    { &username,		"MAILBOX",	NULL,		"Username undefined"			},
    { &maildir,			"MAILDIR",	NULL,		"Mail directory not defined"		},
#if 1	/* Probably not a good idea, but for compatibility... */
    { &mailqdir,		"SPOOLDIR",	"\\spool\\mail","Mailqueue directory not defined"	},
#else
    { &mailqdir,		"SPOOLDIR",	NULL,		"Mailqueue directory not defined"	},
#endif
    { &fullname,		"NAME",		"",		"Full name not defined"					},
    { &startofhdr_store[SYS_HAM],NULL,		"From ",	"SOH for Internet not defined, using default"		},
    { &startofhdr_store[SYS_UUCP],NULL,		"From ",	"SOH for uucp not defined, using default"		},
    { &mboxext_store[SYS_HAM],	NULL,		".txt",		"Mailboxextension for Internet not defined, using default" },
    { &mboxext_store[SYS_UUCP],	NULL,		".",		"Mailboxextension for uucp not defined, using default"	},
    { &uucp_command,		NULL,		"",		"No uucp mailer specified, can't send mail"		},
    { &sequencename,		NULL,		"seqf",		"No sequence file specified, creating one"		},
    { &editor,			"EDITOR",	str_builtin, 	NULL	},
    { &quotestr,		NULL,		"> ",		NULL	},
    { &printdevice,		NULL,		"PRN",		NULL	},
    { &miscview,		NULL,		"program",	NULL	},
};

/*
 * Range checking and default values (if out of range) for integers
 */
static struct intdefault intdefs[] = {
    /* ptr		minval	maxval			defval		*/
    { &directvideo,	0,	1,			1		},
    { &saveall,		0,	3,			0		},
    { &smarthost,	0,	2,			0		},
    { &extpager,	0,	NUM_EXTPAGER - 1,	EP_EXTMOSTLY	},
    { &weedlevel,	0,	NUM_WEEDLEVEL - 1,	HW_WEEDED	},
    { &weedstyle,	0,	2,			0		},
    { &printweed,	0,	NUM_WEEDLEVEL - 1,	HW_FULL		},
    { &enclong,		0,	2,			1		},
};

/************************************************************************/

/*
 * And finally, some more global variables local to this module
 */
static const char setupmemerr[] = "Out of memory during program setup!";

/*
 * Functions defined and used here
 */
static void read_rc(void);
static char *strinit(const char *src);
static char *cmdinit(const char *template);
static void check_strdefs(void);
static void check_intdefs(void);
static void rcerror(int err, const char *label);
static void fatal(const char *x);
static void warning(const char *x);
static void noslash(char *s);
static char *parsestr(const char *s, BOOL escapes);
static void free_strarray(char **array, unsigned nelem);
static void free_strlist(char **list);

/*
 * Functions defined elsewhere and used here
 */
int get_current_attribute(void);
void set_system(void);
BOOL init_colors(BOOL have_color);
void backslash(char *s);
void cbackslash(char *s);
void fwdslash(char *s);
char *protect_name(const char *name);
char rip(char *s);
void define_cursors(unsigned char ovrtop, unsigned char ovrbot,
		    unsigned char instop, unsigned char insbot);
int set_weedlist(char ***list, const char *line);



/*
 * setup_globals()
 *
 * (Used to be called SetupGlobals(), but I hate NamesWithMixedCapitals.)
 *
 * Fundamental setup of all the configurable global variables in PCOak.  After
 * we return from this function, everything should be correctly set up.
 *
 * This functions is called only once, right near the start of main(): there
 * is therefore no point setting default values here for global variable which
 * mirror the compile-time defaults, as they will have those values anyway.
 */
void setup_globals(void)
{
    char pathbuf[PATHLEN];
    char *evp;

    /*
     * Get the current working directory; this is where we spend all our time
     */
    if (getcwd(pathbuf, sizeof(pathbuf)) == NULL)
	fatal("Can't get working directory!");
    noslash(pathbuf);		/* ensure no trailing / or \ */
    cwd = strinit(pathbuf);

    /*
     * Set <homedir> to the directory where the PCOAK.RC file lives; we can
     * then get on with reading the RC file
     */
    if ((evp = getenv("PCOAK")) != NULL ||	/* got PCOak env. variable */
	(evp = getenv("PCELM")) != NULL ||	/* got PCElm backup variable */
	(evp = getenv("HOME")) != NULL)		/* got final backup variable */
	homedir = strinit(evp);
    else					/* can't get "home" variable */
	homedir = strinit(".");
    noslash(homedir);		/* ensure no trailing / or \ */

    /*
     * Set default values for various things, some of which may get re-set
     * later by the user's configuration.  Note that there's little point in
     * re-setting things which are the compile-time defaults for
     * external-scope variables anyway!  This applies as much to array
     * contents (which will all be 0) as to values.
     */
    col_system = get_current_attribute();
    color_card = init_colors(color());

    mfname = strinit(AUSTRLCPY2(pathbuf, cwd, "\\uucpmail.txt"));
    tfname = strinit(AUSTRLCPY2(pathbuf, cwd, "\\mailtext.txt"));
    /* backslash(mfname); */		/* Not needed */
    /* backslash(tfname); */		/* Not needed */

    /*
     * Now read the RC file PCOAK.RC; all the strings are currently NULL, so
     * we can set them with impunity.  Once the file has been read, various
     * checks are made to make sure everything we need has been specified --
     * if anything is missing, it's a fatal error and the function never
     * returns.
     */
    read_rc();

    /*
     * The RC file has been read; all required variables should now have been
     * set to something, so we can do any final tidying up now.
     */
    textattr(col_normal);
    noslash(maildir);
    noslash(mailqdir);
    backslash(maildir);
    backslash(mailqdir);
    mailqueue = strinit(AUSTRLCPY3(pathbuf, mailqdir, "\\*", wrkfileext));

    set_system();	/* Set the system-specific bits */

    if (slash_ed)  /* convert to forward slashes as vi can't handle c:\etc */
	fwdslash(tfname);

    /* It's OK to pass NULL pointers to cbackslash() ... */
    cbackslash(editor);
    cbackslash(gifview);
    cbackslash(jpegview);
    cbackslash(mpegview);
    cbackslash(audioview);
    cbackslash(miscview);
    ustrncpy(notename, username, 8);
    firstname = strinit(notename);
    if ((hdr_fullname = protect_name(fullname)) == NULL)
	fatal(setupmemerr);

    /* That's it! */
}


/*
 * read_rc()
 *
 * Read the RC file; this function assumes that all dynamic pointers are
 * already NULL, so does no checking.  It must only be called once!
 */
static void read_rc(void)
{
    char buf[512];		/* big enough for filenames and file lines */
    FILE *rcfp;			/* handle for the configuration file */
    char *arg, *s, *evp;
    unsigned i;
    size_t l;
    struct configentry *cep;
    int err;			/* 0 = OK; 1 = warn; -1 = fatal */

    AUSTRLCPY3(buf, homedir, "\\", RCFILE1);		/* pcoak.rc */
    if ((rcfp = fopen(buf, "rb")) == NULL)
    {					/* Failed; try backup file */
	AUSTRLCPY3(buf, homedir, "\\", RCFILE2);	/* pcelm.rc */
	if ((rcfp = fopen(buf, "rb")) == NULL)
	{				/* Can't open either config file! */
	    textattr(col_hilite);
	    cprintf("Cannot open %s, check your installation:\r\n", buf);
	    cprintf("Does the PCOAK/PCELM/HOME environment variable contain the complete\r\n");
	    cprintf("path to the PCOAK/PCELM.RC file? Check the manual for more details.\r\n");
	    textattr(col_system);
	    exit(-1);
	}
    }

    while (fgets(buf, sizeof(buf), rcfp) != NULL)
    {
	(void) rip(buf);
	if (buf[0] == '#')		/* Comment line */
	    continue;
	if (strlen(buf) < 5)		/* Too short */
	    continue;
	s = skip_lwsp(buf);		/* Skip leading whitespace */
	for (i = 0, cep = cfglist; i < (int) NELEM(cfglist); i++, cep++)
	    if (strnicmp(s, cep->label, l = strlen(cep->label)) == 0)
	    {
		err = 0;
		arg = skip_lwsp(s + l);
		if (!*arg)	/* Ignore lines with no argument at all */
		    break;	/* out of the for (i) */
		/*
		 * Parse the argument (ignoring \-escapes if a path-type
		 * entry -- C_PATH, C_CMD, C_ECMD -- which may use \ in paths;
		 * the path-type entries are the first enums, with LAST_PATH
		 * being a macro for the last of them, so the test is simply
		 * "cep->type > LAST_PATH").  If the parsing fails, raise a
		 * warning and ignore the line.
		 */
		arg = parsestr(arg, (cep->type > LAST_PATH) ? TRUE : FALSE);
		if (arg == NULL)	/* Invalid argument */
		{
		    rcerror(1, cep->label);
		    break;	/* out of the for (i) */
		}
		switch (cep->type)
		{
		case C_FSTR:	/* First-definition-only string (char **) */
		    if (*((char **) cep->varptr) != NULL)   /* Already set */
			break;
		    /* fall through */

		case C_PATH:	/* File path string (char **) */
		case C_CMD:	/* Command string (char **) */
		case C_ECMD:	/* May-be-empty command string (char **) */
		case C_STR:	/* Generic string (char **) */
		    if (*((char **) cep->varptr) != NULL)   /* Already set; */
			free(*((char **) cep->varptr));     /* discard */
		    if (!*arg && cep->type != C_ECMD)
			*((char **) cep->varptr) = NULL;	  /* Null */
		    else if (cep->type == C_CMD || cep->type == C_ECMD)
			*((char **) cep->varptr) = cmdinit(arg);  /* Command */
		    else
			*((char **) cep->varptr) = strinit(arg);  /* Normal */
		    break;

		case C_STR5:	/* 5-char string, padded (char [6]) */
		    sprintf((char *) cep->varptr, "%-5.5s", arg);	/*OK*/
		    break;

		case C_BOOL:	/* Boolean (BOOL *) */
		    *((BOOL *) cep->varptr) = (*arg != '0');
		    break;

		case C_RBL:	/* Reversed boolean (BOOL *) */
		    *((BOOL *) cep->varptr) = (*arg == '0');
		    break;

		case C_INT:	/* Integer (int *) */
		    *((int *) cep->varptr) = atoi(arg);
		    break;

		case C_CHAR:	/* Single char (char *) */
		    *((char *) cep->varptr) = *arg;
		    break;

		case C_WEED:	/* Header weeding list (char ***) */
		    err = set_weedlist(cep->varptr, arg);
		    if (err < 0)	/* out of memory */
			fatal(setupmemerr);
		    /* If <err> > 0, display message at the end */
		    break;

		case C_SYS:	/* Type of system (SYS_HAM/SYS_UUCP) */
		    used_mailsystem = strnicmp(arg, "uucp", 4) ? SYS_HAM :
			SYS_UUCP;
		    break;

		case C_COLR:	/* Specify colours to use */
		    /* Try to read all the colours; don't worry if we fail */
		    (void) sscanf(arg, "%2x,%2x,%2x,%2x,%2x,%2x",
				  &col_normal,	/* normal color */
				  &col_enhanc,	/* enhanced attr */
				  &col_select,	/* selection in menu */
				  &col_system,	/* attribute at startup */
				  &col_hilite,	/* hilited */
				  &col_heading);/* heading */
		    if (col_heading == 0)	/* Not given */
			col_heading = 0x71;
		    break;

		case C_SORT:	/* Define message sort order */
		    sortby = (atoi(arg) == 0) ? SORT_INCOMING : SORT_DATE;
		    break;

		case C_CRSR:	/* Specify shapes for all cursors */
		    {
			unsigned u1, u2, u3, u4;
			if (sscanf(arg, "%u,%u,%u,%u", &u1, &u2, &u3, &u4) < 4
			    || u1 > 255 || u2 > 255 || u3 > 255 || u4 > 255)
			    err = 1;
			else
			    define_cursors((unsigned char) u1,
					   (unsigned char) u2,
					   (unsigned char) u3,
					   (unsigned char) u4);
		    }
		    break;

		case C_YRTM:	/* The <yeartime> timespan in months or days */
		    {
			long n;
			char c;
			if (sscanf(arg, "%ld %c", &n, &c) != 2 || n <= 0L ||
			    ((c = (char) tolower(c)) != 'd' && c != 'm'))
			    err = 1;
			else
			{
			    time_t now = time(NULL);
			    long span;

			    if (c == 'm')	/* <n> months ago */
				n = (n * 365L) / 12L;	/* avg month length */
			    span = n * 24L * 60L * 60L;	/* days -> seconds */
			    if (now > (time_t) span)
				yeartime = (time_t) ((long) now - span);
			    else
				yeartime = 0L;
			}
		    }
		    break;

		}		/* switch (cep->type) */
		if (err != 0)
		    rcerror(err, cep->label);
		break;	/* out of the for (i) */
	    }
    }
    (void) fclose(rcfp);				/* OK: read only */

    /*
     * Now we can get on with various checks and default-setting operations
     */

#if 1			/* Deprecated: backwards compatibility ONLY */
    if (username != NULL && getenv("MAILBOX") != NULL)
    {				/* Force use of env. variable for this one */
	free(username);
	username = NULL;
    }
#endif

    check_strdefs();		/* Check the string variables */
    check_intdefs();		/* Check the integer variables */

    /* <pager> default needs to use cmdinit(), so not in strdefs() */
    if (!pager && (evp = getenv("PAGER")) != NULL)
	pager = cmdinit(evp);

    /* Check <movedirection> is known character, and uppercase */
    if (strchr("UuDdAa", movedirection) == NULL)	/* Nope! */
	movedirection = 'A';
    else
	movedirection = (char) toupper(movedirection);

    /*
     * <useraddr>, if not defined by the user in the RC file, is set to THE
     * SAME POINTER as <username>.  This is important; if the two pointers
     * are the same, we mustn't free it twice when it comes to free_globals()!
     */
    if (!useraddr)
	useraddr = username;

    /*
     * And now, some post-config score settling
     */
    mmdf_store[SYS_HAM]  =
	(strnicmp(startofhdr_store[SYS_HAM],  "mmdf", 4) == 0);
    mmdf_store[SYS_UUCP] =
	(strnicmp(startofhdr_store[SYS_UUCP], "mmdf", 4) == 0);

    /* Set the printer port number from the defined printer name */
    if (stricmp(printdevice, "PRN") == 0)
	printernumber = 0;
    else
    {
	printernumber = *(printdevice + 3) - '1';    /* LPT1 == 0 and so on */
	if (strnicmp(printdevice, "LPT", 3) || printernumber < 0 ||
	    printernumber > 3)
	{
	    char errtxt[SLINELEN];

	    printernumber = 99;		/* warning */
	    AUSTRLCPY2(errtxt, "Unknown printer ", printdevice);
	    warning(errtxt);
	}
    }

    /* Replace '_' in the quote string by ' ' */
    while ((s = strchr(quotestr, '_')) != NULL)
	*s = ' ';

    /* Sanity checks for <smarthost> and <extpager> */
    if (mailhost == NULL)		/* 'mailhost' not defined */
	smarthost = 0;
    if (!pager || pager[0] == '\0')	/* no external pager defined */
	extpager = EP_INTONLY;

    /* That's it! */
}


/*
 * strinit()
 *
 * Try to take a duplicate copy of <src> and return a pointer to it; this is
 * for the initial program setup, so if the strdup() fails we report a fatal
 * error and abort immediately.
 */
static char *strinit(const char *src)
{
    char *copy = strdup(src);

    if (copy == NULL)
	fatal(setupmemerr);
    return copy;
}


/*
 * cmdinit()
 *
 * Given a command string template <template>, which may contain "%s" (which
 * needs to be replaced by a single '$') or may not contain '$' (which means
 * we have to append " $"), create a dynamic string which contains the correct
 * text and return it.  This is only used at startup; memory failures are
 * fatal.  We handle null strings specially; these are simply copied, without
 * getting " $" appended.
 */
static char *cmdinit(const char *template)
{
    size_t len;
    const char *pp;
    char *str;

    if (!*template)				/* null string! */
	return strinit(template);
    len = strlen(template);
    if ((pp = strstr(template, "%s")) != NULL)	/* Replace "%s" with "$" */
    {
	size_t l1 = (size_t) (pp - template);

	if ((str = malloc(len)) == NULL)	/* <str> is 1 byte shorter */
	    fatal(setupmemerr);
	(void) memcpy(str, template, l1);	/* copy up to "%s" */
	str[l1] = '$';				/* add '$' */
	(void) memcpy(&str[l1 + 1], pp + 2, len - l1 - 1);	/* copy rest */
    }
    else if (strchr(template, '$') == NULL)	/* Append " $" */
    {
	if ((str = malloc(len + 3)) == NULL)	/* <str> is 2 bytes longer */
	    fatal(setupmemerr);
	memcpy(str, template, len);	/* Copy text of <template> */
	strcpy(str + len, " $");	/* Append " $" and terminator */
    }
    else					/* Just copy the template */
	str = strinit(template);
    return str;
}


/*
 * check_strdefs()
 *
 * Check certain string variables: if they are unset, they may have an
 * environment variable to use, or a fixed default
 */
static void check_strdefs(void)
{
    unsigned i;
    struct strdefault *sdp;
    char *evp;

    for (i = 0, sdp = strdefs; i < NELEM(strdefs); i++, sdp++)
    {
	if (*((char **) sdp->ptr) != NULL)	/* Already defined */
	    continue;
	if (sdp->envvar != NULL && (evp = getenv(sdp->envvar)) != NULL)
	{
	    *((char **) sdp->ptr) = strinit(evp);	/* Set from env. */
	    continue;
	}
	if (sdp->defstr == NULL)	/* MUST be user-defined: fatal */
	    fatal(sdp->errstr);
	if (sdp->errstr != NULL)	/* Warn user we're using default */
	    warning(sdp->errstr);
	*((char **) sdp->ptr) = strinit(sdp->defstr);
    }
}


/*
 * check_intdefs()
 *
 * Check certain integer variables: if they are not within the given range,
 * set them to their default values.
 */
static void check_intdefs(void)
{
    unsigned i;
    struct intdefault *idp;

    for (i = 0, idp = intdefs; i < NELEM(intdefs); i++, idp++)
    {
	if (*((int *) idp->ptr) < idp->minval ||
	    *((int *) idp->ptr) > idp->maxval)
	    *((int *) idp->ptr) = idp->defval;
    }
}


/*
 * rcerror()
 *
 * Report on RC file error <err>, which occurred with entry label <label>;
 * this function may never return, if <err> < 0 (which is a fatal error).
 */
static void rcerror(int err, const char *label)
{
    char errbuf[LINELEN];

    AUSTRLCPY4(errbuf, (err < 0) ? "Error: " : "Warning: ",
	       "invalid \"", label, "\" entry");
    if (err < 0)
	fatal(errbuf);
    else
	warning(errbuf);
}


/*
 * fatal()
 *
 * Display an error message and exit (with cleanup)
 */
static void fatal(const char *x)
{
    textattr(col_hilite);
    cputs(x);
    cputs("\r\n");
    textattr(col_system);
    exit(-1);
}


/*
 * warning()
 *
 * Display a warning message
 */
static void warning(const char *x)
{
    textattr(col_hilite);
    cputs(x);
    cputs("\r\n");
    textattr(col_system);
    sleep(2);
}


/*
 * noslash()
 *
 * Replace any terminating / or \\ with null
 */
static void noslash(char *s)
{
    char *cp;
    size_t len = strlen(s);

    if (len > 0)
    {
	cp = s + len - 1;
	if (*cp == '\\' || *cp == '/')
	    *cp = '\0';
    }
}


/*
 * parsestr()
 *
 * Parse string <s> into a static buffer and return a pointer to that buffer,
 * unless the string is all whitespace, or it contains an invalid escaped
 * decimal (i.e. longer than 3 digits but < 100, which seems a pretty silly
 * test to me but is backwards-compatible), in which case we return NULL.
 * Only process backslash-escapes if <escapes> is true (to allow paths and
 * command strings to contain backslashes).
 */
static char *parsestr(const char *s, BOOL escapes)
{
    static char buf[LINELEN + 1];
    char endchar, *t;

    s = skip_lwsp(s);
    if (*s == '\0')
	return NULL;
    if (*s == '"')
    {
	endchar = '"';
	s++;
    }
    else
	endchar = '\0';	/* want the whole string */
    t = buf;
    while ((*s) && (*s != endchar) && (t < buf + sizeof(buf) - 1))
    {
	if (escapes && *s == '\\')
	{
	    s++;
	    switch (*s)
	    {
	    case 'r':		/* Carriage return */
		*t = '\r';
		break;
	    case 'n':		/* Newline */
		*t = '\n';
		break;
	    case 't':		/* Tab */
		*t = '\t';
		break;
	    case 's':		/* Space */
		*t = ' ';
		break;
	    default:		/* Quoted character */
		*t = *s;
		break;
	    }
	    /*
	     * allow escaped decimals, i.e. \1 = Ctrl-A
	     * useful for MMDF-like mailers.
	     */
	    if (isdigit((*s)))
	    {
		unsigned char i = 0, j = 0;

		do
		{
		    if (++j > 3) 	/* More than 3 digits: stop */
			break;
		    i *= 10;
		    i += *s++ - 0x30;
		}
		while (isdigit((*s)));
		if (j > 3 && i < 100)	/* > 3 digits, but < 100? */
		    return NULL;	/* escaped char too big */
		*t = i;
		s--;
	    }
	}
	else
	    *t = *s;
	t++;
	s++;
    }
    *t = '\0';
    return buf;
}


/*
 * free_globals()
 *
 * Free up all global variable dynamic storage allocated here
 */
void free_globals(void)
{
    free(quotestr);
    if (useraddr != username)	/* We have a separate <useraddr> */
	free(useraddr);
    free(username);
    free(replyto);
    free(uucpreplyto);
    free(maildir);
    free(editor);
    free(pager);
    free(gifview);
    free(jpegview);
    free(mpegview);
    free(audioview);
    free(miscview);
    free(multimedia);
    free_strarray(mboxext_store, NELEM(mboxext_store));
    free(uucp_command);
    free(sequencename);
    free(mailqdir);
    free(printdevice);
    free_strlist(weedlist1);
    free_strlist(weedlist2);
    free_strarray(opthdr, NELEM(opthdr));
    free(mailhost);
    free(cwd);
    free(homedir);
    free(firstname);
    free(mailqueue);
    free(tfname);
    free(mfname);
    free(fullname);
    free_strarray(hostname_store, NELEM(hostname_store));
    free_strarray(startofhdr_store, NELEM(startofhdr_store));
    free(hdr_fullname);
}


/*
 * free_strarray()
 *
 * Free the array of char * pointers <array>, which has <nelem> elements
 */
static void free_strarray(char **array, unsigned nelem)
{
    unsigned i;

    for (i = 0; i < nelem; i++)
	free(array[i]);
}


/*
 * free_strlist()
 *
 * Free the dynamic array of char * pointers <list>; free all char * pointers
 * until we encounter a NULL pointer, which ends the list, then free <list>
 * itself.
 */
static void free_strlist(char **list)
{
    char **ptr;

    if (list != NULL)
    {
	for (ptr = list; *ptr != NULL; ptr++)
	    free(*ptr);
	free(list);
    }
}
