/*
  This source is part of PCOak, an electronic mailer for DOS based on PCElm.

  PCElm is Copyright (c) 1988-1993 Martin Freiss and Wolfgang Siebeck
           Copyright (c) 1992-1999 Demon Internet
  PCOak is Copyright (c) 2000-2003 Simon Turner, Pete Disdale and dispc members

  Thanks to an agreement between the original PCElm authors and Demon Internet
  made in late 1999:

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License, version 1, as
	published by the Free Software Foundation.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	See the file COPYING, which contains a copy of the GNU General
	Public License.
*/

#include <stdio.h>
#include <dos.h>
#include <conio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <time.h>
#include <io.h>
#include "pcoak.h"
#include "header.h"
#include "ustring.h"
#include "chars.h"
#include "macros.h"
#include "display.h"

/*
 * Global variables local to this module
 */
static const char update_err[] = "Error updating original mail file";
static long mailfilesize = 0L;	/* Size of mailbox when last read from disk */
static char mfilename[FNAMELEN];  /* Filename for <mfile>; always in <cwd> */

/*
 * External global variables used in this module
 */
extern const char file_err[];		/* "File error" */
extern const char enc_7bit[];		/* "7bit" */
extern const char enc_8bit[];		/* "8bit" */
extern const char enc_binary[];		/* "binary" */
extern const char enc_qp[];		/* "quoted-printable" */
extern const char enc_base64[];		/* "base64" */
extern char *maildir;		/* defined mail directory */
extern char notefile[];		/* Full path name of the mailbox text file */
extern BOOL sortdescend;	/* TRUE for descending, FALSE for ascending */
extern SORTBYTYPE sortby;	/* Sort order (SORT_INCOMING / SORT_DATE) */
extern BOOL index_onscreen;	/* Is the index screen displayed? */

/*
 * Functions defined and used in this module
 */
static void sort_mbox(BOOL userorder);
static int sort_mail(const void *p1, const void *p2);
static int sort_msgno(const void *p1, const void *p2);
int copymail(BOOL isbiffed, BOOL dosort);
void free_mbox(void);
static PHTL new_mbox(PHTL last);
PHTL last_mbox(void);
static void proc_folded_header(PHTL cmsg, HEADERTYPE type,
			       const char *hdrline);
int check_for_new_mail(void);
CTTYPE get_content_type(const char *line);
CETYPE get_content_encoding(const char *line);
static int open_mfile(void);
static int close_mfile(BOOL unlink_it);

/*
 * Functions defined elsewhere and used in this module
 */
FILE *opentemp(char *name, const char *mode);
int mlock(const char *dir, const char *id);
void rmlock(const char *dir, const char *id);
char rip(char *s);
char eoltest(const char *s);
BOOL y_n(const char *msg);
BOOL abort_retry(const char *msg);
void backslash(char *s);
HEADERTYPE htype(const char *p);
BOOL htype_match(const char *p, HEADERTYPE type);
void errordsp(const char *txt);
void fopen_error(const char *filename, const char *descrip);
char *stristr(const char *s1, const char *s2);
void errorbox(const char *title, ...);
time_t parse_date(const char *datestr, short *offset);
void verify_status_line(char *buf, size_t bufsize);
int write_status_line(FILE *fp, BOOL quote);
const char *hdr_body(const char *field, HEADERTYPE type);


/*
 * sort_mbox()
 *
 * Sort the <mbox> list into (a) the user's desired order, or (b) original
 * mailbox order, depending on whether <userorder> is true or not.
 */
static void sort_mbox(BOOL userorder)
{
    PHTL *mbarray;
    PHTL cmsg;
    int i;

    if (mbox == NULL || nmsgs <= 0)
    {
	errorbox("Sort error", "Can't sort an empty mailbox!", NULL);
	return;
    }

    if ((mbarray = malloc((size_t) nmsgs * sizeof(*mbarray))) != NULL)
    {
	for (i = 0, cmsg = mbox; i < nmsgs && cmsg != NULL;
	     i++, cmsg = cmsg->next)
	    mbarray[i] = cmsg;
	if (i != nmsgs || cmsg != NULL)		/* Eh? */
	{
	    errorbox("Sort error", "mbox list not expected size!", NULL);
	    free(mbarray);
	    return;
	}
	qsort(mbarray, (size_t) nmsgs, sizeof(PHTL),
	      (userorder) ? sort_mail : sort_msgno);
	mbox = mbarray[0];
	mbox->prev = NULL;
	for (i = 1, cmsg = mbox; i < nmsgs; i++, cmsg = cmsg->next)
	{
	    cmsg->listno = i;
	    cmsg->next = mbarray[i];
	    mbarray[i]->prev = cmsg;
	}
	cmsg->listno = nmsgs;
	cmsg->next = NULL;
	free(mbarray);
    }
    else
    {
	/* TODO: do a simpler sort here, e.g. Shell, Heapsort */
	errorbox("Sort error", "Can't sort: no memory for array!", NULL);
	return;
    }
}


/*
 * sort_mail()
 *
 * A qsort() comparison function; <p1> and <p2> (pointers to elements in an
 * array of pointers to HTL structures) are to be sorted in the user's chosen
 * order, as defined by <sortby> and <sortdescend>.  Return < 0 if <p1> comes
 * before <p2>, > 0 if if comes after <p2>, or 0 if they have equal ranking.
 */
static int sort_mail(const void *p1, const void *p2)
{
    PCHTL s1, s2;
    int n = 0;

    /* <p1> and <p2> are pointer to PHTL (i.e. pointer to a pointer) */
    s1 = *((PCHTL *) p1);
    s2 = *((PCHTL *) p2);
    if (sortby == SORT_DATE)
	n = (s1->utc > s2->utc) ? 1 : (s1->utc < s2->utc) ? -1 : 0;
    if (n == 0)
	n = s1->msgno - s2->msgno;
    return (sortdescend) ? -n : n;
}


/*
 * sort_msgno()
 *
 * A qsort() comparison function; <p1> and <p2> (pointers to elements in an
 * array of pointers to HTL structures) are to be sorted in order of msgno
 * (i.e. sorted into original mailbox file order).  Return < 0 if <p1> comes
 * before <p2>, > 0 if if comes after <p2>, or 0 if they have equal ranking.
 */
static int sort_msgno(const void *p1, const void *p2)
{
    /* <p1> and <p2> are pointer to PHTL (i.e. pointer to a pointer) */
    return (*((PCHTL *) p1))->msgno - (*((PCHTL *) p2))->msgno;
}


/*
 * copymail()
 *
 * Copy the mail file (global <notename>) to a temporary file, build mbox
 * list, fill in the blanks.  If mailbox not found return -1 instead of
 * exiting!  (That's considerate of you. -ST)
 *
 * <isbiffed> indicates that we've already read the mailbox, but that it has
 * changed since we took our copy (we assume this is because new mail arrived
 * and was added to the end of the old mailbox while our back was turned).
 * Knowing the length of the mailbox when we read it last time (in global
 * variable <mailfilesize>), we read these new entries from the new (longer)
 * mailbox file and add them to both our internal <mbox> list of messages, and
 * our "current but based on the old mailbox" temporary copy of the file,
 * which lives in global (and currently open) stream <mfile>.  This bit used
 * to screw up the .size member of the last "old" <mbox> entry if the <mbox>
 * entries were sorted such that the last entry wasn't the last one in the
 * file, but we're more careful now.  (Fingers crossed. 8-)
 *
 * There used to be rather a lot of static variables here, almost all of which
 * appear to be unnecessary: they were either re-set upon each entry to the
 * function (or should have been!), or could easily be deduced from other
 * information (and, indeed, NEEDED to be thus deduced, since it's imperative
 * that they are in sync with these things).  Specifically, they were:
 *
 * inheader	Reset to TRUE every time we entered anyway.
 *
 * have_soh	Initially set to 0, and *should* have been 0 when the function
 *		exited last time (seeing any known header other than SOH, or
 *		reaching the end of the header, sets it to 0); *MUST* be 0 if
 *		we're reading a new file (non-biffed).
 *
 * OldMailFileSize	Effectively a local mirror of the global variable
 *			mailfilesize (which is only ever set here, and is
 *			always set to the same value as OldMailFileSize); we
 *			now use mailfilesize instead.
 *
 * tpos		The last end-of-line position in the <mfile> stream; correctly
 *		set to 0 if this is a new file (non-biffed), but needs to match
 *		the actual position in <mfile>, i.e. the end of the file: this
 *		must match "ftell(mfile)" after we've seeked to the end, and is
 *		now set to that to avoid any possible discrepancies.
 *
 * There's an argument for having <inheader> and <have_soh> static for the
 * "biffed" case, so that if we see an SOH line at the end of the mailbox when
 * we initially read it, we carry that information forward when we re-read it
 * later, so that the (old) SOH can apply to the (new) remainder of the
 * header; but it doesn't seem worth bothering.  If we were to do this, they
 * would both need to be set for non-biffed reading from scratch, which they
 * weren't in PCElm; so it's less broken than it used to be anyway.
 *
 * We used to malloc() a temporary string store here.  The only reason I can
 * think of for doing this would be if we were in danger of running out of
 * stack space; but it's only ever called from near the top level of the call
 * stack, so I can't see this as a likely problem (if it is, we're in *real*
 * trouble later on with other functions).  copymail() is only called from (a)
 * main(), (b) closenotes(), and (c) change_mail_file().  closenotes() is only
 * called from main() and change_mail_file(); and change_mail_file() is only
 * called from main().  I see no likelihood of running out of stack when we're
 * at most two levels down from main(), neither of which has particular large
 * stack requirements.  Therefore we now use the stack for <tstring>.
 */
int copymail(BOOL isbiffed, BOOL dosort)
{
    long cpos;			/* <mfile> position at start of current line */
    long tpos = 0L;		/* <mfile> position at end of new line */
    char tstring[LLINELEN];
    char *p, t[132], *buf;
    PHTL cmsg;			/* pointer to current message structure */
    BOOL inheader = FALSE;	/* Not in header until we've seen an SOH */
    BOOL have_soh = FALSE;	/* Haven't seen an SOH yet */
    FILE *ifile;
    struct stat sb;
    HEADERTYPE hdrtype;		/* Type of header line just found */
    char fhdrbuf[LLINELEN];	/* Buffer for unfolding header lines into */
    HEADERTYPE fhdrtype;	/* Type of header being built in <fhdrbuf> */
    char last_eol;
    long eol_bytes = (used_mailsystem == SYS_HAM) ? 2L : 1L;
    BOOL writerr = FALSE;
    BOOL memerr = FALSE;

    (void) mlock(maildir, notename);
    if ((ifile = fopen(notefile, "r")) == NULL)
    {
	AUSTRLCPY3(t, "Can't open mailfile ", notefile, ". Create it (Y/N)?");
	backslash(t);
	if (y_n(t))
	{
	    cputs("Yes");
	    /*
	     * Create empty notefile in safe mode, and return NO MESSAGES AVL
	     */
	    if ((ifile = fopen(notefile, "a")) != NULL)
	    {
		(void) fclose(ifile);			/* OK: unused */
		index_onscreen = FALSE;
		rmlock(maildir, notename);
		free_mbox();		/* Free any existing <mbox> */
		return 0;
	    }
	    /* If we get to here, we couldn't open the file: treat as "No" */
	    errordsp("Can't create mailfile!");
	    sleep(3);
	}
	rmlock(maildir, notename);
	return -1;
    }

    /*
     * Allocate a 16K buffer for <ifile> to speed things up
     */
#if 1
    buf = malloc(16535);
#else
    buf = NULL;
#endif
    if (buf)
	setvbuf(ifile, buf, _IOFBF, 16535);

    /*
     * Make sure we've got an open mail file, and set up the <mbox> list of
     * "htextline" structures, which describe the messages in the file
     */
    if (!isbiffed)
    {
	/*
	 * We're taking a new copy of the mailbox: this, at least, is
	 * straightforward.  The old <mfile> is assumed to have been closed,
	 * and <mfilename> has probably been unlinked (but if not, we'll get a
	 * new temp file name this time).  So, first we open a new <mfile>
	 * file for our mailbox copy; then we free any existing <mbox> list
	 * and set <nmsgs> to 0.
	 */
	if (open_mfile() != 0)
	{
	    errorbox("Fatal error", "Can't run without mailbox copy!",
		     "Aborting PCOak", NULL);
	    (void) fclose(ifile);			/* OK: read only */
	    rmlock(maildir, notename);
	    exit(1);		/* FATAL: couldn't open <mfile> */
	}
	mbchanged = FALSE;
	free_mbox();		/* Ensure any existing <mbox> is freed first */
	tpos = 0L;		/* We're at the start of <mfile> */
	cmsg = NULL;		/* We're not in the middle of a message */
    }
    else
    {
	/*
	 * The original mailbox has got longer -- we assume that new messages
	 * have been appended, so that the first <mailfilesize> bytes will be
	 * the same.  <mailfilesize> was set when we read this file for the
	 * first time, i.e. the last call to copymail(FALSE,?); since we
	 * always close one file before opening another, we can be certain
	 * that the last call to copymail() was to read *this* file.  What we
	 * need to do now is read ONLY the NEW messages from the mailbox file,
	 * appending them to our current list of messages in <mbox>; to
	 * achieve this, we seek to the "old" end of the mailbox file <ifile>
	 * (just before the new messages), leaving <mbox> as it is, and also
	 * jump to the end of the "working" file <mfile>, ready to copy the
	 * new stuff from <ifile> to <mfile> while adding it on to the end of
	 * <mbox>.
	 */
	fseek(ifile, mailfilesize, SEEK_SET);
	fseek(mfile, 0L, SEEK_END);  /* move to end of old "copy" mail file */
	tpos = ftell(mfile);	/* current position in <mfile> */
	cmsg = last_mbox();	/* We're looking at the last "old" message */
    }

    if (!stat(notefile, &sb))		/* note the (new) size of the */
	mailfilesize = sb.st_size;	/*   notefile in <mailfilesize> */

    gotoxy(1, display.screenheight);
    clreol();

    fhdrtype = HT_UNKNOWN;	/* Not waiting for a folder header (yet) */
    *fhdrbuf = '\0';		/* Make sure the buffer starts off empty */
    last_eol = '\0';		/* Ignore blank lines at start of file */

    cpos = tpos;		/* Ensure current position is initialised */

    /*
     * Read the whole mailbox, storing information about each message in
     * elements of the <mbox> linked list.
     */
    while (fgets(tstring, sizeof(tstring), ifile) != NULL)
    {
	cpos = tpos;	/* <cpos> = <mfile> position at start of this line */
	
	/*
	 * If we're not in the middle of a message header, look to see if this
	 * is the start of a new message; if we *are* in the middle of a
	 * message header, look to see whether this is the blank line that
	 * marks the end of the header
	 */
	if (!inheader)		/* Not currently in header: is this SOH? */
	    inheader = htype_match(tstring, HT_SOH);	/* Yes! */
	else if (last_eol && ISBLANKLINE(tstring))
	{
	    /*
	     * Found the blank line which ends the header.  Now that we've
	     * seen the entire header, we can do some sanity checking:
	     * specifically, if we've seen some MIME headers but not a
	     * MIME-Version, it's not a "proper" MIME message so we turn off
	     * the MIME flag.
	     *
	     * If we're reading a file which starts with blank lines, or
	     * doesn't start with SOH (so we're ignoring the text), we may get
	     * an "end of header" blank line when we're not actually dealing
	     * with a message anyway; in this case, <cmsg> will be NULL, so we
	     * just ignore it.
	     */
	    if (cmsg == NULL)		/* Should never happen */
	    {
#if 1
		errorbox("Mailbox reading error",
			 "Found end of header, but current mbox is NULL!?",
			 "", "Please report this error!", NULL);
#endif
	    }
	    else
	    {
		if (fhdrtype != HT_UNKNOWN)  /* waiting for a (folded?) line */
		{
		    proc_folded_header(cmsg, fhdrtype, fhdrbuf);
		    fhdrtype = HT_UNKNOWN;
		}
		if ((cmsg->status & S_MIME) &&
		    !(cmsg->headers & H_MIME_VERSION))
		    cmsg->status &= ~S_MIME;	/* turn off S_MIME flag */
	    }
	    inheader = FALSE;
	    have_soh = FALSE;
	    cmsg = NULL;
	}

	last_eol = rip(tstring);	/* remove trailing LF/CRLF */

	/*
	 * Write this line to the working file <mfile>; increase <tpos> up to
	 * take account of our new position in the file.
	 *
	 * Why don't we use ftell()?  I'd guess because calling ftell()
	 * actually does quite a lot of seeking to and fro in the file, and
	 * therefore gives us a serious speed hit which we can avoid by
	 * calculating the change (allowing for the fact that the actual file
	 * will have CRLF at the end of every line in SYS_HAM mode, even
	 * though the files are in text mode and we only appear to have a
	 * plain LF -- in SYS_UUCP mode, the files are in binary mode and will
	 * therefore not have a CR in the first place).  -ST
	 */
	(void) fputs(tstring, mfile);
	tpos += (long) strlen(tstring);		/* Add the string */
	if (last_eol)				/* Line ended with CRLF */
	{
	    (void) fputs("\n", mfile);
	    tpos += eol_bytes;			/* + [CR]LF */
	}
	if (ferror(mfile))	/* Aargh! */
	{
	    writerr = TRUE;
	    break;	/* out of while (fgets) */
	}

	/*
	 * Further processing is only required if we're in the message header
	 */
	if (!inheader)
	    continue;	/* with the while (fgets) */

	/*
	 * If this line starts with whitespace, it's a continuation line;
	 * we're only interested in these if we're actually trying to build
	 * a folded header field in <fhdrbuf>, in which case <fhdrtype> will
	 * be something other than HT_UNKNOWN; and if it isn't a continuation
	 * line but <fhdrbuf> 
	 */
	if (islwsp(tstring[0]))		/* it's a continuation line */
	{
	    if (fhdrtype != HT_UNKNOWN)		/* unfolding into <fhdrbuf> */
	    {
		kill_trail_lwsp(fhdrbuf);
		AUSTRLCAT2(fhdrbuf, " ", skip_lwsp(tstring));
	    }
	    continue;	/* with the while (fgets) */
	}
	else if (fhdrtype != HT_UNKNOWN)
	{
	    /* The header line we were reading is now complete */
	    proc_folded_header(cmsg, fhdrtype, fhdrbuf);
	    fhdrtype = HT_UNKNOWN;
	}

	if ((hdrtype = htype(tstring)) == HT_SOH)	/* Start-of-header */
	{
	    if (!have_soh)		/* This is a real SOH: start new msg */
	    {
		if ((cmsg = new_mbox(cmsg)) == NULL)	/* Out of memory */
		{
		    memerr = TRUE;
		    fseek(mfile, cpos, SEEK_SET);	/* Back to msg start */
		    break;	/* out of while (fgets) */
		}
		cmsg->msgno = ++nmsgs;
		cmsg->position = cpos;
		/* cmsg->content_type = CT_NONE; */	/* Already 0 */
		/* cmsg->content_encoding = CE_NONE; */	/* Already 0 */
		if (cmsg->prev != NULL)	/* note size of previous msg */
		{
		    if (isbiffed)	/* leave last "old" entry alone */
			isbiffed = FALSE;	/* (OK from next one on) */
		    else		/* set .size for last message */
			cmsg->prev->size = cpos - cmsg->prev->position;
		}
		gotoxy(1, display.screenheight);
		cprintf("\rReading mail in %s: %d", notefile, nmsgs);
		have_soh = TRUE;
	    }
	}
	else if (cmsg != NULL)		/* In a message; process header line */
	{
	    switch (hdrtype)
	    {
	    case HT_SUBJECT:
		have_soh = FALSE;
		p = skip_lwsp(tstring + 8);
		if (*p)
		{
		    ustrncpy(cmsg->subject, p, 34);
		    cmsg->headers |= H_SUBJECT;
		}
		break;

	    case HT_FROM:
		have_soh = FALSE;
		p = skip_lwsp(tstring + 5);
		if (*p && cmsg->from == NULL)
		{
		    cmsg->from = strdup(p);
		    if (cmsg->from == NULL)
		    {
			memerr = TRUE;
			fseek(mfile, cmsg->position, SEEK_SET);	/* msg start */
			/* Clear incomplete current message, adjust count */
			if (cmsg->prev != NULL)	   /* stop after prev msg */
			    cmsg->prev->next = NULL;
			nmsgs--;
			if (mbox == cmsg)	   /* Aargh! Empty list */
			    mbox = NULL;
			free(cmsg);
			cmsg = NULL;
			break;	/* out of the switch (hdrtype) */
		    }
		    cmsg->headers |= H_FROM;  /* N.B. ->from is non-NULL */
		}
		break;

	    case HT_CONTENT_TYPE:
		have_soh = FALSE;
		cmsg->headers |= H_CONTENT_TYPE;
		/* wait for fully unfolded line before processing it */
		fhdrtype = HT_CONTENT_TYPE;
		AUSTRCPY(fhdrbuf, tstring);
		break;

	    case HT_CONTENT_XFER_ENCODING:
		have_soh = FALSE;
		cmsg->headers |= H_CONTENT_XFER_ENCODING;
		/* wait for fully unfolded line before processing it */
		fhdrtype = HT_CONTENT_XFER_ENCODING;
		AUSTRCPY(fhdrbuf, tstring);
		break;

	    case HT_MIME_VERSION:
		have_soh = FALSE;
		cmsg->headers |= H_MIME_VERSION;
		break;

	    case HT_REPLYTO:
		have_soh = FALSE;
		p = skip_lwsp(tstring + 9);
		if (*p)
		    cmsg->headers |= H_REPLYTO;
		break;

	    case HT_STATUS:
		have_soh = FALSE;
		if (strchr(hdr_body(tstring, HT_STATUS), 'R') != NULL)
		    cmsg->status |= S_READ;
		break;

	    case HT_DATE:
		have_soh = FALSE;
		p = skip_lwsp(tstring + 5);
		cmsg->utc = (long) parse_date(p, &cmsg->offmins);
		cmsg->headers |= H_DATE;
		break;

	    default:
		/* should we set have_soh = FALSE here as well?  ST-FIXME? */
		/* (On balance, probably not, but I'm open to persuasion.) */
		break;
	    }	/* switch (hdrtype) */			/*lint !e788 */ /*OK*/
	    if (memerr)		/* Out of memory during switch () */
		break;	/* out of the while (fgets) */
	}
#if 1
	else
	{
	    errorbox("NULL cmsg", "cmsg is NULL; not started yet!", NULL);
	}
#endif
    }		/* while (fgets) */

    if (memerr)				/* Out of memory */
    {
	errorbox("Out of memory",
		 "Mailbox read failed catastrophically!",
		 "Please quit (suggest 'X') and restart", NULL);
    }

    /*
     * If we've got any messages stored, we need to note the size of the final
     * message, and the final file position (stored as the starting position
     * of the -- otherwise unused? -- last item in the array)
     */
    if (nmsgs > 0)
    {
	cmsg = last_mbox();
	cmsg->size = ftell(mfile) - cmsg->position;
    }

    (void) fclose(ifile);				/* OK: read only */
    if (fflush(mfile) != 0)		/* Write error flushing file to disk */
	writerr = TRUE;

    rmlock(maildir, notename);
    free(buf);

    if (writerr)
    {
	errorbox("File error", mfilename,
		 "Write error while copying mailbox",
		 "Please quit (suggest 'X') and restart", NULL);
	return nmsgs;
    }

    if (nmsgs > 1 && dosort)		/* Sort the mbox list */
    {
	/* now sort */
	cprintf("\rSorting Mail... ");
	clreol();
	sort_mbox(TRUE);
    }

    return nmsgs;
}


/*
 * free_mbox()
 *
 * Ensure that any allocated mbox stuff is free
 */
void free_mbox(void)
{
    if (mbox)		/* Free existing mbox[] array & contents */
    {
	PHTL m, nm;

	for (m = mbox; m != NULL; m = nm)
	{
	    nm = m->next;
	    free(m->from);
	    free(m);
	}
	mbox = NULL;
    }
    nmsgs = 0;
}


/*
 * new_mbox()
 *
 * Create a new element and add it to the end of the global <mbox> linked list
 * of HTL structures; return a pointer to this new structure.  <last>, if
 * non-NULL, is (probably) the old "last" structure in the list; it should at
 * least be a good starting point for finding the old "last" structure.
 */
static PHTL new_mbox(PHTL last)
{
    PHTL nm;

    /* Use calloc() to ensure all struct members are 0 or NULL pointers */
    if ((nm = calloc(1, sizeof(HTL))) == NULL)
	return NULL;
    if (mbox == NULL)		/* First and only entry in list */
    {
	nm->listno = 1;		/* First entry in the list */
	mbox = nm;		/* Both .prev and .next are NULL; correct */
    }
    else			/* New entry on end of list */
    {
	if (last == NULL)		/* Start at beginning of list */
	    last = mbox;
	while (last->next != NULL)	/* Run to end of list */
	    last = last->next;
	nm->prev = last;
	nm->listno = last->listno + 1;
	last->next = nm;
    }
    return nm;
}


/*
 * last_mbox()
 *
 * Return a pointer to the last entry in the <mbox> linked list
 */
PHTL last_mbox(void)
{
    PHTL m;

    if (mbox == NULL)
	return NULL;
    for (m = mbox; m->next != NULL; m = m->next)
	;
    return m;
}


/*
 * proc_folded_header()
 *
 * We've reached the end of a (possibly folded) header line of type <type>;
 * the unfolded line is in <hdrline>.  Do anything that needs to be done to
 * the message <cmsg>.
 */
static void proc_folded_header(PHTL cmsg, HEADERTYPE type, const char *hdrline)
{
    switch (type)
    {
    case HT_CONTENT_TYPE:
	cmsg->content_type = get_content_type(hdrline);
	if (cmsg->content_type != CT_TEXT_PLNASCII &&
	    cmsg->content_type != CT_NONE)	/* an "interesting" type */
	    cmsg->status |= S_MIME;
	break;

    case HT_CONTENT_XFER_ENCODING:
	cmsg->content_encoding = get_content_encoding(hdrline);
	break;

    default:
	break;
    }							/*lint !e788 */ /*OK*/
}


/*
 * check_for_new_mail()
 *
 * Check for new mail arriving in the "current" mailbox (global <notefile>).
 * If the file is the same size as when we took our working copy of it with
 * copymail(), return 0; otherwise, display an explanatory message in the
 * bottom line and return non-zero: if we can't stat() the file to check it,
 * return -1; if its size has increased, return 1.
 */
int check_for_new_mail(void)
{
    struct stat mstat;
    char errtxt[80];
    int ret = 1;

    if (stat(notefile, &mstat))
    {
	AUSTRLCPY2(errtxt, "Unable to stat ", notefile);
	ret = -1;
    }
    else if (mstat.st_size > mailfilesize)
	AUSTRCPY(errtxt, "New mail has arrived");
    else
	return 0;
    errordsp(errtxt);
    return ret;
}


/*
 * get_content_type()
 *
 * <line> is a MIME Content-Type header line; scan it and return the type.
 *
 * This needs redoing, to try and be more RFC-compliant.  Full compliance with
 * at least the minimum conformance standards in RFC 2049 would be nice.
 * (Although I'm not too keen on RFC 2049's insistence that text in an unknown
 * charset should be treated as "application/octet-stream"; there are too many
 * ASCII-like, but obscure or non-standard, charsets in common use for this to
 * be practical).  ST-FIXME
 *
 * As far as I can work out, this was originally written according to RFC 1341
 * (1992), which explains some of the eccentricities (but not all, by any
 * means; the PCElm implementation fell woefully short of RFC 1341's
 * standards, and introduced a whole load of non-standard/broken stuff to
 * boot).  -ST
 *
 * An unknown Content-Type should be treated as "application/octet-stream",
 * according to RFC 2049 (and RFC 1341 before it).  In order to comply with
 * this, we must be able to recognise at least the main MIME media types and
 * deal with them accordingly, even if we make no real effort to deal with the
 * various subtypes -- if not, we would end up treating "message/rfc822" as an
 * octet stream to be written to a file, which is not exactly ideal!  To avoid
 * confusion about what is actually *seen*, we return CT_UNKNOWN if we don't
 * recognise the Content-Type.
 *
 * If there is no Content-Type seen at all, we return CT_NONE to indicate
 * this; such messages should be treated as "text/plain; charset=us-ascii"
 * (RFC 2045, and RFC 1341 before it), unless they are in a multipart/digest
 * entity, in which case they should be "message/rfc822" (RFC 2049).
 *
 * I've pulled out the (broken and invalid) support for the "richtext" TYPE
 * and CHARSET; although "text/richtext" was valid in RFC 1341, "richtext" was
 * only ever valid as a SUBTYPE of "text".  For reasons that no doubt even
 * Grahame would find hard to explain, PCElm seemed to regard it as a type, a
 * subtype *and* a charset(!) which is, of course, complete rubbish.  The
 * richtext subtype is now obsolete, having been superseded by "text/enriched"
 * as defined in RFC 1896, but it probably doesn't do any harm to leave
 * recognition of it in here.  (We can actually do something special with it
 * in a later release; might be useful for ripping out HTML tags, too.  Hmmm.)
 */
CTTYPE get_content_type(const char *line)
{
    const char *ptr;
    CTTYPE ret = CT_NONE;	/* Treat as CT_TEXT_PLNASCII / CT_MSG_RFC822 */

    ptr = hdr_body(line, HT_CONTENT_TYPE);

    if (strnicmp(ptr, "text", 4) == 0)			/* "text" type */
    {
	ptr += 4;
	if (strnicmp(ptr, "/plain", 6) == 0)
	{
	    ret = CT_TEXT_PLNASCII;
	    if ((ptr = stristr(ptr, "charset")) != NULL)
	    {
		ptr = skip_lwsp(ptr + 7);	/* skip "charset" */
		if (*ptr == '=')		/* got required '=' */
		{
		    ptr = skip_lwsp(ptr + 1);	/* skip '=' + further LWSP */
		    if (*ptr == '\"')		/* quoted; skip the quote */
			ptr++;
		    if (strnicmp(ptr, "iso-8859-1", 10) == 0)
			ret = CT_TEXT_PLN88591;
		    /* If not iso-8859-1, leave as us-ascii for now */
		    /* Needs better handling of charsets: ST-FIXME */
		}
	    }
	}
	else if (strnicmp(ptr, "/richtext", 9) == 0 ||
		 strnicmp(ptr, "/enriched", 9) == 0)
	    ret = CT_TEXT_RICH;
	else if (strnicmp(ptr, "/html", 5) == 0)
	    ret = CT_TEXT_HTML;
	else
	    ret = CT_TEXT_MISC;
    }
    else if (strnicmp(ptr, "image", 5) == 0)		/* "image" type */
    {
	ptr += 5;
	if (strnicmp(ptr, "/gif", 4) == 0)
	    ret = CT_IMAGE_GIF;
	else if (strnicmp(ptr, "/jpeg", 5) == 0)
	    ret = CT_IMAGE_JPEG;
	else
	    ret = CT_IMAGE_MISC;
    }
    else if (strnicmp(ptr, "audio", 5) == 0)		/* "audio" type */
    {
	ptr += 5;
	if (strnicmp(ptr, "/au", 3) == 0)
	    ret = CT_AUDIO_AU;
	else
	    ret = CT_AUDIO_MISC;
    }
    else if (strnicmp(ptr, "video", 5) == 0)		/* "video" type */
    {
	ptr += 5;
	if (strnicmp(ptr, "/mpeg", 5) == 0)
	    ret = CT_VIDEO_MPEG;
	else
	    ret = CT_VIDEO_MISC;
    }
    else if (strnicmp(ptr, "application", 11) == 0)	/* "application" type*/
    {
	ptr += 11;
	if (strnicmp(ptr, "/octet-stream", 13) == 0)
	    ret = CT_APP_OCTET;
	else
	    ret = CT_APP_MISC;
    }
    else if (strnicmp(ptr, "multipart", 9) == 0)	/* "multipart" type */
    {
	ptr += 9;
	if (strnicmp(ptr, "/mixed", 6) == 0)
	    ret = CT_MULTI_MIXED;
	else if (strnicmp(ptr, "/alternative", 12) == 0)
	    ret = CT_MULTI_ALTERN;
	else if (strnicmp(ptr, "/digest", 7) == 0)
	    ret = CT_MULTI_DIGEST;
	else
	    ret = CT_MULTI_MISC;
    }
    else if (strnicmp(ptr, "message", 7) == 0)		/* "message" type */
    {
	ptr += 7;
	if (strnicmp(ptr, "/rfc822", 7) == 0 ||
	    strnicmp(ptr, "/rfc2822", 8) == 0)
	    ret = CT_MSG_RFC822;
	else if (strnicmp(ptr, "/delivery-status", 16) == 0)
	    ret = CT_MSG_DELIVSTAT;
	else
	    ret = CT_MSG_MISC;
    }
    else      /* unknown Content-Type: treat as CT_APP_OCTET (RFC 2049) */
	ret = CT_UNKNOWN;

    return ret;
}


/*
 * get_content_encoding()
 *
 * <line> is a MIME Content-Transfer-Encoding header line; scan it and return
 * the encoding.
 */
CETYPE get_content_encoding(const char *line)
{
    const char *ptr;
    CETYPE ret = CE_NONE;	/* Should be treated as CE_7BIT (RFC 2045) */

    ptr = hdr_body(line, HT_CONTENT_XFER_ENCODING);
    if (strnicmp(ptr, enc_7bit, 4) == 0)
	ret = CE_7BIT;
    else if (strnicmp(ptr, enc_8bit, 4) == 0)
	ret = CE_8BIT;
    else if (strnicmp(ptr, enc_binary, 6) == 0)
	ret = CE_BINARY;
    else if (strnicmp(ptr, enc_qp, 16) == 0)
	ret = CE_QUOTED;
    else if (strnicmp(ptr, enc_base64, 6) == 0)
	ret = CE_BASE64;
    /* this one shouldn't really be here -- it's *never* been valid. -ST */
    else if (stristr(ptr, "uuencode") != NULL)
	ret = CE_UUENCODE;
    else		/* unknown encoding: treat as CE_7BIT (RFC 2045) */
	ret = CE_UNKNOWN;
    return ret;
}


/*
 * closenotes()
 *
 * Close the temp file; if it's been changed and <savechanges> is true, copy
 * mail back to the mailbox file.  If the mailbox file is larger now than it
 * was when we took the current temporary copy, we assume that new mail has
 * been appended to the mailbox since we took our copy, and call
 * copymail(TRUE,FALSE) to add this new mail to our temporary file and
 * internal <mbox> list *before* we overwrite the mailbox file with our
 * changed version!
 *
 * When copying mail back to the mailbox file, we take care to copy it back in
 * the same order that it started off: the <mbox> array may have been sorted
 * into date and/or descending order, but the .msgno fields represent the
 * order of messages as read from the mailbox, so we can (and do) re-sort it
 * back into original mailbox order first; we can then run through the <mbox>
 * list in order, copying all messages that aren't marked for deletion back to
 * the mailbox file, and this should ensure that the mailbox ordering is
 * preserved.  We rely on the .position and .size members of the <mbox> HTL
 * structures to be correct!
 *
 * We guarantee this: once this function has been called, any currently open
 * notes file will have been closed, and the global <mfile> will be NULL.
 */
void closenotes(BOOL savechanges)
{
    PCHTL cmsg;
    char tstring[LLINELEN];
    int i;
    BOOL inheader, done_status, skip;
    long end;
    FILE *nfile = NULL;
    char last_eol;
    BOOL locked = TRUE;		/* assume we get a lock, for now */
    BOOL writerr = FALSE;

    if (mfile == NULL)		/* no temp. file! */
	return;

    if (!mbchanged || !savechanges)	/* No update: just close the file */
    {
	(void) close_mfile(TRUE);	/* Close and unlink the file */
	return;
    }

    /* Re-sort the <mbox> list into .msgno order */
    cputs("Updating Mailbox... ");
    sort_mbox(FALSE);

    /*
     * Prepare to update the mailbox file (file name <notename> in directory
     * <maildir>; full file path in <notefile>).  We need to do the following:
     *   (a) Lock the mailbox file;
     *   (b) Check to see if new mail has been added to it (is it larger?),
     *       and if so, call copymail(TRUE, FALSE) to add the new messages to
     *       our mailbox copy and the <mbox> list (without sorting them);
     *   (c) Open the mailbox file for writing.
     * If any of the above stages fail, we can't (or shouldn't) go ahead with
     * the update; warn the user and leave our mailbox copy file intact, for
     * possible manual disaster recovery.
     */
    while (mlock(maildir, notename))	/* Part (a) */
    {
	if (abort_retry("Mail file is busy: (A)bort or (R)etry ? "))
	{
	    locked = FALSE;		/* Couldn't lock the notefile */
	    break;	/* out of the while (mlock) */
	}
    }
    if (locked)				/* Lock OK: move on to Part (b) */
    {
	if ((i = check_for_new_mail()) == 0 ||		/* no new mail */
	    (i > 0 && copymail(TRUE, FALSE) >= 0))	/* got new mail */
	{
	    if ((nfile = fopen(notefile, "w")) == NULL)		/* Part (c) */
		fopen_error(notefile, "original mail file for update!");
	}
    }
    if (nfile == NULL)		/* Something has gone wrong: give up now */
    {
	if (locked)		/* Remove lock file before giving up */
	    rmlock(maildir, notename);
	/* Close our copy of the notes file, but leave the file intact */
	(void) close_mfile(FALSE);	/* Don't unlink <mfilename> */
	if (access(mfilename, 0) == 0)	/* <mfilename> still exists */
	    errorbox(file_err, notefile, update_err,
		     "Changes discarded, temp file left intact:",
		     mfilename, NULL);
	else
	    errorbox(file_err, notefile, update_err, "Changes lost", NULL);
	return;
    }

    /* Copy tmp file back to notes file using position values from <mbox> */
    for (cmsg = mbox; cmsg != NULL; cmsg = cmsg->next)
    {
	if (cmsg->status & S_DELETE)	/* message is deleted: skip it */
	    continue;	/* with the for (cmsg) */
	done_status = FALSE;
	fseek(mfile, cmsg->position, SEEK_SET);		/* Start of next msg */
	end = cmsg->position + cmsg->size - 1L;		/* End of next msg */
	inheader = TRUE;
	last_eol = 1;
	while (ftell(mfile) < end &&
	       fgets(tstring, sizeof(tstring), mfile) != NULL)
	{
	    if (inheader)
	    {
		skip = FALSE;
		if (last_eol)
		{
		    if (htype_match(tstring, HT_STATUS))	/* Status: */
		    {
			done_status = TRUE;
			if (cmsg->status & S_READ)
			    verify_status_line(tstring, sizeof(tstring));
			else if (!(cmsg->status & (S_DELETE | S_READ)))
			    /* Unread: ignore the Status line */
			    skip = TRUE;
		    }
		    else if (ISBLANKLINE(tstring))	/* End of header */
		    {
			inheader = FALSE;
			if (!done_status && (cmsg->status & S_READ))
			{
			    /* Rely on ferror() check below */
			    (void) write_status_line(nfile, FALSE);
			    done_status = TRUE;
			}
		    }
		}
		last_eol = eoltest(tstring);
		if (skip)
		    continue;    /* with the while (ftell) */
	    }
	    (void) fputs(tstring, nfile);
	    if (ferror(nfile))
		break;	/* out of the while (ftell/fgets) */
	}
	/*
	 * If there was no body, we won't have seen the header-ending blank
	 * line, so we won't have written out the Status: header line
	 */
	if (!done_status && (cmsg->status & S_READ))
	    (void) write_status_line(nfile, FALSE);
	if (ferror(nfile))
	{
	    writerr = TRUE;
	    break;	/* out of the for (cmsg) */
	}
    }

    /* Close the open note file and remove its lock file */
    nmsgs = 0;
    if (fclose(nfile) != 0)		/* Write error on closing it */
	writerr = TRUE;
    rmlock(maildir, notename);		/* Remove lock file anyway */

    if (writerr)
    {
	(void) close_mfile(FALSE);	/* Don't unlink the old temp file */
	errorbox("Error writing mail file", notefile,
		 "Write error while updating mailbox (disk full?)",
		 "Old temp file left intact", "Aborting PCOak", NULL);
	exit(1);	/* too drastic: ST-FIXME */
    }

    /* That's it!  Close and unlink the old temp file; we're done here. */
    (void) close_mfile(TRUE);		/* Close and unlink the temp file */
}


/*
 * open_mfile()
 *
 * Try to open a new temporary file for our local copy of the mailbox; the
 * filename is to be stored in <mfilename>, and the open file handle is to be
 * stored in <mfile>.  We use this function to try and collect together all
 * the openings of this important file into one place.  (Actually, it's only
 * called once, from copymail() itself.)
 */
static int open_mfile(void)
{
    if (mfile != NULL)
	(void) close_mfile(TRUE);	/* close and unlink the old file */
    if ((mfile = opentemp(mfilename, "w+")) == NULL)
	return 1;
    return 0;
}


/*
 * close_mfile()
 *
 * Close the currently open temporary file for our local copy of the mailbox;
 * the filename is stored in <mfilename>, and the open file handle is stored
 * in <mfile>.  If <unlink_it> is TRUE, we unlink the old temporary file once
 * we've closed it.  We use this function to try and collect together all the
 * closings of this important file into one place.  (It's only called from
 * closenotes(), but from various different points within the function.)
 */
static int close_mfile(BOOL unlink_it)
{
    if (mfile == NULL)		/* Nothing to do! */
	return 1;
    (void) fclose(mfile);				/* OK: don't care */
    mfile = NULL;
    if (unlink_it && mfilename[0])
	(void) unlink(mfilename);
    mfilename[0] = '\0';
    return 0;
}
