/*
  This source is part of PCOak, an electronic mailer for DOS based on PCElm.

  PCElm is Copyright (c) 1988-1993 Martin Freiss and Wolfgang Siebeck
           Copyright (c) 1992-1999 Demon Internet
  PCOak is Copyright (c) 2000-2003 Simon Turner, Pete Disdale and dispc members

  Thanks to an agreement between the original PCElm authors and Demon Internet
  made in late 1999:

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License, version 1, as
	published by the Free Software Foundation.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	See the file COPYING, which contains a copy of the GNU General
	Public License.
*/

/*
 * dates.c -- date parsing and handling routines
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include "ustring.h"
#include "macros.h"

#define TZO(n)		((n) * 60)	/* Timezone offset: hours -> minutes */

/*
 * Global variables, local to this module
 */
static char *days[7] = {	/* Days of the week */
    "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"
};

static char *months[12] = {	/* Months of the year */
    "Jan", "Feb", "Mar", "Apr", "May", "Jun",
    "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};

static struct tzoff {		/* Timezone names & offsets from UTC */
    const char *name;
    int offset;
} tzoff[] = {
    /* RFC 822 codes to begin with */
    { "UT",	 TZO(+0)  },	/* Universal Coordinated Time: +0000 */
    { "GMT",	 TZO(+0)  },	/* Greenwich Mean Time: +0000 */
    { "EST",	 TZO(-5)  },	/* Eastern Standard Time: -0500 */
    { "EDT",	 TZO(-4)  },	/* Eastern Daylight Time: -0400 */
    { "CST",	 TZO(-6)  },	/* Central Standard Time: -0600 */
    { "CDT",	 TZO(-5)  },	/* Central Daylight Time: -0500 */
    { "MST",	 TZO(-7)  },	/* Mountain Standard Time: -0700 */
    { "MDT",	 TZO(-6)  },	/* Mountain Daylight Time: -0600 */
    { "PST",	 TZO(-8)  },	/* Pacific Standard Time: -0800 */
    { "PDT",	 TZO(-7)  },	/* Pacific Daylight Time: -0700 */
    { "Z",	 TZO(+0)  },	/* Military Z: +0000 */
    /* Other RFC 822 military codes carry no info: -0000 (RFC 1123, 2822) */
    /* Known non-RFC 822 codes from here on */
    { "BST",	 TZO(+1)  },	/* British Summer Time: +0100 */
    { "UTC",	 TZO(+0)  },	/* UTC: +0000 */
    { "UCT",	 TZO(+0)  },	/* UTC alternative name: +0000 */
    { "WET",	 TZO(+0)  },	/* Western Europe Time: +0000 */
    { "WEST",	 TZO(+1)  },	/* Western Europe Summer Time: +0100 */
    { "CET",	 TZO(+1)  },	/* Central Europe Time: +0100 */
    { "CEST",	 TZO(+2)  },	/* Central Europe Summer Time: +0200 */
    { "MET",	 TZO(+1)  },	/* Middle(?) Europe Time: +0100 */
    { "MES",	 TZO(+2)  },	/* Middle(?) Europe Summer Time: +0200 */
    { "EET",	 TZO(+2)  },	/* Eastern Europe Time: +0200 */
    { "EEST",	 TZO(+3)  },	/* Eastern Europe Summer Time: +0300 */
    { "Eastern", TZO(-5)  },	/* Eastern ???? Time: -0500 / -0400 */
    { "Central", TZO(-6)  },	/* Central ???? Time: -0600 / -0500 */
    { "Pacific", TZO(-8)  },	/* Pacific ???? Time: -0800 / -0700 */
};


/*
 * parse_date()
 *
 * Parse a Date: header body into UTC-based time_t format, and return this
 * time_t.  Also fill in the UTC -> message time offset, in minutes, to
 * <*offset> (value is +ve when message time is ahead of UTC).
 *
 * Armed with the true UTC time and this UTC -> message offset, we can sort on
 * UTC or message time, and (more importantly) we can produce UTC *or* message
 * timestamps by calling gmtime() on the original time_t value (for UTC) or
 * the time_t value plus the offset (for message time).
 *
 * Here is a selection of Date: formats that have been observed, all of which
 * parse correctly (excepting a couple of badly-formed timezones -- marked
 * with 'x' below -- but that's a minor problem), including the badly
 * non-conformant "mm/dd/yy" date, some non-Y2K-compliant years and various
 * known but nonetheless bogus timezone specifications; failures to comply
 * with RFCs 822 and 1123 are noted in the following categories:
 *	W	Bogus day of week
 *	D	Invalid date format (bogus month, mm/dd/yy etc.)
 *	Y	Missing/bogus year
 *	2	Non-Y2K-compliant year
 *	H	Missing hour digit (1-digit hour instead of 2-digit)
 *	O	Invalid order of parts (UN*X ctime() format instead of RFC 822)
 *	T	Bogus or missing timezone specifier
 *	!	Well, everything, really
 *
 * Errs	Date: header contents				  Parses to (UTC)
 * ----	---------------------				  ---------------
 *	Date: Tue, 6 Jul 93 17:52 PDT			  1993-07-07 00:52:00
 *	Date: 25 Jul 94 20:04 GMT			  1994-07-25 20:04:00
 *	Date: Tue, 02 Apr 1996 17:30:20 GMT		  1996-04-02 17:30:20
 *	Date: Thu, 27 Jan 00 22:26:17 EST		  2000-01-28 03:26:17
 *	Date: 14 Sep 2000 08:59:07 Z			  2000-09-14 08:59:07
 *	Date: Tue, 21 may 1998 21:58:06 -0500 (EST)	  1998-05-22 02:58:06
 *	Date:  Sat, 9 Nov 1999 00:10:13 +0000		  1999-11-09 00:10:13
 * H	Date:     Wed, 12 Jul 95 9:09:07 +0100		  1995-07-12 08:09:07
 * H	Date: 12 Nov 96  9:23:30 EDT			  1996-11-12 13:23:30
 * 2	Date: Sat, 1 Jan 100 16:15:21 BST		  2000-01-01 15:15:21
 * 2	Date: Tue, 29 Feb 19100 15:07:37 GMT		  2000-02-29 15:07:37
 * O	Date: Tue Jul  8 06:12:34 1997 -0700 (PDT)	  1997-07-08 13:12:34
 * OT	Date: Sat Dec 31 19:34:35  1994			  1994-12-31 19:34:35
 * T	Date: Fri, 04 Dec 1998 03:03:37			  1998-12-04 03:03:37
 * T	Date: Sun, 5 May 96 18:05 MET DST		  1996-05-05 16:05:00
 * T	Date: 08 Aug 00 20:49:02 +-0800			  2000-08-08 20:49:02 x
 * T	Date: Tue, 3 Sep 1996 17:51:41 +-100		  1996-09-03 17:51:41 x
 * T	DATE: 10 Oct 00 08:49:21 p.m.			  2000-10-10 20:49:21
 * W	Date: Sa, 14 Apr 2001 19:29:59 -1100		  2001-04-15 06:29:59
 * WT	Date: za, 14 aug 1999 19:59:08			  1999-08-14 19:59:08
 * HT	Date: Sun, 6 Oct 96 7:47:18 BST			  1996-10-06 06:47:18
 * HT	DATE: 17 Apr 01 5:42:29 PM			  2001-04-17 17:42:29
 * DT	Date: Monday, April 16, 2001 13:33:15		  2001-04-16 13:33:15
 * DT	Date: 07 December 1995 12:58			  1995-12-07 12:58:00
 * DT	Date: 25/12/94 15:32:01				  1994-12-25 15:32:01
 * DT	Date: 1998/09/02 08:17:14			  1998-09-02 08:17:14
 * DHT	Date: 3/27/99 8:18:46 PM ))			  1999-03-27 20:18:46
 * DHT	Date: 12/15/98 2:43:45 AM Pacific Daylight Time	  1998-12-15 09:43:45
 * DHT	Date: 11/13/1999 8:37:49 AM Pacific Daylight Time 1999-11-13 15:37:49
 *
 * and these are hopelessly broken formats which don't parse at all, and which
 * IMHO aren't worth bothering with (some of the foreign naming ones would be
 * possible, but what's the point?):
 *
 * YT	Date: Tue, 6 Feb 1 9:12:33 -07
 * WDT	Date: Di, 19 Mai 1998 01:29:14
 * WDT	Date: jeu., 30 juil. 1998 11:16:10
 * WDT	Date: lun., 3 aot 1998 01:56:24
 * WD	Date: sab, 02 dic 2000 15.46.09 +0100
 * WDT	Date: , 28  2000 17:08:54
 * WY	Date: 3, 29 Aug  15:9:39 +1000 (AUS EST)
 * !	Date: Today
 * !	Date: Armageddon
 * !	DATE: Feb 20/2001
 *
 * Note that there are many other Date: formats in use, but they generally
 * only occur within the *bodies* of mail messages (e.g. in the Micro$oft
 * quote-the-last-message block), rather than in the actual mail headers.
 *
 * Probably the most frequent problem is with bogus timezone formats.  I've
 * seen the following bogus formats:
 *
 *	+-100		+0-100		-0-1200		-0-800
 *	-00800 (PST)	-0860		--100		+03d00
 *	0000		GMT+2		GMT-2		BST-1
 *	MDT-7MST	UCT		ADT		IST
 *
 * as well as the following multi-word concoctions:
 *
 *	MET DST
 *	GMT +0000
 *	GMT Daylight Time
 *	Central Standard Time
 *	Central Daylight Time
 *	Eastern Daylight Time
 *	Pacific Standard Time
 *	Pacific Daylight Time
 *
 * Invalid and unsupported timezone strings are assumed to mean UTC (RFC 2822
 * suggests "-0000", which seems reasonable); I really don't think it's a good
 * idea to try and second-guess what they are supposed to mean (for example,
 * one of the above entries "+-100" actually means BST, i.e. +0100; but one of
 * the others, "+0-800" actually means PST, i.e. -0800).  Additionally, IME
 * the biggest source of broken timezones is junk mail (UCE/UBE etc.), and I
 * really don't see the point of wasting time supporting that!
 *
 * We allow the day-of-month, month, year and time to be in any order; if the
 * date is given as xx/yy/zz, it must be the first field, and it is assumed to
 * be in USA order (mm/dd/yy) [all the examples I have seen have been in this
 * order], unless the first field is > 1900, in which case we assume
 * yyyy/mm/dd, or is > 12, in which case we assume dd/mm/yy.  The timezone
 * must come after the time; we allow a non-timezone "am" or "pm" as a time
 * modifier, which must also come after the time, but this is not considered
 * part of the "timezone".
 *
 * We use a time_t value to represent the time, which means that the earliest
 * date we can handle is 1970-01-01.  Properly specified 4-digit years must
 * therefore be >= 1970 to be valid, and 2-digit years are assumed to be 19xx
 * if xx is >= 70, else 20xx; I know this breaks RFC 2822, which says that
 * 2-digit years should run from 1950-2049 instead of 1970-2069, but (a)
 * that's patently silly, and (b) it would prevent us using time_t!  In
 * accordance with RFC 2822, 3-digit years are assumed to be non-Y2K offsets
 * from 1900 (but, for parsing resilience, we only allow 100-199); we make
 * special Y2K exceptions for 5-digit years 19100 and up, which we assume are
 * actually 2000 and up; and we make a very special case for the wonderful
 * "19:0" type years which some software produces for 2000+.
 */
time_t parse_date(const char *datestr, short *offset)
{
    extern const char chs_wsp[];	/* " \t\r\n" -- whitespace chars */
    struct tm tm;
    char buf[81];		/* Maximum sensible length for date string */
    const char *fbuf;
    char cc;
    unsigned int xx, nx, hh, mm, ss;
    int i, n, nf, dd, mn, yy, tt, tz, dow;
    int incomment, isnumeric;
    time_t utc;
    int tzmins = 0L;
    long correction;

    ustrncpy(buf, datestr, sizeof(buf) - 1);	/* Local copy, for strtok() */
    dd = mn = yy = tt = tz = dow = -1;
    hh = mm = ss = 0;
    incomment = 0;

    for (fbuf = strtok(buf, chs_wsp), nf = 0; fbuf != NULL;
	 fbuf = strtok(NULL, chs_wsp), nf++)
    {
	if (fbuf[0] == '(')			/* Start of RFC 822 comment */
	{
	    incomment = 1;
	    continue;	/* with the for (fbuf) */
	}
	else if (incomment)			/* Currently in a comment */
	{
	    if (fbuf[strlen(fbuf) - 1] == ')')	/* End of comment! */
		incomment = 0;
	    continue;	/* with the for (fbuf) */
	}

	isnumeric = isdigit(fbuf[0]);	/* Is the first character numeric? */

	/*
	 * If this is the first field and it doesn't start with a number,
	 * check for the day-of-week; even though we don't do anything with
	 * it, this at least lets us eliminate this field quickly and get on
	 * with the next one
	 */
	if (nf == 0 && dow < 0 && !isnumeric)
	{
	    for (i = 0; i < (int) NELEM(days) && dow < 0; i++)
		if (strnicmp(fbuf, days[i], 3) == 0)
		    dow = i;
	    if (dow >= 0)
		continue;	/* with the for (fbuf) */
	}

	/*
	 * If this is the first field, check for mm/dd/yy[yy] date format;
	 * move on to the next field if we find it, marking dd, mn and yy as
	 * known
	 */
	if (nf == 0 && isnumeric && strchr(fbuf, '/') != NULL)
	{			/* May be mm/dd/yy, dd/mm/yy, yyyy/mm/dd */
	    if (sscanf(fbuf, "%2d/%2d/%4d%c", &mn, &dd, &yy, &cc) == 3 ||
		(sscanf(fbuf, "%4d/%2d/%2d%c", &yy, &mn, &dd, &cc) == 3 &&
		 yy >= 1970))
	    {
		if (mn > 12 && mn <= 31 && dd >= 1 && dd <= 12)
		{				/* Might be dd/mm/yy... */
		    int tmp = mn;		/* Swap <mn> and <dd> */
		    mn = dd;
		    dd = tmp;
		}
		if (mn >= 1 && mn <= 12 &&			/* Month OK */
		    dd >= 1 && dd <= 31 &&			/* Day OK */
		    ((yy >= 0 && yy < 100) || yy >= 1970))	/* Year OK */
		{
		    mn--;		/* Move month from 1-12 to 0-11 */
		    if (yy < 100)		/* 2-digit year */
			yy += (yy >= 70) ? 1900 : 2000;
		    continue;	/* with the for (fbuf) */
		}
	    }
	    dd = mn = yy = -1;	/* Reset these: they are all unknown */
	}

	/*
	 * If not already found, look for day of month number; move on to the
	 * next field if we find it.  If we've already seen the month, allow a
	 * comma at the end of the field
	 */
	if (dd < 0 && isnumeric)
	{
	    if ((n = sscanf(fbuf, "%u%c", &xx, &cc)) == 1 ||
		(n == 2 && mn >= 0 && cc == ',') && xx >= 1 && xx <= 31)
	    {
		dd = (int) xx;
		continue;	/* with the for (fbuf) */
	    }
	}

	/*
	 * If not already found, look for the month name, which must match for
	 * the first 3 characters, case-insensitively; move on to the next
	 * field if we find it
	 */
	if (mn < 0 && !isnumeric)
	{
	    for (i = 0; i < (int) NELEM(months) && mn < 0; i++)
		if (strnicmp(fbuf, months[i], 3) == 0)
		    mn = i;
	    if (mn >= 0)
		continue;	/* with the for (fbuf) */
	}

	/*
	 * If not already found, look for the year, which may be 2-digit
	 * (00-99), 3-digit (100-199), 4-digit (1970-) or 5-digit (19100-);
	 * move on to the next field if we find it.  As a very special case,
	 * we also support the broken non-Y2K-compliant years 19:0-19:9 as
	 * 2000-2009 respectively
	 */
	if (yy < 0 && isnumeric)
	{
	    if (sscanf(fbuf, "%u%n%c", &xx, &nx, &cc) == 1 &&
		(nx == 2 || (nx == 4 && xx >= 1970) ||
		 (nx == 3 && xx >= 100 && xx <= 199) ||
		 (nx == 5 && xx >= 19100)))
	    {
		if (nx == 2)		/* 2 digit year: add century */
		    xx += (xx >= 70) ? 1900 : 2000;
		else if (nx == 3)	/* 3 digit non-Y2K year: add 1900 */
		    xx += 1900;
		else if (nx == 5)	/* 5 digit non-Y2K year: fix */
		    xx -= 19000 - 1900;		/* subtract 19000, add 1900 */
		yy = (int) xx;
		continue;	/* with the for (fbuf) */
	    }
	    if (strncmp(fbuf, "19:", 3) == 0 && strlen(fbuf) == 4 &&
		isdigit(fbuf[3]))
	    {
		yy = 2000 + (fbuf[3] - '0');
		continue;	/* with the for (fbuf) */
	    }
	}

	/*
	 * If not already found, look for the time string as hh:mm[:ss]; move
	 * on to the next field if we find it.  Note that we allow times up to
	 * 23:59:60; the 60-second part after 23:59 allows for a leap second,
	 * in accordance with STD 12.
	 */
	if (tt < 0 && isnumeric)
	{
	    if (strchr(fbuf, ':') != NULL &&
		(n = sscanf(fbuf, "%2u:%2u:%2u%c", &hh, &mm, &ss, &cc)) > 1 &&
		n < 4 && hh < 24 && mm < 60 &&
		(ss < 60 || (ss == 60 && hh == 23 && mm == 59)))
	    {
		tt = 1;
		continue;	/* with the for (fbuf) */
	    }
	    hh = mm = ss = 0;	/* Reset these: try again later */
	}

	/*
	 * If we've found the time, but not the timezone, look for the
	 * timezone code; move on to the next field if we find it
	 */
	if (tz == -1 && tt >= 0)
	{
	    if (hh <= 12 && (stricmp(fbuf, "pm") == 0 ||	/* Afternoon */
			     stricmp(fbuf, "p.m.") == 0))
	    {
		if (hh < 12)
		    hh += 12;	/* Keep looking for timezone: leave tz = -1 */
		continue;	/* with the for (fbuf) */
	    }
	    else if (hh <= 12 && (stricmp(fbuf, "am") == 0 ||	/* Morning */
				  stricmp(fbuf, "a.m.") == 0))
	    {
		hh += 0;	/* Keep looking for timezone: leave tz = -1 */
		continue;	/* with the for (fbuf) */
	    }
	    else if (fbuf[0] == '+' || fbuf[0] == '-')	/* +hhmm or -hhmm ? */
	    {
		if (strlen(fbuf) == 5 && isdigit(fbuf[1]) &&  /* 822: 4DIGIT */
		    sscanf(fbuf + 1, "%u%n", &xx, &nx) == 1 && nx == 4 &&
		    xx < 2400 && (xx % 100) < 60)
		{
		    tzmins = (int) ((xx / 100) * 60 + (xx % 100));
		    if (fbuf[0] == '-')
			tzmins = -tzmins;
		    tz = 1;
		}
		continue;	/* with the for (fbuf) */
	    }
	    else if (!isnumeric)	/* May be special timezone string */
	    {
		for (i = 0; i < (int) NELEM(tzoff) && tz < 0; i++)
		    if (stricmp(fbuf, tzoff[i].name) == 0)
		    {
			tzmins = tzoff[i].offset;	/* offset in mins */
			tz = 1;
		    }
		continue;	/* with the for (fbuf) */
	    }
	    else			/* Bogus: ignore it */
		continue;	/* with the for (fbuf) */
	}

	/*
	 * If we've found both the time and the timezone, some clueless idiot
	 * may have put a "Daylight" or "DST" modifier next to it...
	 */
	if (tz >= 0 && tt >= 0)
	{
	    if (stricmp(fbuf, "Daylight") == 0 || stricmp(fbuf, "DST") == 0)
	    {
		tzmins += 60;		/* Daylight time is 1 hour ahead */
		continue;
	    }
	    if (stricmp(fbuf, "Standard") == 0 || stricmp(fbuf, "Time") == 0)
		continue;
	}
    }

    if (dd < 0 || mn < 0 || yy < 0 || tt < 0)	/* Parsing failed */
    {
	*offset = 0;
	return 0L;			/* 1970-01-01 00:00:00 UTC */
    }

    /*
     * Now find the time_t (i.e. UTC) representation of the (message) time
     * we've found.  The message time is <tzmins> ahead of UTC, which we will
     * deal with later (we find the time_t representation assuming that
     * message time is UTC, and then subtract <tzmins> minutes to get true
     * UTC).
     *
     * We use mktime(), which will apply our local timezone rules, so the
     * answer will be out by the number of seconds given in global <timezone>
     * (which is the offset from UTC to the basic, non-daylight, local time
     * zone); the Borland mktime() also applies our daylight saving rules,
     * which complicates things still further.  Once we've got the result, we
     * therefore need to subtract <timezone> seconds from it, and add 3600 if
     * it's DST, to remove the effects of the local timezone and get a true
     * time_t representation of the time.
     *
     * We can then subtract <tzmins> from the result to give the *true*
     * timestamp in UTC; when we want to recover the original message time, we
     * can add <tzmins> on to the time_t value and then use gmtime() (which
     * won't do anything with local timezone or DST) to get a textual version
     * of the message time.
     */
    tm.tm_sec   = (int) ss;
    tm.tm_min   = (int) mm;
    tm.tm_hour  = (int) hh;
    tm.tm_mday  = dd;
    tm.tm_mon   = mn;
    tm.tm_year  = yy - 1900;
    tm.tm_wday  = 0;
    tm.tm_yday  = 0;
    tm.tm_isdst = 0;		/* If possible, disable daylight savings */

    utc = mktime(&tm);		/* Initial stab at time_t representation */
    correction = - timezone + ((tm.tm_isdst) ? 3600L : 0L);

    *offset = tzmins;
    return (time_t) ((long) utc + correction - tzmins * 60L);
}


/*
 * dispdate()
 *
 * Return a pointer to a static buffer containing a display date ready for use
 * on the main index page; the time in UTC is <utc>, which we may want to know
 * for comparison purposes, but we want the display date to be in the message
 * timezone, which is <offset> seconds ahead of UTC.  We achieve this by
 * adding <offset> to the actual UTC time <utc>, to produce a pseudo-UTC time
 * which can be fed into gmtime() and will produce a struct tm that represents
 * the local message time, as seen in the "Date:" header field.  This is a
 * useful thing to do, as calling gmtime() doesn't involve any use of our
 * local timezone, daylight saving time etc.
 *
 * If the UTC time is earlier than global <yeartime>, display the year rather
 * than the hh:mm time.
 */
char *dispdate(time_t utc, long offset)
{
    extern time_t yeartime;
    static char str[16];
    struct tm *gtm;
    time_t t = (time_t) ((long) utc + offset);

    gtm = gmtime(&t);		/* Find unmolested time fields */
    if (utc < yeartime)		/* Display year, not time */
	sprintf(str, "%2d %3.3s %5d",					/*OK*/
		gtm->tm_mday, months[gtm->tm_mon], gtm->tm_year + 1900);
    else			/* Display time, not year */
	sprintf(str, "%2d %3.3s %02d:%02d",				/*OK*/
		gtm->tm_mday, months[gtm->tm_mon], gtm->tm_hour, gtm->tm_min);
    return str;
}


/*
 * rfcdate()
 *
 * Return a pointer to a static buffer containing a "Date" string (RFC 2822
 * <date-time> format) for local time <t>.
 *
 * (ST: Yeehah! Finally! We're going to get timezones right!)
 */
char *rfcdate(time_t t)
{
    static char str[40];
    struct tm *ltm;
    int utcoffset;

    ltm = localtime(&t);	/* Local time for this system */
    utcoffset = (int) (-timezone / 60L);
    if (ltm->tm_isdst)
	utcoffset += 60;
    sprintf(str,						  	/*OK*/
	    "%.3s, %d %.3s %04d %02d:%02d:%02d %+05d (%.5s)",
	    days[ltm->tm_wday], ltm->tm_mday, months[ltm->tm_mon],
	    ltm->tm_year + 1900, ltm->tm_hour, ltm->tm_min, ltm->tm_sec,
	    (utcoffset / 60) * 100 + (utcoffset % 60),
	    tzname[(ltm->tm_isdst) ? 1 : 0]);
    return str;
}
