/*
  This source is part of PCOak, an electronic mailer for DOS based on PCElm.

  PCElm is Copyright (c) 1988-1993 Martin Freiss and Wolfgang Siebeck
           Copyright (c) 1992-1999 Demon Internet
  PCOak is Copyright (c) 2000-2002 Simon Turner, Pete Disdale and dispc members

  Thanks to an agreement between the original PCElm authors and Demon Internet
  made in late 1999:

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License, version 1, as
	published by the Free Software Foundation.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	See the file COPYING, which contains a copy of the GNU General
	Public License.
*/

/*
 * display.c -- functions to support the global <display> structure
 */

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <dos.h>
#include <dir.h>
#include "pcoak.h"
#include "display.h"

/*
 * A structure describing a cursor setup.  Cursors under DOS are drawn from
 * one scanline to another (top to bottom); in mono screen modes, the scan
 * lines correspond to the pixel rows in the font characters, but in colour
 * modes they can only adopt certain positions, and only run up to 7 even if
 * the character height is (say) 12 or 14.  The top usable scan line seems to
 * depend on the adapter!  On a colour VGA display under Win98, in a 25 line
 * DOS box, using 1-7 gives a full-height block if windowed, but 1 line short
 * of full height if full screen; using 0-7 gives a full height block in full
 * screen, but gives an over-high one (extending one row into the next line)
 * if windowed!  I reckon 0 is the correct top value to use.  Typical values
 * -- and these are the ones we use internally -- are:
 *
 *				Mono		Colour
 *				-----		------
 * Default underline		11-12		 6-7
 * Half-height block		 5-12		 4-7
 * Full-height block		 0-12		 0-7
 *
 * It's worth noting that you seem to be able to set values outside the 0-7
 * range in colour mode -- e.g. 0-12 for a block cursor -- but the adapter
 * internally resets them to lie in the expected range at some stage.
 *
 * Note further that the scanlines are limited to bits 0-4 of the byte,
 * i.e. the range 0-31.  Values above this for the bottom line will have
 * unpredictable effects; for the top line, bit 7 should always be 0, but
 * bits 5 and 6 control the cursor "blink":
 *	6 5	Result
 *	---	------
 *	0 0	Normal
 *	0 1	Invisible
 *	1 0	Blink erratically
 *	1 1	Blink slowly
 *
 * Note that some or all of this may not work, especially in a windowed DOS
 * box.  PCElm used to have an (unused) function called NoCursor() which (for
 * either mono or colour display, which seems odd) set the top line to 32
 * (i.e. 0 with bit 5 set) and the bottom line to 0; this works in a full
 * screen DOS box, but not a windowed one.
 */
struct cursor_info {
    unsigned char top_line;
    unsigned char bottom_line;
};

/*
 * Global variables defined and used here
 */
struct disp_info display;
static struct cursor_info nrm_crsr;	/* Normal display cursor */
static struct cursor_info ovr_crsr = {255,255};	/* Cursor for overwrite mode */
static struct cursor_info ins_crsr = {255,255};	/* Cursor for insert mode */

/*
 * Global variables defined elsewhere and used here
 */
extern char *cwd;		/* current directory */

/*
 * Functions local to the module
 */
static void disp_getxy(void);
static void disp_window(int left, int top, int right, int bottom);
static void puttextinfo(const struct text_info *t);
static void getcursor(unsigned char *top_line, unsigned char *bottom_line);


/*
 * disp_init()
 *
 * Initialise the display setup
 */
void disp_init(void)
{
    extern struct text_info _video;
    unsigned char far *screen_lines = MK_FP(0x40, 0x84);
    unsigned char far *screen_cols  = MK_FP(0x40, 0x4A);

    gettextinfo((struct text_info *) &display);		/*lint !e740 */	/*OK*/
    if (display.currmode != 7)
    {
	if (*screen_lines > 1)
	    display.screenheight = _video.screenheight = (*screen_lines) + 1;
	else
	    display.screenheight = _video.screenheight = 25;
	display.screenwidth = _video.screenwidth  = *screen_cols;
    }
    disp_window(1, 1, display.screenwidth, display.screenheight);
    /* Note the initial cursor size settings */
    getcursor(&nrm_crsr.top_line, &nrm_crsr.bottom_line);
    disp_helplines(0);

    /* If not yet defined, set sizes for Overwrite and Insert cursors */
    if (ovr_crsr.top_line == 255)	/* Overwrite unset */
	ovr_crsr = nrm_crsr;
    if (ins_crsr.top_line == 255)	/* Insert unset */
    {
	ins_crsr.top_line = 0;
	ins_crsr.bottom_line = 12;
    }
}


/*
 * disp_helplines()
 *
 * Note the number of lines which are given over to help on the main index
 * display; this changes the number of lines available for mail items.  We
 * need to leave (MAIN_TOP_LINES + 1) free at the top (for the title lines
 * etc.), (MAIN_BOT_LINES + 1) free at the bottom (for the Command prompt and
 * other prompts), and room for <nlines> of help information -- this, if > 0,
 * needs blank lines around it as well.
 */
void disp_helplines(int nlines)
{
    display.main_lines = (int) display.screenheight - (MAIN_TOP_LINES + 1) -
	(MAIN_BOT_LINES + 1) - ((nlines > 0) ? (nlines + 1) : 0);
}


/*
 * disp_getxy()
 *
 * Note the current cursor position in global <display>
 */
static void disp_getxy(void)
{
    display.curx = (unsigned char) wherex();
    display.cury = (unsigned char) wherey();
}


/*
 * disp_fixtop()
 *
 * Set a scrolling window such that the top <nfixed> lines are fixed and don't
 * scroll
 */
void disp_fixtop(int nfixed)
{
    if (nfixed < 0 || nfixed >= display.screenheight)
	return;
    disp_window(1, 1 + nfixed, display.screenwidth, display.screenheight);
}


/*
 * disp_window()
 *
 * Call the system window() command, noting the settings in <display>
 */
static void disp_window(int left, int top, int right, int bottom)
{
    display.winleft = (unsigned char) left;
    display.wintop = (unsigned char) top;
    display.winright = (unsigned char) right;
    display.winbottom = (unsigned char) bottom;
    window(left, top, right, bottom);
}


/*
 * disp_fullscreen()
 *
 * Set the display window to the full screen
 */
void disp_fullscreen(void)
{
    disp_window(1, 1, display.screenwidth, display.screenheight);
}


/*
 * disp_gotoxy()
 *
 * Move to absolute cursor position (<x>, <y>), but obeying the current window
 * settings; if the given cursor position isn't within the current window,
 * leave it where it is
 */
void disp_gotoxy(int x, int y)
{
    if (x >= display.winleft && x <= display.winright &&
	y >= display.wintop  && y <= display.winbottom)
	gotoxy(x - display.winleft + 1, y - display.wintop + 1);
}


/*
 * disp_clear()
 *
 * Set the display window to the full screen and clear the screen using the
 * current attribute
 */
void disp_clear(void)
{
    disp_window(1, 1, display.screenwidth, display.screenheight);
    clrscr();
}


/*
 * puttextinfo()
 *
 * The opposite of gettextinfo.  (Why isn't this in the BC++ lib? -ST)
 */
static void puttextinfo(const struct text_info *t)
{
    extern struct text_info _video;

    textmode(t->currmode);
    if (_video.screenheight != t->screenheight)
	_video.screenheight = t->screenheight;	/* correct row count */
    window(t->winleft, t->wintop, t->winright, t->winbottom);
    textattr(t->attribute);
    gotoxy(t->curx, t->cury);
}


/*
 * color()
 *
 * Return TRUE if the current graphics mode is a colour one, FALSE if not
 */
BOOL color(void)
{
    union REGS REG;

    REG.h.ah = 15;
    REG.h.bh = 00;
    REG.h.ch = 0;
    REG.h.cl = 0;
    int86(0x10, &REG, &REG);
    return (REG.h.al != 7) ? TRUE : FALSE;	/* TRUE if video mode is 7 */
}


/************************************************************************/
/*	Cursor stuff							*/
/************************************************************************/

/*
 * getcursor()
 *
 * Get the current cursor top and bottom lines into <*top_line> and
 * <*bottom_line> respectively
 */
static void getcursor(unsigned char *top_line, unsigned char *bottom_line)
{
    union REGS r;

    r.x.ax = 0x0300;
    r.x.bx = 0;
    r.x.cx = 0;
    int86(0x10, &r, &r);
    *top_line = r.h.ch;
    *bottom_line = r.h.cl;
}


/*
 * setcursor()
 *
 * Set the cursor's top and bottom scanlines to <top_line> and <bottom_line>
 * respectively
 */
static void setcursor(unsigned char top_line, unsigned char bottom_line)
{
    union REGS r;

    r.x.ax = 0x0100;
    r.x.bx = 0;
    r.h.ch = top_line;
    r.h.cl = bottom_line;
    int86(0x10, &r, &r);
}


/*
 * inscursor()
 *
 * Set the cursor to the text editing "Insert" shape
 */
void inscursor(void)
{
    setcursor(ins_crsr.top_line, ins_crsr.bottom_line);
}


/*
 * ovrcursor()
 *
 * Set the cursor to the text editing "Overwrite" shape
 */
void ovrcursor(void)
{
    setcursor(ovr_crsr.top_line, ovr_crsr.bottom_line);
}


/*
 * normcursor()
 *
 * Set the cursor to the "Normal" shape, to be used for all non-text editing
 * work (and to be restored on shell, exit etc.)
 */
void normcursor(void)
{
    setcursor(nrm_crsr.top_line, nrm_crsr.bottom_line);
}


/*
 * define_cursors()
 *
 * Define the top and bottom lines for both Insert and Overwrite cursors
 */
void define_cursors(unsigned char ovrtop, unsigned char ovrbot,
		    unsigned char instop, unsigned char insbot)
{
    ovr_crsr.top_line = ovrtop;
    ovr_crsr.bottom_line = ovrbot;
    ins_crsr.top_line = instop;
    ins_crsr.bottom_line = insbot;
}


/************************************************************************/
/*	system() calls which protect the screen settings		*/
/************************************************************************/

/*
 * disp_system()
 *
 * Run "system(cmd)", making sure we protect our screen settings: if the user
 * runs something as part of the system() which changes them, we need to
 * change them back once we return.  Similarly, we need to ensure that we
 * return to our original working directory (which is where our temp. files
 * live), or all hell may break loose.  If <clear> is true, clear the screen
 * first.
 *
 * If <cmd> begins with the special character NPCMD, skip over this character
 * when calling system(); the leading NPCMD character is an internal indicator
 * that this command shouldn't get a "press any key to continue" prompt after
 * completion, and shouldn't be passed to system() for obvious reasons!
 */
int disp_system(const char *cmd, BOOL clear)
{
    int ret;

    disp_getxy();		/* Store the current cursor position */
    if (display.wintop != 1)	/* Temporarily revert to fullscreen mode */
	window(1, 1, display.screenwidth, display.screenheight);
    if (clear)
    {
	textattr(col_system);	/* system screen attribute */
	clrscr();
    }
    ret = system((cmd[0] == NPCMD) ? (cmd + 1) : cmd);
    if (display.wintop != 1)	/* Restore <display> windowed mode */
	window(1, display.wintop, display.screenwidth, display.screenheight);
    puttextinfo((struct text_info *) &display);		/*lint !e740*/ /* OK */
    normcursor();		/* Make sure we've got the default cursor */
    if (clear)
	textattr(col_normal);
    setdisk(cwd[0] - 'A');	/* 0 = A, 1 = B, 2 = C etc. */
    chdir(cwd);
    return ret;
}
