/*
  This source is part of PCOak, an electronic mailer for DOS based on PCElm.

  PCElm is Copyright (c) 1988-1993 Martin Freiss and Wolfgang Siebeck
           Copyright (c) 1992-1999 Demon Internet
  PCOak is Copyright (c) 2000-2002 Simon Turner, Pete Disdale and dispc members

  Thanks to an agreement between the original PCElm authors and Demon Internet
  made in late 1999:

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License, version 1, as
	published by the Free Software Foundation.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	See the file COPYING, which contains a copy of the GNU General
	Public License.
*/

#include <stdio.h>
#include <ctype.h>
#include <string.h>		/* NULL is defined here -ST */
#include "pcoak.h"		/* for BOOL typedef */
#include "header.h"
#include "chars.h"

/*
 * External global variables, defined elsewhere and used here
 */
extern BOOL mmdf;

/*
 * Global variables defined and used here; also used elsewhere
 */
const char *headername[NUM_FIXED_HEADERTYPES] = {
    "",					/* HT_SOH -- variable! See set_soh() */
    "From:",				/* HT_FROM */
    "Reply-To:",			/* HT_REPLYTO */
    "Date:",				/* HT_DATE */
    "To:",				/* HT_TO */
    "Cc:",				/* HT_CC */
    "Bcc:",				/* HT_BCC */
    "Message-Id:",			/* HT_MSGID */
    "Subject:",				/* HT_SUBJECT */
#if 0	/* Will be needed later */
    "In-Reply-To:",			/* HT_INREPLYTO */
    "References:",			/* HT_REFERENCES */
#endif
    "Status:",				/* HT_STATUS */
    "MIME-Version:",			/* HT_MIME_VERSION */
    "Content-Type:",			/* HT_CONTENT_TYPE */
    "Content-Transfer-Encoding:",	/* HT_CONTENT_XFER_ENCODING */
    "Content-ID:",			/* HT_CONTENT_ID */
    "Content-Description:",		/* HT_CONTENT_DESCRIPTION */
    "Content-Disposition:",		/* HT_CONTENT_DISPOSITION */
};

/*
 * Static functions defined and used here
 */
static BOOL exact_match(const char *p, const char *hdr);
static BOOL fieldname_match(const char *p, const char *hdr);


/*
 * set_soh()
 *
 * Sets the HT_SOH entry in global headername[] to the appropriate string for
 * the new system Start Of Header, <soh> -- THIS *MUST* BE A STATIC STRING!
 * (In reality, it will be one of the startofhdr_store[] strings from tcmain.c)
 */
void set_soh(const char *soh)
{
    headername[HT_SOH] = soh;
}


/*
 * htype()
 *
 * Return the header token type for header line <p>
 */
HEADERTYPE htype(const char *p)
{
    const char *s;
    int i;

    s = p;
    if (mmdf != 0)
    {
	int is_soh = 1;

	/*
	 * MMDF type mailbox (yecch). Check if this line contains
	 * Ctrl-A's only. Ignore CR and LF.
	 */
	do
	{
	    if (*s != 0x01)
	    {
		if (*s == '\n')
		    continue;
		if (*s == '\r')
		    continue;
		is_soh = 0;
		break;
	    }
	    else
		is_soh++;
	}
	while (*++s);
	if (is_soh > 3)
	    return HT_SOH;
    }

    if (exact_match(p, headername[HT_SOH]))
	return HT_SOH;			/* Exact match for SOH string */

    for (i = 0; i < (int) NUM_FIXED_HEADERTYPES; i++)
	if (fieldname_match(p, headername[i]))
	    return (HEADERTYPE) i;	/* Cast will always be OK -ST */

    if (strnicmp(p, "Content-", 8) == 0 && isalnum(p[8]))
	return HT_CONTENT_UNKNOWN;	/* Unknown MIME Content-* header */

    return HT_UNKNOWN;		/* Completely unknown header */
}


/*
 * htype_match()
 *
 * Return TRUE if header line <p> appears to be header type <type>, false if
 * not -- a quicker bet than having to use "htype() == HT_SOH" tests when we
 * know the expected result.
 */
BOOL htype_match(const char *p, HEADERTYPE type)
{
    if (type == HT_SOH)		/* Case-sensitive match for variable SOH */
	return exact_match(p, headername[HT_SOH]);
    else if (type < NUM_FIXED_HEADERTYPES)
	return fieldname_match(p, headername[type]);
    return FALSE;
}


/*
 * hdr_body()
 *
 * Return a pointer to the first non-LWSP character in the body of header
 * field <field>, whose type is <type>
 */
const char *hdr_body(const char *field, HEADERTYPE type)
{
    const char *p;

    if (type == HT_UNKNOWN)	/* Find the first colon */
    {
	if ((p = strchr(field, ':')) == NULL)
	    return field;
	p++;
    }
    else
	p = field + strlen(headername[type]);
    return skip_lwsp(p);
}


/*
 * exact_match()
 *
 * Return TRUE if header line <p> starts with an exact match for type
 * <hdr>, false if not; we do a case-insensitive match, and insist on there
 * being whitespace after the colon (which is the last character in the header
 * field name <hdr>)
 */
static BOOL exact_match(const char *p, const char *hdr)
{
    return (strncmp(p, hdr, strlen(hdr)) == 0) ? TRUE : FALSE;
}


/*
 * fieldname_match()
 *
 * Return TRUE if header line <p> appears to be "fixed" header field type
 * <hdr>, false if not; we do a case-insensitive match, but we no longer
 * insist on there being whitespace after the colon (which is the last
 * character in the header field name <hdr>); RFC [2]822 have never required
 * any space after the colon, and the new Demon WebMail sending mechanism
 * seems fond of generating "Cc:address" lines with no space.
 */
static BOOL fieldname_match(const char *p, const char *hdr)
{
    size_t l = strlen(hdr);
    if (l > 0 && strnicmp(p, hdr, l) == 0)
	return TRUE;		/* header matched up to & including ':' */
    return FALSE;
}
