/*
  This source is part of PCOak, an electronic mailer for DOS based on PCElm.

  PCElm is Copyright (c) 1988-1993 Martin Freiss and Wolfgang Siebeck
           Copyright (c) 1992-1999 Demon Internet
  PCOak is Copyright (c) 2000-2001 Simon Turner, Pete Disdale and dispc members

  Thanks to an agreement between the original PCElm authors and Demon Internet
  made in late 1999:

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License, version 1, as
	published by the Free Software Foundation.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	See the file COPYING, which contains a copy of the GNU General
	Public License.
*/

#include <stdlib.h>
#include <stdio.h>
#include <conio.h>
#include <ctype.h>
#include <dos.h>
#include <stdarg.h>
#include <string.h>
#include "pcoak.h"
#include "keyboard.h"
#include "ustring.h"
#include "chars.h"
#include "display.h"

#define ERR_LINES	5

struct savebox {		/* Struct for saving screen areas */
    int winx1, winx2, winy1, winy2;	/* Corners for window */
    char *screenbuff;			/* Buffer for saved screen data */
    struct text_info ti;		/* Original text_info settings */
};

/*
 * Global variables defined and used here
 */
const char *pressakey = "\fPress a key to continue";

/*
 * Global variables defined elsewhere and used here
 */
extern char keep_savefile[];	/* Name of last file saved (recall using F3) */
extern BOOL insertmode;		/* Insert or overwrite in maxgets()? */
extern BOOL tty;		/* tells if stdin is a tty */

/*
 * Functions defined and used in this module
 */
void beep(void);
char maxgets(char *s, int maxlen);

/*
 * Functions defined elsewhere and used in this module
 */
char *seqstr(size_t n, char ch);
unsigned int getkey(void);


/*
 * beep()
 *
 * Make a medium-pitched beep sound with the speaker
 */
void beep(void)
{
    sound(440);
    delay(250);
    nosound();
}


/* go one word to the left */
static int wordleft(const char *ststart, int pos)
{
    pos--;
    /* skip over blanks */
    while (pos && isspace(ststart[pos]))
	pos--;
    /* skip over current word */
    while (pos && !isspace(ststart[pos]))
	pos--;
    if (isspace(ststart[pos]))
	pos++;
    return pos;
}


/* go one word to the right */
static int wordright(const char *ststart, int pos)
{
    int k, len = (int) strlen(ststart);

    k = pos;
    /* skip over current word */
    while (k < len && !isspace(ststart[k]))
	k++;

    /* skip over blanks */
    while (++k < len)
	if (!isspace(ststart[k]))
	    break;
    if (k < len)
	pos = k;
    else
	pos = len;
    return pos;
}


static int find_pos(const char *s, int pos)
{
    int i, n;

    i = n = 0;
    while (i < pos)
	if (s[i++] == '\t')
	    n += (8 - (n % 8));
	else
	    n++;
    return n;
}


/*
 * maxgets()
 *
 * gets, but using global attribute for screen display
 * *s is the default string
 */
char maxgets(char *s, int maxlen)
{
    char *ststart;
    int i, xs, ys, n;
    unsigned int ch;
    size_t l;
    BOOL exitloop = FALSE;
    char retchar = '\0';	/* Return null unless otherwise noted */

    xs = wherex();
    ys = wherey();
    ststart = s;
    if (strlen(ststart) + (size_t) xs > display.screenwidth)
    {
	char t[LINELEN];	/* much bigger than we need -ST */

	i = display.screenwidth - xs + 1;
	ustrncpy(t, s, (size_t) i);
	cputs(t);
    }
    else
	cputs(s);
    gotoxy(xs, ys);
    if (insertmode)	/* display appropriate cursor */
	inscursor();
    else
	ovrcursor();

    i = 0;
#if 0			/* Don't start at end of string, but at beginning */
    i = strlen(s);
    s += i;
#endif

    while (!exitloop)
    {					/* returning on ^M */
	do
	    ch = getkey();
	while (ch == 0);
	switch (ch)
	{
	case KLF:
	case KCR:
	case KKPENTER:	/* Keypad Enter */
	    kill_trail_lwsp(ststart);
	    exitloop = TRUE;		/* return '\0' (default) */
	    break;

	case KINS:
	case KKP0:	/* Keypad Insert */
	    if ((insertmode = !insertmode) != FALSE)	/*lint !e731*/ /* OK */
		inscursor();
	    else
		ovrcursor();
	    break;

	case KDEL:
	case KKPCOMMA:	/* Keypad Del */
	    if ((size_t) i < strlen(ststart))
	    {
		cputs(" ");
		s++;
		i++;
	    }
	    else
		break;
	    /* fall through */

	case KBS:
	    if (i > 0)
	    {
		l = strlen(ststart);
		if (maxlen == display.screenwidth)
		{
		    /* TABS allowed */
		    gotoxy(xs, ys);
		    clreol();	/* sledgehammer - wipe the lot out */
		}
		else
		{
		    gotoxy(wherex() - 1, ys);
		    cputs(seqstr(l - (size_t) i + 1, ' '));
		}
		memmove(ststart + i - 1, ststart + i, l - (size_t) i + 1);  /*OK*/
		s--;
		i--;
		if (maxlen == display.screenwidth)
		    cputs("\r");
		else
		    gotoxy(xs, ys);
		cputs(ststart);
		gotoxy(xs + find_pos(ststart, i), ys);
	    }
	    break;

	case KTAB:
	    if (maxlen == display.screenwidth)
	    {
		if ((size_t) i == strlen(ststart))
		    *(s + 1) = '\0';	/* keep the string terminated */
		*s++ = '\t';	/* GBD use the tab char and not ch */
		i++;
		if (maxlen == display.screenwidth)
		    cputs("\r");
		else
		    gotoxy(xs, ys);
		cputs(ststart);
		gotoxy(xs + find_pos(ststart, i), ys);
	    }
	    break;

	case CTRL_U:
	case KESC:
	    if (i == 0)		/* At start of line: abort edit */
	    {
		if (*s == ' ' || *s == '\0')
		{
		    *s = '\0';
		    exitloop = TRUE;	/* return '\0' (default) */
		    break;
		}
	    }
	    gotoxy(xs, ys);
	    if (maxlen == display.screenwidth)
	    {
		/* TABS allowed */
		clreol();	/* sledgehammer - wipe the lot out */
	    }
	    else
		cputs(seqstr(strlen(ststart), ' '));
	    gotoxy(xs, ys);
	    s = ststart;
	    *s = '\0';
	    i = 0;
	    break;

	case KF3:		/* overwrite contents with keep_savefile */
	    i = 0;
	    gotoxy(xs, ys);
	    s = ststart;
	    cputs(seqstr(strlen(s), ' '));
	    ustrncpy(s, keep_savefile, (size_t) maxlen - 1);
	    gotoxy(xs, ys);
	    cputs(s);
	    gotoxy(xs, ys);
	    break;

	case KLFT:
	case KKP4:	/* Keypad Left */
	    if (i)
	    {
		i--;
		s--;
		gotoxy(xs + find_pos(ststart, i), ys);
	    }
	    break;

	case KRGT:
	case KKP6:	/* Keypad Right */
	    if ((size_t) i < strlen(ststart))
	    {
		i++;
		s++;
		gotoxy(xs + find_pos(ststart, i), ys);
	    }
	    break;

	case CKLFT:
	case CKKP4:	/* Keypad Ctrl-Left */
	    if (i)
	    {
		i = wordleft(ststart, i);
		s = ststart + i;
		gotoxy(xs + find_pos(ststart, i), ys);
	    }
	    break;

	case CKRGT:
	case CKKP6:	/* Ketpad Ctrl-Right */
	    if ((size_t) i < strlen(ststart))
	    {
		i = wordright(ststart, i);
		s = ststart + i;
		gotoxy(xs + find_pos(ststart, i), ys);
	    }
	    break;

	case KEND:
	case KKP1:	/* Keypad End */
	    i = (int) strlen(ststart);
	    s = ststart + i;
	    gotoxy(xs + find_pos(ststart, i), ys);
	    break;

	case KHOME:
	case KKP7:	/* Keypad Home */
	    i = 0;
	    s = ststart;
	    gotoxy(xs, ys);
	    break;

	default:
	    if (ch > 31 && ch < 256)
	    {
		if (i >= maxlen)
		{
		    if (maxlen == display.screenwidth)
		    {
			/* when sending a message auto wrap
			   on display.screenwidth chars. Pass the character
			   entered back to send.c so it can
			   add it into the buffer */
			*s = '\0';
			putch('\r');
			retchar = (char) ch;
			exitloop = TRUE;
			break;
		    }
		    beep();
		    break;
		}
		l = strlen(ststart);
		/* If in Insert mode, insert a space for the new character */
		if (insertmode && (size_t) i < l && l < (size_t) maxlen)
		{
		    memmove(ststart + i + 1, ststart + i, l - (size_t) i + 1);  /*OK*/
		    *s = ' ';
		    n = wherex();
		    if (maxlen == display.screenwidth)
			cputs("\r");
		    else
			gotoxy(xs, ys);
		    cputs(ststart);
		    gotoxy(n, ys);
		}
		else if ((size_t) i == l)
		    *(s + 1) = '\0';	/* keep the string terminated */
		else if (*s == '\t')
		{
		    /* overwriting a tab so treat differently */
		    *s++ = (unsigned char) ch;
		    i++;
		    if (maxlen == display.screenwidth)
			cputs("\r");
		    else
			gotoxy(xs, ys);
		    cputs(ststart);
		    gotoxy(xs + find_pos(ststart, i), ys);
		    break;
		}
		*s++ = (unsigned char) ch;
		i++;
		putch((unsigned char) ch);
	    }
	    else
	    {
		/*
		  printf("\nGot %d\n", (int) ch);
		  */
		beep();
	    }
	    break;
	}
    }

    normcursor();		/* redisplay normal cursor */
    return retchar;
}


/*
 * drawbox()
 *
 * Draw a box using current attribute: the box runs from (x1, y1) to (x2, y2)
 * inclusive (the old version used to stop 1 short in the x direction, even
 * though it did the full y extent)
 */
static void drawbox(int x1, int y1, int x2, int y2)
{
    char border[6] = { 218, 191, 192, 217, 196, 179};
    int i;

    gotoxy(x1, y1);
    cprintf("%c%s%c", border[0], seqstr((size_t) (x2 - x1 - 1), border[4]),
	    border[1]);
    for (i = y1 + 1; i < y2; i++)
    {
	gotoxy(x1, i);
	cprintf("%c%s%c", border[5], seqstr((size_t) (x2 - x1 - 1), ' '),
		border[5]);
    }
    gotoxy(x1, y2);
    cprintf("%c%s%c", border[2], seqstr((size_t) (x2 - x1 - 1), border[4]),
	    border[3]);
}


/*
 * vdispwin()
 *
 * Display the text given by the varargs <args> in a box of appropriate size,
 * starting the text itself at <x>, <y> -- the box will be centred in the
 * appropriate direction if either of these is <= 0.  The list of strings to
 * display is ended by a NULL pointer; the cursor will be left at the end of
 * the last row when we return.  <title>, if non-NULL, is displayed in the top
 * of the surrounding box within sqaure brackets "[ title ]".  The text is
 * drawn using attribute <winattr>, and the description of the box, the
 * previous text state and the underlying text are saved to the savebox
 * structure <sbp>.  If <squeeze_blanks> is true, consecutive blank lines in
 * the input get squeezed down to a single blank.  Return the maximum length
 * of string that can be displayed in this box; <maxlen> is the user's
 * required minimum length on the end of the final line (may be 0).
 *
 * If a string begins with formfeed (^L), we centre that line in the box if we
 * can (ignoring the leading formfeed).
 */
static int vdispwin(int x, int y, int winattr, struct savebox *sbp, int maxlen,
		    const char *title, int nextra, BOOL squeeze_blanks,
		    va_list args)
{
    va_list ap;
    const char *ptr;
    int i, rows, maxrows, spare, l, maxtextlen;
    BOOL prev_blank;

    maxrows = (int) display.screenheight - 2;
    if (y > 2)			/* y-start below row 2 */
	maxrows -= (y - 2);

    /*
     * Run through arguments, counting them and noting max length; seed the
     * maximum length to be the larger of the room needed for the title and
     * the user's required <maxlen>
     */
    maxtextlen = (title) ? ((int) strlen(title) + 5) : 0;
    if (maxlen > maxtextlen)		/* user's required maxlen dominates */
	maxtextlen = maxlen;
    ap = args;
    prev_blank = FALSE;
    l = 0;
    for (i = nextra; i < maxrows && (ptr = va_arg(ap, const char *)) != NULL; )
    {
	if ((l = (int) strlen(ptr)) == 0 && squeeze_blanks)	/* blank */
	{
	    if (prev_blank)		/* last line was blank too */
		continue;	/* with the for (i) */
	    else
		prev_blank = TRUE;
	}
	else			/* a non-blank line */
	{
	    prev_blank = FALSE;
	    if (*ptr == '\f')		/* starts with formfeed */
		l--;			/* reduce length to skip FF */
	    if (l > maxtextlen)
		maxtextlen = l;
	}
	i++;
    }
    rows = i;		/* includes <nextra> blank rows on the end */

    /*
     * Check length of last line, with <maxlen> on the end; <l> will be the
     * length of the last line we display
     */
    if ((l += maxlen) > maxtextlen)
	maxtextlen = l;

    if (x < 1)		/* centre the screen horizontally */
    {
	int maxcols = (int) display.screenwidth - 2;
	spare = (maxtextlen < maxcols) ? (maxcols - maxtextlen) : 0;
	sbp->winx1 = 1 + spare / 2;
	sbp->winx2 = sbp->winx1 - 1 + (int) display.screenwidth - spare;
	x = sbp->winx1 + 1;
    }
    else		/* use the user's specified x */
    {
	sbp->winx1 = (x > 1) ? (x - 1) : 1;
	sbp->winx2 = sbp->winx1 + 1 + maxtextlen;
	if (sbp->winx2 > (int) display.screenwidth)
	    sbp->winx2 = (int) display.screenwidth;
    }
    if (y < 1)		/* centre the screen vertically */
    {
	spare = (rows < maxrows) ? (maxrows - rows) : 0;
	sbp->winy1 = 1 + spare / 2;
	sbp->winy2 = sbp->winy1 - 1 + (int) display.screenheight - spare;
	y = sbp->winy1 + 1;
    }
    else		/* use the user's specified y */
    {
	sbp->winy1 = (y > 1) ? (y - 1) : 1;
	sbp->winy2 = sbp->winy1 + 1 + rows;
	if (sbp->winy2 > (int) display.screenheight)
	    sbp->winy2 = (int) display.screenheight;
    }
    maxlen = sbp->winx2 - sbp->winx1 - 1;

    /*
     * Find the size of buffer needed to save the underlying box and try to
     * allocate it; if we succeed, save the underlying text to the buffer.
     * Also save the current text state, so it can be restored later.
     *
     * (We make the memory big enough for 1 extra column and 1 extra row,
     * because I'm not entirely sure I trust the library to get it right, and
     * this is *not* the place to have a buffer overrun.)
     */
    l = (sbp->winx2 - sbp->winx1 + 2) * (sbp->winy2 - sbp->winy1 + 2) * 2;
    if ((sbp->screenbuff = malloc((size_t) l)) != NULL)
	gettext(sbp->winx1, sbp->winy1, sbp->winx2, sbp->winy2,
		sbp->screenbuff);
    gettextinfo(&sbp->ti);

    /*
     * Set the full-screen window and the user's specified attribute, then
     * display the box, title and text
     */
    window(1, 1, display.screenwidth, display.screenheight);
    textattr(winattr);
    drawbox(sbp->winx1, sbp->winy1, sbp->winx2, sbp->winy2);
    if (title && *title)
    {
	gotoxy(x + 1, y - 1);
	cprintf("[ %.*s ]", maxlen - 5, title);
    }
    gotoxy(x, y);
    ap = args;
    prev_blank = FALSE;
    for (i = nextra; i < rows && (ptr = va_arg(ap, const char *)) != NULL; )
    {
	if (!*ptr && squeeze_blanks)		/* a blank line */
	{
	    if (prev_blank)		/* last line was blank too */
		continue;	/* with the for (i) -- ignore this line */
	    else
		prev_blank = TRUE;
	}
	else				/* a non-blank line */
	    prev_blank = FALSE;
	if (*ptr == '\f')	/* starts with FF: centre the line */
	{
	    int spc = (maxlen - (int) strlen(++ptr)) / 2;
	    if (spc < 0)
		spc = 0;
	    gotoxy(x + spc, y++);
	}
	else			/* line starts at left edge (x) */
	    gotoxy(x, y++);
	cprintf("%.*s", maxlen, ptr);
	i++;
    }
    /* cursor is now at end of last row */
    return maxlen;
}


/*
 * restwin()
 *
 * Restore the window settings and saved underlying text described by <sbp>
 */
static void restwin(struct savebox *sbp)
{
    textattr(sbp->ti.attribute);
    window(sbp->ti.winleft, sbp->ti.wintop,
	   sbp->ti.winright, sbp->ti.winbottom);
    if (sbp->screenbuff != NULL)
    {
	puttext(sbp->winx1, sbp->winy1, sbp->winx2, sbp->winy2,
		sbp->screenbuff);
	free(sbp->screenbuff);
	sbp->screenbuff = NULL;
    }
    gotoxy(sbp->ti.curx, sbp->ti.cury);
}


/*
 * getwin()
 *
 * Build a window with frame around, using attribute <winattr>, with text
 * starting at <x>, <y>, and prompt title <prompt>, displaying the varargs
 * lines of text which follow; these are ended by a NULL pointer.  The last
 * vararg string argument is the "prompt" line; vdispwin() will leave the
 * cursor at the end of this line on exit, so the cursor is in the right place
 * for maxgets() to display the default <input> string and get an input from
 * the user; <maxgets> is the maximum length of this input string *after* any
 * "prompt" text that vdispwin displays at the start of the line; if the box
 * gets too wide, it may have to be reduced before calling maxgets().  Once
 * maxgets() has returned, we close the window and restore the screen.  We
 * return the length of the final input string.
 */
int getwin(int x, int y, int maxlen, int winattr, char *input,
	   const char *prompt, ...)
{
    va_list argptr;
    struct savebox sb;
    int maxboxlen;

    va_start(argptr, prompt);
    maxboxlen = vdispwin(x, y, winattr, &sb, maxlen + 1, prompt, 1, TRUE,
			 argptr);
    va_end(argptr);
    if (maxboxlen < maxlen)		/* reduce maxlen to fit in the box */
	maxlen = maxboxlen;
    maxgets(input, maxlen);	/* displays <input> at current cursor pos */
    restwin(&sb);
    if (*input == ' ')
	return 0;
    return (int) strlen(input);
}


/*
 * dispwin()
 *
 * Display a boxed window, using attribute <winattr>, with the varargs strings
 * specified (the list is ended with a NULL pointer) and the given <title>
 * displayed in the top row of the box; the top-left corner for the text is
 * <x>, <y>, with the box being centred in the appropriate direction if either
 * of these is <= 0.  Wait for a key to be pressed (if interactive) or for 3
 * seconds (if not), then restore the screen and return the key pressed.
 */
int dispwin(int x, int y, int winattr, const char *title, BOOL squeeze_blanks,
	    ...)
{
    va_list argptr;
    struct savebox sb;
    int key;

    va_start(argptr, squeeze_blanks);
    (void) vdispwin(x, y, winattr, &sb, 0, title, 0, squeeze_blanks, argptr);
    va_end(argptr);
    if (tty)			/* Interactive: wait for keypress */
    {
	/* ST-FIXME: why use getch()?  Ctrl-C doesn't work ... */
	if ((key = getch()) == 0)	/* extended key: get 2nd half */
	    key = 100 + getch();	/* use "correct" key code: ST-FIXME */
    }
    else			/* Non-interactive: wait for 3 secs */
    {
	sleep(3);
	key = 0;
    }
    restwin(&sb);
    return key;
}


/*
 * errorbox()
 *
 * Display a boxed window with the varargs strings specified (the list is
 * ended with a NULL pointer) and the given <title> displayed in the top row
 * of the box (defaults to "Error!" if null); the box is centred in the
 * screen, and uses global <col_hilite> as the attribute.  Prompt for a key to
 * be pressed and return when the user has cleared the box.  There can be a
 * maximum of ERR_LINES lines of (user) text; we will append a blank line and
 * "Press a key to continue" to the user's supplied lines if this is an
 * interactive run.
 */
void errorbox(const char *title, ...)
{
    const char *p, *lines[ERR_LINES + 2];	/* <lines> is [0] to [6] */
    va_list ap;
    int i;

    va_start(ap, title);
    for (i = 0; i < ERR_LINES && (p = va_arg(ap, const char *)) != NULL; i++)
	lines[i] = p;
    va_end(ap);
    if (tty)			/* Interactive: add "Press a key" message */
    {
	lines[i++] = "";
	lines[i++] = pressakey;
    }
    for (; i < ERR_LINES + 2; i++)
	lines[i] = NULL;
    (void) dispwin(0, 0, col_hilite, (title) ? title : "Error!", FALSE,
		   lines[0], lines[1], lines[2], lines[3], lines[4],
		   lines[5], lines[6]);			/*lint !e771*/	/*OK*/
}


/*
 * scroll a given window up or down,
 * value of nolines: negative for scrolldown, positive for scrollup,
 * zero to clear window.
 */
void winscrl(int nolines)
{
    int scrlup;
    struct text_info ti;
    union REGS REG;

    scrlup = !(nolines < 0);
    nolines = scrlup ? nolines : -nolines;
    gettextinfo(&ti);		/* get current window info */
    REG.h.bl = 0;
    REG.h.bh = ti.attribute;
    REG.h.ch = ti.wintop - 1;		/* turbo-c counts from 1 up, */
    REG.h.cl = ti.winleft - 1;		/* bios from 0 ! */
    REG.h.dh = ti.winbottom - 1;
    REG.h.dl = ti.winright - 1;
    do
    {
	REG.h.ah = (scrlup) ? 6 : 7;
	REG.h.al = (nolines == 0) ? 0 : 1;	/* no to scroll */
	int86(0x10, &REG, &REG);		/* execute intr */
	nolines--;
    }
    while (nolines > 0);
}
