/*
  This source is part of PCOak, an electronic mailer for DOS based on PCElm.

  PCElm is Copyright (c) 1988-1993 Martin Freiss and Wolfgang Siebeck
           Copyright (c) 1992-1999 Demon Internet
  PCOak is Copyright (c) 2000 Simon Turner, Pete Disdale and dispc members

  Thanks to an agreement between the original PCElm authors and Demon Internet
  made in late 1999:

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License, version 1, as
	published by the Free Software Foundation.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	See the file COPYING, which contains a copy of the GNU General
	Public License.
*/

/*
 * miscutil.c -- misc. utility functions
 *
 * Useful C Library string searching functions:
 *   strchr	Find the first instance of a given character
 *   strrchr	Find the last instance of a given character
 *   strspn	Find the first instance of a character NOT from the set
 *		    (returns length of leading segment from set)
 *   strcspn	Find the first instance of a character from the set
 *		    (returns length of leading segment NOT from set)
 *   strpbrk	Find the first instance of a character from the set
 *
 * Useful local string searching functions:
 *   cstail	Find the start of a trailing segment from the set
 *   nstail	Find the start of a trailing segment NOT from the set
 *   csskip	Find the first char after a leading segment from the set
 *   nsskip	Find the first char after a leading segment NOT from the set
 *   stristr	Case-insensitive (not so quick) version of strstr()
 *
 * Like strchr() et al, our functions return a non-const pointer, even though
 * we're given a const input string, so that they can be called for both const
 * and non-const strings; for const strings, we would expect the result to be
 * assigned to a const pointer anyway.
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>

//const char chs_eol[]  = "\r\n";		/* End-of-line chars */
const char chs_wsp[]  = " \t\r\n";	/* Whitespace chars */
const char chs_lwsp[] = " \t";		/* Linear-whitespace chars */
const char chs_path[] = ":\\/";		/* Path delimiter chars */
const char chs_sep[]  = ", \t";		/* Address separator chars */


/*
 * cstail()
 *
 * Return a pointer to the start of any trailing segment of string <s> in
 * which all characters are from the set <set>; if there is such a segment,
 * this will be first character of that segment; if not, it will be the
 * string's terminating null.
 */
char *cstail(const char *s, const char *set)
{
    register const char *eos;
    register const char *ss;

    for (eos = s + strlen(s) - 1; eos >= s; eos--)
    {
	for (ss = set; *ss; ss++)
	    if (*eos == *ss)
		goto loop_end;		/* Sorry; it's *much* more efficient */
	return (char *) eos + 1;
    loop_end:
    }
    return (char *) s;
}


/*
 * nstail()
 *
 * Return a pointer to the start of any trailing segment of string <s> in
 * which all characters are not from the set <set>; if there is such a
 * segment, this will be first character of that segment; if not, it will be
 * the string's terminating null.
 */
char *nstail(const char *s, const char *set)
{
    register const char *eos;
    register const char *ss;

    for (eos = s + strlen(s) - 1; eos >= s; eos--)
	for (ss = set; *ss; ss++)
	    if (*eos == *ss)	/* <eos> is from the set: stop here */
		return (char *) eos + 1;
    return (char *) s;
}


/*
 * csskip()
 *
 * Skip over leading characters in <s> which are all from <set>; return a
 * pointer to the first character in <s> which is not from <set>.  If all
 * characters are from this set, the result will point to the string's
 * terminating null.
 *
 * This is a function, rather than a macro, to prevent unwanted side-effects
 * of using the string <s> twice in the same expression.
 */
char *csskip(const char *s, const char *set)
{
    register const char *ss;

    for (; *s; s++)
    {
	for (ss = set; *ss; ss++)
	    if (*s == *ss)
		goto loop_end;		/* Sorry; it's *much* more efficient */
	return (char *) s;
    loop_end:
    }
    return (char *) s;		/* This will be the terminating null */
}


/*
 * nsskip()
 *
 * Skip over leading characters in <s> which are not from <set>; return a
 * pointer to the first character in <s> which is from <set>.  If no
 * characters are from this set, the result will point to the string's
 * terminating null.
 *
 * This is a function, rather than a macro, to prevent unwanted side-effects
 * of using the string <s> twice in the same expression.
 */
char *nsskip(const char *s, const char *set)
{
    register const char *ss;

    for (; *s; s++)
	for (ss = set; *ss; ss++)
	    if (*s == *ss)	/* <s> is from the set: stop here */
		return (char *) s;
    return (char *) s;		/* This will be the terminating null */
}


/*
 * stristr()
 *
 * Search for the first instance of string <s2> within string <s1>, case
 * insensitively.  This is not going to be as quick as strstr(), but it's much
 * more useful (and we don't use it all that often, so the speed hit won't be
 * the end of the world as we know it).
 *
 * No doubt it could be sped up by using a better algorithm, but that can come
 * later.
 */
char *stristr(const char *s1, const char *s2)
{
    char c;
    const char *es1;
    size_t l;

    if (!*s2)
	return (char *) s1;
    c = (char) tolower(*s2);
    es1 = s1 + strlen(s1) + 1 - (l = strlen(s2));
    for (; s1 < es1; s1++)
	if (tolower(*s1) == c)		/* might be a match: try it */
	    if (strnicmp(s1, s2, l) == 0)	/* yes! */
		return (char *) s1;
    return NULL;
}


/*
 * getstring()
 *
 * Extract a string from <src> and copy it into <dst> (which has size
 * <dstsize>); this is either a simple string delimited by whitespace, or it
 * is delimited by quotes "...".  If "equals" is true, we expect the first
 * non-whitespace character in <line> to be '=', which will be followed (after
 * more optional whitespace) by the string we're trying to find.  Return the
 * length of string that we find and copy into <dst>, which may be 0; if we
 * have problems (e.g. bad format), return < 0.
 */
int getstring(char *dst, size_t dstsize, const char *src, int equals)
{
    char *end = dst + dstsize - 1;	/* ptr to last byte in <dst> */
    char *dp;

    *dst = '\0';
    src = csskip(src, chs_lwsp);	/* skip leading whitespace */
    if (equals)
    {
	if (*src != '=')		/* bad format */
	    return -1;
	src = csskip(src + 1, chs_lwsp);    /* skip the '=' & further LWSP */
    }
    if (*src == '\"')		/* a quoted-string */
	for (dp = dst, src++; dp < end && *src && *src != '\"'; dp++, src++)
	    *dp = *src;
    else			/* delimited by whitespace */
	for (dp = dst; dp < end && *src && !isspace(*src); dp++, src++)
	    *dp = *src;
    *dp = '\0';		/* ensure <dst> is terminated */
    return (int) (dp - dst);
}


#if 0			/* not used at present */
#include <stdarg.h>

/*
 * fmtstr()
 *
 * Formats the printf()-stlye arguments into the given buffer <buf>, or a
 * local static buffer if <buf> is NULL, and returns a pointer to the buffer.
 */
static char *fmtstr(char *buf, const char *fmt, ...)
{
    static char fmtbuf[1024];	/* static local buffer, if <buf> is NULL */
    va_list ap;

    if (buf == NULL)
	buf = fmtbuf;
    va_start(ap, fmt);
    vsprintf(buf, fmt, ap);
    va_end(ap);
    return buf;
}
#endif /* 0 */
