/*
  This source is part of PCOak, an electronic mailer for DOS based on PCElm.

  PCElm is Copyright (c) 1988-1993 Martin Freiss and Wolfgang Siebeck
           Copyright (c) 1992-1999 Demon Internet
  PCOak is Copyright (c) 2000-2002 Simon Turner, Pete Disdale and dispc members

  Thanks to an agreement between the original PCElm authors and Demon Internet
  made in late 1999:

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License, version 1, as
	published by the Free Software Foundation.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	See the file COPYING, which contains a copy of the GNU General
	Public License.
*/

/*
 * Small Editor for linking into various projects
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <alloc.h>
#include <mem.h>
#include <conio.h>
#include <ctype.h>
#include <dos.h>
#include "pcoak.h"
#include "notepad.h"
#include "keyboard.h"
#include "display.h"

#define ED_MAXLINE		160	/* Max. line length */
#define ED_SCRNWIDTH		(display.screenwidth - 2)
#define ED_LINESONSCREEN	(display.screenheight - 4)
#define ED_DISPLAYLINES		(display.screenheight)
#define ED_DISPLAYCOLS		(display.screenwidth)
#define ED_MAXSCRNWIDTH		132

typedef unsigned char LINE[ED_MAXLINE + 1];
typedef unsigned char STR64[65];
typedef unsigned char BYTE;
typedef struct linerec {
    struct linerec *last;
    LINE data;
    struct linerec *next;
} LINEREC, *LPTR;
typedef unsigned char SCREENLINE[ED_MAXSCRNWIDTH + 1];

/*
 * Global variables local to this module
 */
static SCREENLINE lword, blankline;
static STR64 find;
static unsigned int code;
static BOOL filefound, finished, changed, aborted;
static FILE *workfile;
static LPTR ln, nextln, firstln;
static int i, j,	/* cursor position, i == line, j == col */
			/* j (euggghh!) appears to be indexed from 1 */
    nlines,		/* length of file */
    top,		/* first line on screen */
    k, n, offset;
static char *ed_title;
static BOOL command(int key);

#define HELPLINES	20

static const char *help_text[HELPLINES] = {
"    This editor uses the following commands:",
"",
"    Cursor keys like left/right arrow, up/down arrow",
"    Home, End, PgUp and PgDn work as expected.",
"    Ins and Del dto.",
"",
"    Special commands are:",
"    \315\315\315\315\315\315\315\315\315\315\315\315\315\315\315\315\315\315\315\315\315",
"    \2Alt-D\2         \2D\2elete current line",
"    \2Alt-F\2         \2F\2ind text in buffer",
"    \2Alt-L\2         Set \2l\2eft margin",
"    \2Alt-N\2         Insert a \2n\2ew line",
"    \2Alt-R\2         Set \2r\2ight margin",
"    \2Alt-W\2         Toggle word \2w\2rap on/off",
"    \2Alt-Q\2         \2Q\2uit with save",
"    \2ESC\2           Quit with save",
"    \2Alt-X\2         E\2x\2it without save",
"    \2Alt-T\2         Set \2t\2ab width",
"",
"",
};

/*
 * Global string constants (defined in str.c) used here
 */
extern const char no_mem[];		/* "Not enough memory" */
extern const char cant_savescr[];	/* "* * * Can't save screen\r\n" */

/*
 * Functions defined elsewhere and used in this module
 */
void beep(void);
int getwin(int x, int y, int maxlen, int winattr, char *input,
	   const char *prompt, ...);
void get_cr(const char *msg);
char rip(char *s);
void writef(int x, int y, int attr, const char *line);
char *seqstr(size_t n, char ch);
void errordsp(const char *txt);
unsigned int getkey(void);
void write_error(const char *filename, const char *descrip);


/* instr() moved here from instr.c; it's now only used here -ST */
/*
 * instr(pos, string, target)
 * Beginning at offset pos, in string, find target.
 * returns starting position in string of target if found, or -1 if not found
 * <?> can be used as wild card in target
 * This routine is not case sensitive.
 */
static int instr(int p, char *s, char *t)
{
    int ii, jj, kk, ll;
    char up[256];
    register char *a, *b;

    ll = (int) strlen(s + p) + p;	/* So we don't wild card beyond EOS */
    strcpy(up, s);			/* ST-FIXME */
    strupr(up);			/* convert both to upper case for compare */
    strupr(t);				/* That's not very nice! -ST */

    for (ii = p; up[ii]; ii++)	/* Loop thru string.  */
    {
        a = &up[ii];
        kk = 0;				/* Reset target to beginning.  */
        for (jj = ii, b = &t[kk]; *b; a++, b++, jj++)  /* Loop thru target. */
	{
            while (*b == '?' && jj <= ll)
                a++, b++;			/* Skip if ? in target. */
            if (*a != *b)		/* Check for char match.  */
                break;			/* Incr thru string if not. */
        }
        if (*b == '\0')		/* Reached end of target without a mismatch.*/
            return ii;		/* Return offset where match started.  */
    }
    return -1;			/* Reached end of string without a match.  */
}


/* ST-FIXME : this is identical to bmutil::cmpspace() !?!? */
/* Renamed from cmpspace() to prevent name clash */
static int cmpspace2(char *s)
{
    char *p = s;

    while (*p)
	if (*p++ != ' ')
	    return TRUE;
    return FALSE;
}


static int line_blank(LPTR temp_ln)
{
    if (temp_ln == NULL || cmpspace2(temp_ln->data) == 0 ||
	strcmp(temp_ln->data, "-- ") == 0 ||
	strncmp(temp_ln->data, ">", 1) == 0)
	return TRUE;
    return FALSE;
}


static void deleteln(LPTR ln_ptr)		/* delete a line */
{
    LPTR new;

    if (nlines < 2)	/* This is another workaround the empty file */
	return;
    /* delete the line; change firstln if the line to kill is the 1st */
    new = ln_ptr->next;
    if (ln_ptr == firstln)
	firstln = new;
    new->last = ln_ptr->last;
    ln_ptr->last->next = new;
    free(ln_ptr);
    ln_ptr = new;
    nlines--;
    changed = TRUE;
}


static int shuffle_up(LPTR start_ln)
{
    int gap, f = FALSE, nn;
    LPTR temp_ln = ln;
    SCREENLINE more;
    char *p;

    while (!line_blank(temp_ln->next))
    {
	if (temp_ln->data[strlen(temp_ln->data) - 1] == ' ')
	    nn = (int) strlen(temp_ln->data) - 1;
	else
	    nn = (int) strlen(temp_ln->data);
	gap = npsetup.edit_rera - nn;
	if (gap < 2)
	    break;
	strcpy(more, temp_ln->next->data);
	p = more;
#if 0
	/* I think this is a mistake.  In any case it causes a bug further
	   on as we are moving strlen(more) out of the source string and
	   not allowing for any stripping done here */
	while (*p && *p == ' ')
	    p++;	/* remove leading spaces */
	if (p != more)
	    memmove(more, p, strlen(p) + 1);
#endif
	/* data to be move is in more.  Now see how much of it can be moved */
	while (TRUE)
	{
	    p = strrchr(more, ' ');
	    if (p)
	    {
		*p = '\0';
		nn = (int) (p - more);
	    }
	    else
		nn = (int) strlen(more);
	    if (nn < gap)
		break;
	    if (p == NULL)
	    {
		/* can't move anything */
		nn = -1;
		break;
	    }
	}
	if (nn == -1)
	    break;
	f = TRUE;
	if (temp_ln->data[strlen(temp_ln->data) - 1] != ' ')
	    strcat(temp_ln->data, " ");
	strcat(temp_ln->data, more);
	temp_ln = temp_ln->next;
	memmove(temp_ln->data, temp_ln->data + strlen(more),
		strlen(temp_ln->data) - strlen(more) + 1);
	p = temp_ln->data;
	while (*p && *p == ' ')
	    p++;	/* remove leading spaces */
	if (p != temp_ln->data)
	    memmove(temp_ln->data, p, strlen(p) + 1);
	if (strlen(temp_ln->data) == 0)
	{
	    if (temp_ln != start_ln)
		deleteln(temp_ln);
	    break;
	}
    }
    return f;
}


/*
 * Insert a line full of data after another into linked list
 */
static int insafter(LINE *data, LPTR se)
{
    LPTR newse;

    newse = (LPTR) malloc(sizeof(LINEREC));
    if (!newse)
    {
	errordsp(no_mem);
	return FALSE;
    }
    newse->last = se;
    newse->next = se->next;
    se->next = newse;
    newse->next->last = newse;
    memcpy(newse->data, (char *) data, sizeof(LINE));
    return TRUE;
}


/*
 * Insert a character within a string
 *
 * i       ch              Char to be inserted
 * i       st              Location at which to insert it
 * io      str             String in which to insert it
 */
static void insert(char ch, char str[], int st, int maxlen)
{
    memmove(&str[st + 1], &str[st], strlen(str) - (size_t) st + 1);
    str[maxlen] = '\0';
    str[st] = ch;
}


/*
 * Extinguishing a character from within a string (thanks JLW!)
 *
 * i       st              Location at which to delete
 * io      str             String in which to delete
 */
static void delete(char str[], int st)
{
    int l = (int) strlen(str) - 1;

    if (st <= l)
    {
	memmove(&str[st], &str[st + 1], strlen(str) - (size_t) st + 1);
	str[l] = '\0';
    }
}


/*
 * Set up first line of circular buffer
 */
static void set_firstln(void)
{
    ln = malloc(sizeof(LINEREC));
    if (ln == NULL)
	return;
    memset(ln, 0, sizeof(LINEREC));
    firstln = ln;
    firstln->next = firstln;
    firstln->last = firstln;
}


/*
 * read a file into buffer
 */
static void readfile(const char *fname, BOOL *abortedp)
{
    BOOL ovflw;
    unsigned long maxlines;
    LINE inputline;

    *abortedp = FALSE;

    if ((workfile = fopen(fname, "r")) != NULL)
    {
	ovflw = FALSE;
	maxlines = coreleft() / sizeof(LINEREC);
	nlines = 0;
	writef(1, 1, col_normal, blankline);
	while (fgets(inputline, ED_MAXLINE - 1, workfile) != NULL)
	{
	    if (ovflw)
		break;
	    (void) rip(inputline);
	    if (strlen(inputline) >= ED_MAXLINE)
	    {
		ovflw = TRUE;
		errordsp(no_mem);
	    }
	    else
	    {
		if (firstln == NULL)
		{
		    set_firstln();
		    strcpy((char *) firstln->data, inputline); /* copy data */
		}
		else
		    (void) insafter((LINE *) inputline, firstln->last);
		if (++nlines > maxlines)    /*lint !e574 !e737*/ /* ST-FIXME */
		{
		    errordsp(no_mem);
		    ovflw = TRUE;
		}
	    }
	}	/* while fgets() */
	(void) fclose(workfile);			/* OK: read only */
	if (!ovflw)
	    filefound = TRUE;
	else
	{
	    set_firstln();
	    nlines = 1;
	}
    } /* if fopen != NULL */
    else
    {
	filefound = TRUE;
	nlines = 1;
	set_firstln();
    }
}


/*
 * Save current buffer to file
 */
static void writefile(const char *fname)
{
    FILE *outfile;
    BOOL err = FALSE;

    if ((outfile = fopen(fname, "w")) != NULL)
    {
	ln = firstln;
	do
	{
	    (void) fputs(ln->data, outfile);
	    if (fputs("\n", outfile) == EOF)
	    {
		err = TRUE;
		break;	/* out of the do ... while */
	    }
	    ln = ln->next;
	}
	while (ln != firstln);
	if (fclose(outfile) != 0)
	    err = TRUE;
	if (err)
	    write_error(fname, "saving edited file");
    }
}


/*
 * Show the ruler
 */
static void rulerline(void)
{
    BYTE c, jj;
    unsigned char textline[ED_MAXSCRNWIDTH + 1];

    for (jj = 0; jj < ED_DISPLAYCOLS - 2; jj++)
    {
	if (((jj + offset) % npsetup.edit_tabsize) == 0)
	    textline[jj] = '\xd1';
	else if (((jj + 1) % 5) == 0)
	    textline[jj] = '\xcf';
	else
	    textline[jj] = '\xcd';
	c = (BYTE) 48 + ((jj + 1 + offset) / 10) % 10;	/*lint !e732*//*ST-FIXME*/
	if (((jj + 1) % 10) == 0)
	    textline[jj] = (char) c;
	if (npsetup.wrap)
	{
	    if (npsetup.edit_lira >= offset)
		textline[npsetup.edit_lira - offset] = '\x10';
	    if (npsetup.edit_rera >= offset)
		textline[npsetup.edit_rera - offset] = '\x11';
	}
    }
    textline[ED_DISPLAYCOLS - 2] = '\0';
    writef(1, 2, col_hilite, textline);
}


/*
 * Show current editor status
 */
static void statusline(void)
{
    SCREENLINE textline;

    sprintf(textline, "%4d/%3d      %-*.*s",
	    i, j, ED_DISPLAYCOLS - 30,
	    ED_DISPLAYCOLS - 30, ed_title ? ed_title : "");
    writef(1, 1, col_normal, textline);
    if (offset > 1)
    {
	sprintf(textline, "\xae%3d\xae", offset - 1);
	writef(10, 1, col_normal, textline);
    }
    textline[0] = npsetup.wrap ? 'W' : ' ';
    textline[1] = npsetup.edit_insrt ? 'I' : ' ';
    textline[2] = changed ? '*' : ' ';
    textline[3] = '\0';
    writef(ED_DISPLAYCOLS - 4, 1, col_normal, textline);
}


/*
 * Write a line to screen
 */
static void writeline(BYTE row, BYTE attr)
{
    SCREENLINE contents;
    int len;

    memset(contents, 0x20, sizeof(contents));
    contents[ED_DISPLAYCOLS - 2] = '\0';
    len = (int) strlen(ln->data);
    if (len > offset - 1)
	sprintf(contents, "%-*.*s", ED_DISPLAYCOLS - 2, ED_DISPLAYCOLS - 2,
		&(ln->data[offset - 1]));
    writef(1, row, attr, contents);
    if (len == ED_DISPLAYCOLS - 2)
	writef(ED_DISPLAYCOLS - 2, row, attr, "+");
    else if (len > 0)
	writef(ED_DISPLAYCOLS - 2, row, attr, "<");

}


/*
 * rewrite the complete screen (lower 23 lines)
 */
static void screen(void)
{
    int row;
    LPTR topln;

    ln = firstln;
    if (top < 1)
	top = 1;
    if (top > 1)
	for (k = 1; k < top; k++)
	    ln = ln->next;
    topln = ln;
    for (row = 3; row < ED_DISPLAYLINES; row++)
    {
	writeline((BYTE) row, (BYTE) col_normal);
	ln = ln->next;
	if (ln == firstln)
	    break;
    }
    for (row++; row <= ED_DISPLAYLINES; row++)
    {
	gotoxy(1, row);
	clreol();
    }
    ln = topln;

    for (row = i - top; row > 0 && ln->next != firstln; row--, ln = ln->next)
	;
}


/*
 * Go up one page
 */
static void pageup(void)
{
    int kk;

    for (kk = 0; kk < ED_LINESONSCREEN - 1; kk++)
	if (i > 1)
	{
	    i--;
	    ln = ln->last;
	}
    screen();
}


/*
 * Go down one page
 */
static void pagedown(void)
{
    int kk;

    for (kk = 0; kk < ED_LINESONSCREEN; kk++)
	if (i < nlines)
	{
	    i++;
	    ln = ln->next;
	    if (i - top >= ED_LINESONSCREEN)
		top++;
	}
    screen();
}


/*
 * make sure the cursor is visible on screen
 */
static void updatecursor(void)
{
    BOOL shifted = FALSE;

    if (i < 1)
    {
	i = 1;
	ln = firstln;
    }
    if (i > nlines)
    {
	i = nlines;
	ln = firstln->last;
    }
    if (j < 1)
	j = 1;
    if (j > ED_MAXLINE)
	j = ED_MAXLINE;
    if (j > (offset + ED_SCRNWIDTH - 3))
    {
	offset = 10 * (j / 10) - 59;
	shifted = TRUE;
    }
    if (j < offset)
    {
	offset = 10 * ((j - 10) / 10) + 1;
	shifted = TRUE;
    }
    if (i < top)
    {
	top = i;
	shifted = TRUE;
    }

    if (i > (top + ED_LINESONSCREEN))
    {
	top = i - ED_LINESONSCREEN;
	shifted = TRUE;
    }
    if (shifted)
    {
	rulerline();
	screen();
    }
    gotoxy(j - offset + 1, i - top + 3);
}


/*
 * Insert one line after the current one
 */
static void insertln(char *contents)
{
    if (coreleft() / 16 < 50)
    {
	errordsp(no_mem);
	return;
    }
    (void) insafter((LINE *) contents, ln);
    nlines++;
    changed = TRUE;
}


/*
 * Split up the current line
 */
static void cutline(int force_move_to_next)
{
    SCREENLINE more;

    strcpy(more, lword);
    strcat(more, &(ln->data[j - 1]));
    ln->data[j - 1] = '\0';
    if (force_move_to_next || line_blank(ln->next))
    {
	insertln(more);
	ln = ln->next;
	j = npsetup.edit_lira;
    }
    else
    {
	if (more[strlen(more) - 1] != ' ')
	    strcat(more, " ");
	memmove(ln->next->data + strlen(more), ln->next->data,
		strlen(ln->next->data) + 1);
	strncpy(ln->next->data, more, strlen(more));	/* ST-FIXME */
    }
}


static int allow_wordwrap(void)
{
    LPTR ln_ptr;

    if (!npsetup.wrap)
	return FALSE;
    ln_ptr = firstln;
    if (strncmp(firstln->data, "Date: ", 6) == 0)
    {
	/* we have the headers in core so don't allow wordwrap if we are
	   in the headers */
	do
	{
	    if (strlen(ln_ptr->data) == 0)
		break;	/* end of headers */
	    if (ln_ptr == ln)
		return FALSE;	/* we *are* in the headers */
	    ln_ptr = ln_ptr->next;
	}
	while (ln_ptr != firstln);
    }
    /* check to see if we are in the signature.  Back up from the current
       line until we hit top or we hit the "-- " line. */
    ln_ptr = ln;
    /* note we have to a different scan here as we may well be starting
       on the very first line */
    while (TRUE)
    {
	if (strcmp(ln_ptr->data, "-- ") == 0)
	    return FALSE;	/* we are in the .sig */
	if (ln_ptr == firstln)
	    break;
	ln_ptr = ln_ptr->last;
    }
    return TRUE;
}


/*
 * Wordwrap at end of line
 */
static void wordwrap(void)
{
    int remember_j = j, temp_j = j;
    LPTR remember_ln = NULL, temp_ln = ln;

    if (!allow_wordwrap())
	return;
    while (TRUE)
    {
	j = (int) strlen(temp_ln->data);
	while (j > 1)	/* move past and remove trailing spaces */
	{
	    if (temp_ln->data[j - 1] != ' ')
		break;
	    j--;
	}
	temp_ln->data[j] = '\0';
	while (j > 1)
	{
	    j--;
	    if (temp_ln->data[j - 1] == ' ')
		if (j < npsetup.edit_rera + 1)
		    break;
		else
		    while (j > 1)	/* move past double spaces */
		    {
			j--;
			if (temp_ln->data[j - 1] != ' ')
			    break;
		    }
	}
	if (j == 1)
	{
	    /* no space character found */
	    (void) command(KCR);
	}
	else
	{
	    j++;
	    n = temp_j - j;
	    ln = temp_ln;
	    cutline(FALSE);
	    if (remember_ln == NULL)
	    {
		remember_ln = ln;
		if (ln == temp_ln && j < temp_j)
		    remember_ln = ln = ln->next;
		if (ln != temp_ln)
		{
		    i++;
		    j = npsetup.edit_lira + n;
		    remember_j = j;
		}
	    }
	}
	temp_ln = temp_ln->next;
	if (line_blank(temp_ln) ||
	    strlen(temp_ln->data) <= (size_t) npsetup.edit_rera + 1)
	    break;
    }
    ln = remember_ln;
    j = remember_j;
    if (ln == NULL)
    {
	printf("\nERROR ln IS NULL!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!");
	sleep(10);
    }
    screen();
}


/*
 * put current line on top of previous line
 */
static void stackline(void)
{
    LPTR oldln;

    if (ln == firstln)
	return;
    oldln = ln;
    ln = ln->last;
    j = (int) strlen(ln->data) + 1;
    strcat(ln->data, oldln->data);
    ln->next = oldln->next;
    oldln->next->last = ln;
    free(oldln);
    i--;
    nlines--;
    screen();
}


static void addchar(unsigned char ch)
{
    static unsigned char chstr[2];
    int charsneeded, len;

    chstr[0] = ch;
    changed = TRUE;
    len = (int) strlen(ln->data);
    charsneeded = j - len;
    if (charsneeded > 1)
    {
	strcat(ln->data, seqstr((size_t) charsneeded, ' '));
	len += charsneeded;
    }
    if (j == len + 1)
	strcat(ln->data, chstr);
    else
    {
	if (npsetup.edit_insrt)
	    insert(ch, ln->data, j - 1, ED_MAXLINE);
	else
	    ln->data[j - 1] = ch;
    }
    j++;
    writeline((BYTE) (i - top + 3), (BYTE) col_normal);
    if ((strlen(ln->data) > (unsigned) npsetup.edit_rera + 1))
	wordwrap();
}


/*
 * Set left margin
 */
static void setleftmargin(void)
{
    char buffer[5];
    int value;

    sprintf(buffer, "%d", npsetup.edit_lira);
    if (getwin(15, 17, 3, col_select, buffer, "Set left margin (< 40)",
	       NULL) > 0)
    {
	value = atoi(buffer);
	if ((value > 0) && (value < 40))
	    npsetup.edit_lira = value;
    }
}


/*
 * Set size for tab characters
 */
static void settabsize(void)
{
    char buffer[5];
    int value;

    sprintf(buffer, "%d", npsetup.edit_tabsize);
    if (getwin(15, 17, 3, col_select, buffer, "Set tab size (2 < tab < 20)",
	       NULL) > 0)
    {
	value = atoi(buffer);
	if ((value > 2) && (value < 20))
	    npsetup.edit_tabsize = value;
    }
}


/*
 * Set right margin
 */
static void setrightmargin(void)
{
    char buffer[5];
    int value;

    sprintf(buffer, "%d", npsetup.edit_rera);
    if (getwin(15, 17, 3, col_select, buffer, "Set right margin (> 30)",
	       NULL) > 0)
    {
	value = atoi(buffer);
	if ((value > 30) && (value < 160))
	    npsetup.edit_rera = value;
    }
}


/*
 * if file has been changed, ask for save
 */
static int asksave(void)
{
    char buffer[5];

    if (!npsetup.confirm_abort)
	return FALSE;
    strcpy(buffer, "No");
    if (getwin(15, 17, 3, col_select, buffer, "Exit without saving?",
	       NULL) > 0)
	return strnicmp(buffer, "yes", strlen(buffer));
    return TRUE;
}


/*
 * go one word to the right
 */
static void wordright(void)
{
    int kk, len;

    kk = j - 1;
    len = (int) strlen(ln->data);
    /* skip over current word */
    do
	kk++;
    while (!isspace(ln->data[kk]) && (kk < len));

    /* skip over blanks */
    if (kk < len)
    {
	do
	    kk++;
	while (isspace(ln->data[kk]) && (kk < len));
    }
    if (kk < len)
	j = kk + 1;
    else
	j = len + 1;
}


/*
 * go one word to the left
 */
static void wordleft(void)
{
    if ((size_t) j > strlen(ln->data))
	j = (int) strlen(ln->data);   /* in case we cursor down into a space */
    j--;
    /* skip over blanks */
    if (j > 0)
	while (isspace(ln->data[j - 1]) && (j > 0))
	    j--;

    /* skip over current word, if avl */
    while (!isspace(ln->data[j - 1]) && (j > 1))
	j--;

    if (j > 1 && isspace(ln->data[j - 1]))
	j++;
    if (j < 1)
	j = 1;
}


/*
 * provide some help on this editor
 */
static void EDHelp(void)
{
    char *saved_screen;
    int x, y, kk;	/* cursor position for screen save */

    x = wherex();
    y = wherey();
    /* get memory for screen */
    saved_screen = malloc(2 * ED_DISPLAYCOLS * ED_DISPLAYLINES);
    if (saved_screen != NULL)
    {
	if (!gettext(1, 1, ED_DISPLAYCOLS, ED_DISPLAYLINES, saved_screen))
	{
	    cputs(cant_savescr);
	    sleep(1);
	    free(saved_screen);
	    saved_screen = NULL;
	}
    }
    clrscr();
    for (kk = 0; kk < 14; kk++)
	writef(1, kk + 2, col_normal, help_text[kk]);
    if (strncmp(npsetup.key_end, "ESC", 3))
	writef(1, kk + 2, col_normal, help_text[14]);
    else
	writef(1, kk + 2, col_normal, help_text[15]);
    for (kk = 16; kk < HELPLINES; kk++)
	writef(1, kk + 1, col_normal, help_text[kk]);
    get_cr(NULL);
    if (saved_screen)
    {		/* now restore */
	puttext(1, 1, ED_DISPLAYCOLS, ED_DISPLAYLINES, saved_screen);
	free(saved_screen);
	saved_screen = NULL;
	gotoxy(x, y);
    }
}


/*
 * search for a string in text.
 *
 * Due to instr(), this routine is not case-sensitive, and a
 * '?' may be used as wildcard.
 */
static void do_search(void)
{
    int spos, oldline = i;
    LPTR nxtln;

    if (getwin(15, 17, sizeof(find) - 1, col_select, find, "Find text",	
	       NULL) > 0)
	for (nxtln = ln; i < nlines; nxtln = nxtln->next, i++)
	    if ((spos = instr(0, nxtln->data, find)) > 0)
	    {
		j = spos + 1;
		ln = nxtln;
		updatecursor();
		return;
	    }
    beep();
    i = oldline;
}


/*
 * interpret keycodes
 */
static BOOL command(int key)
{
    switch (key)
    {
    case KF1:
	EDHelp();
	break;

    case CKRGT:
    case CKKP6:
	wordright();
	break;

    case CKLFT:
    case CKKP4:
	wordleft();
	break;

    case KTAB:
	if (npsetup.edit_insrt)
	{
	    addchar(' ');
	    while ((j % npsetup.edit_tabsize) && (j < ED_MAXLINE))
		addchar(' ');
	}
	else
	{
	    do
		j++;
	    while ((j % npsetup.edit_tabsize) && (j < ED_MAXLINE));
	}
	break;

    case KHOME:
    case KKP7:
	if (j > npsetup.edit_lira)
	    j = npsetup.edit_lira;	/* move to start of current line */
	else
	{
	    /* move to start of document */
	    ln = firstln;
	    i = 1;
	    screen();
	}
	break;

    case KEND:
    case KKP1:
	j = (int) strlen(ln->data) + 1;
	break;

    case KRGT:
    case KKP6:
	j++;
	break;

    case KLFT:
    case KKP4:
	if (j > 1)
	    j--;
	break;

    case KUP:
    case KKP8:
	if (i > 1)
	{
	    i--;
	    ln = ln->last;
	}
	break;

    case KDN:
    case KKP2:
	if (i < nlines)
	{
	    i++;
	    ln = ln->next;
	    if (i - top >= ED_LINESONSCREEN)
	    {
		top++;
		screen();
	    }
	}
	break;

    case KBS:
	if (j == 1)
	{
	    if (i > 1)
		stackline();
	    break;
	}
	j--;
	/* Fall through */

    case KDEL:
    case KKPCOMMA:
	if (strlen(ln->data) == 0 && j == 1)
	{
	    deleteln(ln);
	    screen();
	    break;
	}
	delete(ln->data, j - 1);
	writeline((BYTE) (i - top + 3), (BYTE) col_normal);
	changed = TRUE;
	if (allow_wordwrap() && shuffle_up(ln))
	    screen();
	break;

    case KCR:
	if (npsetup.edit_insrt || i == nlines)
	{
	    if (i == nlines)
		(void) command(KEND);
	    if ((size_t) j > strlen(ln->data))
		j = (int) strlen(ln->data) + 1;
	    cutline(TRUE);
	    i++;
	    screen();
	    changed = TRUE;
	}
	else
	{
	    (void) command(KDN);
	    (void) command(KHOME);
	}
	break;

    case KPGUP:
    case KKP9:
	pageup();
	break;

    case KPGDN:
    case KKP3:
	pagedown();
	break;

    case CKPGUP:
    case CKKP9:
	i = top = 1;
	ln = firstln;
	screen();
	break;

    case CKPGDN:
    case CKKP3:
	while (i < nlines)
	    pagedown();
	break;

    case ALTW:
	npsetup.wrap = !npsetup.wrap;
	rulerline();
	statusline();
	break;

    case ALTN:				/* Alt-N insert blank line */
	ln = ln->last;
	insertln("");
	screen();
	break;

    case ALTD:				/* Alt-D delete line */
	deleteln(ln);
	screen();
	break;

    case ALTF:				/* Alt-F find string from cursor */
	do_search();
	break;

    case ALTL:				/* Alt-L set left margin */
	setleftmargin();
	rulerline();
	break;

    case ALTR:				/* Alt-R set right margin */
	setrightmargin();
	rulerline();
	break;

    case ALTT:				/* Alt-T set tabsize */
	settabsize();
	rulerline();
	break;

    case KINS:
    case KKP0:
	npsetup.edit_insrt = !npsetup.edit_insrt;
	if (npsetup.edit_insrt)
	    inscursor();
	else
	    ovrcursor();
	statusline();
	break;

    case ALTX:
	if (changed && asksave())
	    break;
	changed = FALSE;			/* Alt-X exit without save */
	finished = TRUE;
	break;

    case ALTQ:						/* Alt-Q */
	if (strncmp(npsetup.key_end, "ESC", 3))
	    finished = TRUE;
	break;

    case KESC:						/* ESC */
	if (strncmp(npsetup.key_end, "ESC", 3) == 0)
	    finished = TRUE;
	break;

    case ' ':	/* space */
	{
	    char *p = strchr(ln->data, ' ');

	    if (i > 1 && (p == NULL || ln->data + j < p))
	    {
		/* if we are not on the first line */
		/* on the first word and we have split it using a space char -
		   if wordwrapping see if we should wrap to the previous line */
		/* see if the previous line is blank or not */
		if (!line_blank(ln->last) && allow_wordwrap())
		{
		    LPTR temp_ln = ln;

		    addchar((unsigned char) code); /* add it to this line first */
		    /* now shuffle up (if we can) starting from previous line */
		    ln = ln->last;
		    if (shuffle_up(temp_ln))
		    {
			screen();
			/* j is going to be wrong here though */
			j = npsetup.edit_lira;
		    }
		    ln = temp_ln;
		    return TRUE;
		}
	    }
	    /* else fall through */
	}
    /* fall through */

    default:
	if (key > 31 && key < 256)	/* Valid range to be inserted */
	    return FALSE;
    }

    return TRUE;	/* key has been handled */
}


/*
 * main function
 */
void notepad(const char *fname, char *title)
{
    char t[ED_MAXSCRNWIDTH];

    strcpy(blankline, seqstr(ED_DISPLAYCOLS, ' '));
    if (strlen(title) > 60)
    {
	strncpy(t, title, 60);		/* ST-FIXME */
	strcpy(t + 60, "...");
	/*
	 * The following line assigns the address of automatic variable <t>
	 * to the global static variable <ed_title> -- this is potentially
	 * very nasty, but it *should* be OK in this case.  I'd rather find
	 * a better way, though.  ST-FIXME
	 */
	ed_title = t;				/*lint !e789 */
    }
    else if ((ed_title = title) != NULL)
	(void) rip(ed_title);
    strcpy(blankline, seqstr(display.screenwidth, ' '));
    filefound = aborted = FALSE;
    firstln = NULL;
    readfile(fname, &aborted);
    if (aborted)
    {
	free(ln);
	return;
    }
    if (filefound)
    {
	j = i = top = offset = 1;
	find[0] = lword[0] = '\0';
	finished = changed = FALSE;
	clrscr();
	statusline();
	rulerline();
	screen();
	if (npsetup.edit_insrt)
	    inscursor();		/* Set the "Insert" mode cursor */
	else
	    ovrcursor();		/* Set the "Overwrite" mode cursor */
	updatecursor();
	do
	{
	    code = getkey();
	    if (code == 0)
		continue;
	    if (!command((int) code))
		addchar((unsigned char) code);
	    statusline();
	    updatecursor();
	}
	while (!finished);
	if (changed)
	    writefile(fname);
    }	/* if (filefound) */

    /*
     *	free the whole bunch
     */
    ln = firstln->next;
    while (ln != firstln)
    {
	nextln = ln->next;
	free(ln);
	ln = nextln;
    }
    free(firstln);
    normcursor();			/* Restore the normal cursor */
}
