/*
  This source is part of PCOak, an electronic mailer for DOS based on PCElm.

  PCElm is Copyright (c) 1988-1993 Martin Freiss and Wolfgang Siebeck
           Copyright (c) 1992-1999 Demon Internet
  PCOak is Copyright (c) 2000-2003 Simon Turner, Pete Disdale and dispc members

  Thanks to an agreement between the original PCElm authors and Demon Internet
  made in late 1999:

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License, version 1, as
	published by the Free Software Foundation.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	See the file COPYING, which contains a copy of the GNU General
	Public License.
*/

#include <stdio.h>
#include <conio.h>
#include <string.h>
#include <mem.h>
#include <dir.h>
#include <time.h>
#include <stdlib.h>
#include <dos.h>
#include <io.h>
#include <ctype.h>
#include "pcoak.h"
#include "header.h"
#include "ustring.h"
#include "chars.h"
#include "macros.h"
#include "display.h"

/* RFC 2822 limits on header and body line lengths: SHOULD be obeyed */
#define MAX_HDRLINE	78	/* Max length of a header line */
#define MAX_BODYLINE	78	/* Max length of a body line */

/* Length of encoded base64 line & no. of bytes to encode for that length */
#define B64_LINELEN	76	/* No. of chars in an encoded base64 line */
#define B64_NBYTES	(B64_LINELEN / 4 * 3)	/* No. of bytes to encode */

/* No. of bytes to encode on a uuencode line (should be 45) */
#define UU_NBYTES	45	/* No. of bytes to uuencode per line */

/* File type checking stuff */
#define BINARY_SAMPLE	512L	/* Examine >= 512 bytes for binary */
#define BINARY_RATIO	8	/* >= 1 in 8 chars are 8-bit/ctrl for binary */

typedef enum filetype 		/* Type of a file */
{
    FT_ASCSHRT,		/* Pure ASCII text, short lines (<= MAX_BODYLINE) */
    FT_ASCLONG,		/* Pure ASCII text, long lines (> MAX_BODYLINE) */
    FT_ACZSHRT,		/* ASCII + ^Z text, short lines (<= MAX_BODYLINE) */
    FT_ACZLONG,		/* ASCII + ^Z text, long lines (> MAX_BODYLINE) */
    FT_TEXT,		/* Mostly text, some binary; recommend q-p encoding */
    FT_BINARY,		/* Binary; recommend base64 encoding */
    NUM_FILETYPES	/* >>>MUST<<< be last: count of FT_* enums */
} FILETYPE;

/*
 * Global variables local to this module
 */
static char header_file[PATHLEN];
static char send_file[PATHLEN];	/* for uuencoding etc. */
static char *reciprest = NULL;	/* hold *all* receivers inc. "Cc" and "Bcc" */
static char *expanded_to = NULL;    /* "To", after passing through alias */
static char *expanded_cc = NULL;    /* "Cc", after passing through alias */
static char *expanded_bcc = NULL;   /* "Bcc", after passing through alias */
static size_t reciprest_size = 0;	/* allocated length of <reciprest> */
static size_t expanded_to_size = 0;	/* alloc. length of <expanded_to> */
static size_t expanded_cc_size = 0;	/* alloc. length of <expanded_cc> */
static size_t expanded_bcc_size = 0;	/* alloc. length of <expanded_bcc> */

static const char quote_tmpfile[]  = "!QWERTY.$$$";
static const char fwd_prompt[]	   = "Forward to: ";
static const char mail_prompt[]	   = "Mail To: ";
static const char cc_prompt[]	   = "     Cc: ";
static const char subj_prompt[]	   = "Subject: ";
static const char not_sent[]	   = "Not sent";
static const char to_subj_fmt[]	   = "To: %s Subject: %s\n\n";
static const char no_subj_spec[]   = "No Subject specified";
static const char sending_fmt[]	   = "\r\nSending to: %-20s ";
static const char sig_werr[]       = "adding signature";
static const char mailtext_oerr[]  = "mailtext file";
static const char mailtext_oerr2[] = "mailtext file after edit";
static const char backup_oerr[]    = "backup file";
static const char closing_file[]   = "closing file";

static const char *ft_desc[NUM_FILETYPES] = {
    "pure ASCII",			/* FT_ASCSHRT */
    "pure ASCII with over-long lines",	/* FT_ASCLONG */
    "ASCII, ending with Ctrl-Z",	/* FT_ACZSHRT */
    "ASCII + Ctrl-Z, over-long lines",	/* FT_ACZLONG */
    "ASCII with some binary chars",	/* FT_TEXT */
    "binary",				/* FT_BINARY */
};

/*
 * External global variables used in this module
 */
extern const char no_mail[];		/* "No mail" */
extern const char no_mem[];		/* "Not enough memory" */
extern const char aborted[];		/* "Aborted" */
extern const char bracket_none[];	/* "(none)" */
extern const char message_done[];	/* "Message done" */
extern const char cant_savescr[];	/* "* * * Can't save screen\r\n" */
extern const char copying_msg[];	/* "copying message" */
extern const char str_builtin[];	/* "builtin+" */
extern const char str_sent[];		/* "sent" */
extern const char enc_7bit[];		/* "7bit" */
extern const char enc_8bit[];		/* "8bit" */
extern const char enc_binary[];		/* "binary" */
extern const char enc_qp[];		/* "quoted-printable" */
extern const char enc_base64[];		/* "base64" */
extern const char wrkfileext[];		/* ".wrk" */
extern BOOL use_ed;		/* Drop into editor when creating messages */
extern BOOL dotmenu;		/* If true then use . .q .e .? etc. */
extern char *hostname;		/* name of this host from rc file */
extern char *username;		/* name of this user from rc file */
extern char *useraddr;		/* local address of this user from rc file */
extern char *tfname;		/* file for mailtext */
extern char *mfname;		/* file for mailtext w/ header */
extern char *uucp_command;	/* cmd to use when spawning rmail */
extern char *cwd;		/* working directory (directory at startup) */
extern char *opthdr[];		/* Array of optional extra header lines */
extern BOOL nosigdashes;	/* Supress the --_ before .signatures */
extern int saveall;		/* Save all outgoing messages to =sent (0-3) */
extern char escape_char;	/* "Dot menu" escape character (default '.') */
extern BOOL tty;		/* tells if stdin is a tty */
extern char *replyto;		/* address for Reply-To: header field */
extern char *uucpreplyto;	/* address for Reply-To: header (uucp mode) */
extern char *editor;		/* user's favourite text editor */
extern int smarthost;		/* declared/initialised in tcmain.c */
extern char *mailhost;		/* declared/initialised in tcmain.c */
extern char *hdr_fullname;	/* User's full name, for use in header */
extern BOOL cnfmsendabort;	/* Confirm aborting message to be sent? */
extern int enclong;		/* 0 = never, 1 = always, 2 = prompt */

/*
 * Functions defined and used in this module
 */
static int split_write(FILE *tfile, const char *name, const char *body);
static CTTYPE select_content_type(const char *fname);
static int append_signature(FILE *tfile);
FILE *invoke_editor(FILE *tfile, const char *to, const char *subject,
		    BOOL mailing, BOOL reopen);
static int copy_message(FILE *tfile, long sequence);
static int write_messageid(FILE *tfile, long sequence);
static int free_dosmtp_ptrs(void);
static int create_header_file(const char *to, const char *cc, const char *bcc,
			      const char *subject);
static int write_headers(FILE *tfile, const char *to, const char *cc,
			 const char *bcc, const char *subject);
static int conv_base64_ch(char ch);
static int add_headers(int done_already, BOOL ismime, CTTYPE ct_type,
		       CETYPE ct_enc);
static int create_mailbody(FILE *mfp, const char *to, const char *subject,
			   BOOL ismime, CTTYPE ct_type, CETYPE ct_enc,
			   SENDTYPE action);
static FILE *reedit_mail(FILE *tfile, const char *to, const char *subject,
			 CTTYPE ct_type, CETYPE ct_enc, int header);
char *buildcmd(const char *cmdline, const char *rcvr, const char *subject,
	       const char *fname);
int dosmtpsend(FILE *mfp, const char *to, const char *subject, const char *cc,
	       BOOL ismime, CTTYPE ct_type, CETYPE ct_enc, SENDTYPE action);
void backslash(char *s);
static void clear_up(void);
static int check_recipients(const char *s, char *scratch, size_t scratchsize);
static void backup_name(char *origname);
static int file_backup(const char *fname, char *bkname, size_t bksize);
int get_addr_filename(char *into, size_t intosize, const char *addr);
static FILETYPE get_filetype(FILE *fp, char *buff, size_t bufsize,
			     BOOL istext);
static int copy_txtfile(FILE *fpdst, FILE *fpsrc, char *buf, size_t bufsize);
static void getname(const char *from, char *realname, size_t realnamesize);

/*
 * Functions defined elsewhere and used in this module
 */
char maxgets(char *s, int maxlen);
int getwin(int x, int y, int maxlen, int winattr, char *input,
	   const char *prompt, ...);
int savemsg(PHTL cmsg, FILE *tfile, int weed, unsigned options);
long get_msgid(void);
int mlock(const char *dir, const char *id);
void rmlock(const char *dir, const char *id);
char *next_address(char *into, size_t intosize, const char *from,
		   char **addr_ptr, size_t *addr_len);
BOOL fgets_msg(char *into, size_t intosize, FILE *fp, BOOL unfold);
char rip(char *s);
char eoltest(const char *s);
BOOL y_n(const char *msg);
HEADERTYPE htype(const char *p);
BOOL htype_match(const char *p, HEADERTYPE type);
const char *hdr_body(const char *field, HEADERTYPE type);
void colputs(int attr, const char *str);
void clreos(void);
void errordsp(const char *txt);
void fopen_error(const char *filename, const char *descrip);
void write_error(const char *filename, const char *descrip);
void rename_error(const char *fromname, const char *toname);
FILE *opentemp(char *name, const char *mode);
int expand_alias(char *st, size_t stsize);
unsigned int getkey(void);
void notepad(const char *fname, char *title);
void beep(void);
void untag_msg(PHTL cmsg);
int check_status(char *buf, size_t bufsize, FILE *fp, BOOL quote);
int write_status_line(FILE *fp, BOOL quote);
char *rfcdate(time_t t);
const char *select_max(int maxnum);
void folderpath(char *path, size_t pathsize, const char *foldername);


/*
 * split_write()
 *
 * Write the message header field <name><body> to <tfile>; if necessary, fold
 * it at appropriate points so that no line is longer than MAX_HDRLINE
 * characters if possible.  (Strictly, we can fold the line anywhere where
 * there may be linear-white-space; actually, we only do it at unquoted
 * whitespace or immediately after an unquoted comma.)
 *
 * Return 0 if we manage to write it all OK, non-zero if not (< 0 on memory
 * errors etc., > 0 on write errors).
 */
static int split_write(FILE *tfile, const char *name, const char *body)
{
    char *temp, *lp, *sp, *ep, *p, *bp, *bcp;
    const char *sep;
    size_t nl, reqsize;
    BOOL inquotes;
    int commentlevel;
    int status = 0;

    /* Build a single array containing the entire line */
    /* We shouldn't need to do this malloc stuff... ST-FIXME */
    sep = (islwsp(body[0])) ? "" : " ";
    reqsize = (nl = strlen(name) + strlen(sep)) + strlen(body) + 1;
    if ((temp = malloc(reqsize)) == NULL)
    {
	errordsp(no_mem);
	sleep(3);
	return -1;
    }
    if (*sep)
	PUSTRLCPY3(temp, reqsize, name, sep, body);
    else
	PUSTRLCPY2(temp, reqsize, name, body);

    /*
     * Print out the full text from <temp>, trying not to print any lines
     * longer than MAX_HDRLINE chars; we move <lp> (the line start point)
     * through the <temp> array to do this.  Each <lp> line is checked for
     * being > MAX_HDRLINE chars long; each time we find this, we search for
     * the last plausible break-point before the MAX_HDRLINE mark or, failing
     * that, the first such point *after* that mark.  "Plausible" break-points
     * are (a) at an LWSP-char or (b) immediately after a comma, IF said point
     * is not in a quoted-string "..." or a comment (...).
     *
     * (Actually, RFC 822 specifically says that quoted-strings and comments
     * MAY be folded, but IMHO it looks rather ugly, so we don't allow it.)
     *
     * We don't want to break the first line immediately after the name (the
     * name should by followed by space, which would therefore otherwise be a
     * candidate for folding); we therefore need to start searching after the
     * end of the name string, rather than the start of the line, for the
     * first string only.  To facilitate this, we maintain <sp> as the search
     * start point; sp > lp for the first line, but sp = lp for all subsequent
     * lines.
     */
    lp = temp;				/* first line starts at beginning! */
    sp = temp + nl;			/* first search starts after name */

    while (strlen(lp) > MAX_HDRLINE)	/* remaining text is too long */
    {
	/*
	 * Starting at <sp>, run through the string forwards, noting the
	 * positions of possible break-points in <bp>; if we reach the
	 * MAX_HDRLINE point and we've found a possible break-point, fold the
	 * string at that point: if not, keep searching until we find the
	 * first possible break-point, and fold the string there (this line
	 * will be over MAX_HDRLINE chars, but that's too bad).
	 *
	 * We don't consider the first point in the line to be a valid
	 * break-point under any circumstances.  Moreover, we don't break a
	 * line in a quoted-string "...", or in a comment (...).
	 */
	inquotes = FALSE;		/* not in a quote here */
	commentlevel = 0;		/* not in a comment here */
	bp = bcp = NULL;		/* no break points found yet */
	ep = lp + MAX_HDRLINE - ((lp > temp) ? 1 : 0);	/* end point */
	for (p = sp; *p; p++)
	{
	    if (*p == '\"')		/* start or end of quoted-string */
		inquotes = !inquotes;
	    else if (!inquotes)
	    {
		if (*p == '(')
		    commentlevel++;
		else if (*p == ')')
		    commentlevel--;
		else if (commentlevel <= 0 && p > sp &&
			 (islwsp(*p) || *p == ','))
		{		/* at unquoted, uncommented LWSP or comma */
		    if (bp == NULL || p < ep || *p != ',')
		    {
			bp = p;		/* try not to break at col-80 comma */
			if (*p == ',')
			    bcp = p;	/* note position of comma */
		    }
		}
	    }
	    if (p >= ep && bp != NULL)	/* >= 80th char; got a break point */
	    {
		/*
		 * If we've seen a comma, but this break point isn't at or
		 * just after that comma, break at the comma instead
		 */
		if (bcp != NULL && bcp != bp && bcp != bp - 1)
		    bp = bcp;
		break;	/* out of the for (p) */
	    }
	}

	if (bp == NULL)		/* no break-point: print the whole line */
	    break;	/* out of the while (strlen) */

	if (*bp == ',')		/* got a comma: break *after* the comma */
	    bp++;

	/* Continuation lines must start with LWSP */
	if (lp > temp && !islwsp(lp[0]) && fputc(' ', tfile) == EOF)
	{
	    status = 1;
	    break;	/* out of the while (strlen) */
	}
	if (fprintf(tfile, "%.*s\n", (int) (bp - lp), lp) == EOF)
	{
	    status = 2;
	    break;	/* out of the while (strlen) */
	}
	lp = bp;
	sp = lp;
    }

    if (status == 0)	/* Everything OK up to now: write last line */
    {
	/* Continuation lines must start with LWSP */
	if (lp > temp && !islwsp(lp[0]) && fputc(' ', tfile) == EOF)
	    status = 3;
	else if (fprintf(tfile, "%s\n", lp) == EOF)
	    status = 4;
    }
    free(temp);
    return status;
}


/*
 * select_content_type()
 *
 * Return an appropriate Content-Type for filename <fname>, based on its file
 * extension
 */
static CTTYPE select_content_type(const char *fname)
{
    const char *p;

    if ((p = strrchr(fname, '.')) != NULL)
    {
	p++;
	if (stricmp(p, "gif") == 0)
	    return CT_IMAGE_GIF;
	if (stricmp(p, "jpg") == 0)
	    return CT_IMAGE_JPEG;
	if (stricmp(p, "mpg") == 0)
	    return CT_VIDEO_MPEG;
	if (stricmp(p, "au") == 0)
	    return CT_AUDIO_AU;
    }
    return CT_APP_OCTET;
}


/*
 * mailto()
 *
 * Send mail to someone, out of the blue; this calls dosmtpsend() once it's
 * worked out who to send it to, what subject to use etc.
 *
 * We're given <buf>, which is a dual-purpose buffer: we use it to prompt the
 * user for the initial "To" list of recipients, and we also copy a final
 * status message (typically "Not sent" or "Message sent") into it to be
 * passed back to the calling function.  It is <bufsize> bytes long.
 *
 * <action> says which of the following types of mailto() action we are doing:
 *	SEND_NEWMSG	Create a new message from scratch
 *	SEND_FILE	Send the contents of a file, encoded if applicable
 *	SEND_FWDQUOTE	Forward a message to someone
 *
 * We will never be given SEND_REPLYQUOTE, since that's handled by reply()
 * instead of mailto().
 *
 * Return 0 if all goes well, > 0 on error.
 */
int mailto(char *buf, size_t bufsize, SENDTYPE action)
{
    char *to, *s1;
    char *cc;
    char subject[100];
    int result = 0;
    BOOL ismime = FALSE;
    CTTYPE ct_type = CT_NONE;
    CETYPE ct_enc = CE_NONE;
    FILE *tfile;

    /* Why do we malloc <cc>, rather than using the stack? Any ideas? -ST */
    cc = malloc(LLINELEN);
    if (cc == NULL)
    {
	PUSTRCPY(buf, bufsize, no_mem);
	return 1;
    }
    *cc = '\0';

    /*
     * Get the list of recipients; stop now if this step is aborted
     */
    cputs("\r\n");
    cputs((action == SEND_FWDQUOTE) ? fwd_prompt : mail_prompt);
    maxgets(buf, 79);
    if (*buf == '\0')		/* Nothing to do: give up now */
    {
	free(cc);
	/* <buf> is already a null string; leave it that way */
	return 0;
    }

    to = skip_lwsp(buf);		/* Skip leading blanks in "To" line */

    /*
     * Get the list of Cc recipients
     */
    cputs("\r\n");
    cputs(cc_prompt);
    maxgets(cc, 79);
    if (*cc)
    {
	s1 = skip_lwsp(cc);		/* Skip leading blanks in "Cc" line */
	if (s1 > cc)
	    memmove(cc, s1, strlen(s1) + 1);
    }

    /*
     * Get the subject, unless we're forwarding
     */
    cputs("\r\n");
    if (action != SEND_FWDQUOTE)	/* Prompt for subject */
    {
	cputs(subj_prompt);
	*subject = '\0';
	maxgets(subject, 79);
	if (subject[0] == '\0')
	    AUSTRCPY(subject, bracket_none);
    }

    *send_file = '\0';

    if (action == SEND_FILE)		/* Get filename to send */
    {
	while (TRUE)
	{
	    /*
	     * Ask the user for the name of the file to send; open <tfile>
	     * if we get one (remember to close it again if the user
	     * decides not to forward this file!)
	     */
	    if (getwin(10, 17, 64, col_select, send_file,
		       "Enter the filename containing the message", NULL) == 0)
	    {
		free(cc);
		PUSTRCPY(buf, bufsize, not_sent);
		return 2;
	    }
	    if ((tfile = fopen(send_file, "rb")) == NULL)
	    {
		if (y_n("I can't find that file. Try another?") == FALSE)
		{
		    free(cc);
		    PUSTRCPY(buf, bufsize, not_sent);
		    return 3;
		}
		continue;	/* with the while (TRUE) */
	    }

	    ismime = FALSE;
	    if (enablemime)		/* Consider encoding the file */
	    {
		char tstring[LINELEN], choice[3];
		FILETYPE ftype;

		ftype = get_filetype(tfile, tstring, sizeof(tstring), FALSE);
		choice[0] = (ftype == FT_BINARY) ? '1'
		    : (ftype == FT_ASCSHRT || ftype == FT_ACZSHRT)  ? '4'
		    : '2';
		choice[1] = '\0';
		AUSTRLCPY2(tstring, "File is ", ft_desc[ftype]);
		if (getwin(3, 13, 1, col_select, choice,
			   "Transmitting a File",
			   tstring,
			   "",
			   " 1. Use MIME base64 encoding",
			   " 2. Use MIME quoted-printable encoding",
			   " 3. Use uuencode encoding (deprecated)",
			   " 4. Don't encode as a MIME message",
			   " 5. Select a different file",
			   "",
			   select_max(5),
			   NULL) == 0)
		{				/* Give up altogether */
		    (void) fclose(tfile);		/* OK: read only */
		    free(cc);
		    PUSTRCPY(buf, bufsize, not_sent);
		    return 4;
		}
		rewind(tfile);		/* rewind to start of file */
		if (*choice == '1')		/* Use base64 encoding */
		{
		    ismime = TRUE;
		    ct_type = select_content_type(send_file);
		    ct_enc = CE_BASE64;
		}
		else if (*choice == '2')	/* Use q-p encoding */
		{
		    ismime = TRUE;
		    ct_type = CT_TEXT_PLN88591;	/* assume it's text */
		    ct_enc = CE_QUOTED;
		}
		else if (*choice == '3')	/* Use UUencoding */
		{
		    ismime = TRUE;
		    ct_type = CT_APP_OCTET;
		    ct_enc = CE_UUENCODE;
		}
		else if (*choice == '5')	/* Choose another file */
		{
		    (void) fclose(tfile);		/* OK: read only */
		    continue;	/* with the while (TRUE) */
		}
		/* else '4', we're not encoding it; fall through */
	    }
	    /* else !enablemime -- no encoding, just send it. */
	    break;	/* out of the while (TRUE) */
	}
    }

    /*
     * Note: <ismime> can only be true at this point if <send_file> has been
     * set to the name of an existing file to be transmitted; it will be false
     * otherwise.
     */

    if (action == SEND_FWDQUOTE)
    {
	/*
	 * We're forwarding message(s) to the recipient(s); run through the
	 * list of tagged files, and for each one, create a temporary file,
	 * copy the message (with full headers) into that file using
	 * savemsg(), and then call dosmtpsend() to complete sending the
	 * message.
	 */
	PHTL cmsg;
	char tempname[FNAMELEN];
	int err = 0;

	for (cmsg = mbox; cmsg != NULL && err == 0; cmsg = cmsg->next)
	    if (MSGISTAGGED(cmsg))
	    {
		untag_msg(cmsg);
		gotoxy(1, display.screenheight);
		clreol();
		cputs(subj_prompt);
		/* ST-FIXME: cmsg->subject is truncated, only 35 chars! */
		ustrncpy(subject, cmsg->subject, 74);
		AUSTRCAT(subject, " (fwd)");
		maxgets(subject, 79);
		if (subject[0] == '\0')
		    AUSTRCPY(subject, bracket_none);
		if ((tfile = opentemp(tempname, "w+")) == NULL)
		{
		    fopen_error(tempname, "temp file");
		    err = 1;
		}
		else
		{
		    /* We now decode q-p as we're forwarding it */
		    err = savemsg(cmsg, tfile, HW_FULL,
				  SM_DECODE | SM_QUOTE | SM_FORWARD);
		    if (err < 0)	/* write error during savemsg() */
			write_error(tempname, copying_msg);
		    else if (err == 0)	/* saved OK: savemsg() does fflush() */
		    {
			rewind(tfile);
			if (dotmenu || !use_ed)
			{
			    clrscr();
			    cprintf(to_subj_fmt, to, subject);
			}
			result = dosmtpsend(tfile, to, subject, cc, FALSE,
					    ct_type, ct_enc, action);
			gotoxy(1, display.screenheight - 1);
			clreol();
			if (result == -2)	/* Write error */
			    err = 2;
		    }
		    /* We're done with the temp file; close & unlink it */
		    (void) fclose(tfile);		/* OK: don't care */
		    unlink(tempname);
		}
	    }
	if (err != 0)		/* We had an error of some sort */
	    result = 5;
    }
    else	/* not SEND_FWDQUOTE: something else */
    {
	/*
	 * Not forwarding; we're either sending a file to somebody
	 * (SEND_FILE), in which case <tfile> will be open for that file), or
	 * we're just composing a new message (SEND_NEWMSG).
	 */
	if (dotmenu || !use_ed)
	{
	    clrscr();
	    cprintf(to_subj_fmt, to, subject);
	}
	if (*to != '\0')	/* "To" list isn't null */
	    result = dosmtpsend((action == SEND_FILE) ? tfile : NULL,
				to, subject, cc, ismime, ct_type, ct_enc,
				action);	/*lint !e644 !e645 */ /*OK*/
	else
	    result = 5;
	if (action == SEND_FILE)	/* <tfile> still open; close it */
	    (void) fclose(tfile);			/* OK: read only */
    }

    if (result == 0)
	PUSTRCPY(buf, bufsize, message_done);
    else
	PUSTRCPY(buf, bufsize, not_sent);

    free(cc);
    return result;
}


/*
 * append_signature()
 *
 * Append signature at end of message <tfile> and fflush() the file to make
 * sure it gets written to disk; return 0 if OK, > 0 if not.  We make sure
 * that we only append the sig once by using local static variable <done>;
 * before starting to process a new sig-less message body, we are called with
 * <tfile> set to NULL to make us reset the <done> variable.
 */
static int append_signature(FILE *tfile)
{
    static BOOL done = FALSE;

    if (tfile == NULL)		/* Reset for new message */
    {
	done = FALSE;
	return 0;
    }
    if (done)			/* Already appended sig to this message */
	return 0;
    if (homedir[0])
    {
	char sfilename[PATHLEN];
	FILE *sfile;

	AUSTRLCPY4(sfilename, homedir, "\\", username, ".sig");
	if ((sfile = fopen(sfilename, "rt")) != NULL)
	{
	    char buf[LINELEN];
	    int err = 0;

	    if ((!nosigdashes && fputs("\n-- \n", tfile) == EOF) ||
		copy_txtfile(tfile, sfile, buf, sizeof(buf)) != 0)
		err = 1;
	    if (fflush(tfile) != 0)
		err = 2;
	    (void) fclose(sfile);			/* OK: read only */
	    done = TRUE;	/* for better or worse */
	    return err;
	}
    }
    return 0;
}


/*
 * invoke_editor()
 *
 * There is a file to be edited on stream <tfile> (which may be NULL, if it's
 * not currently open): if <mailing> is true, this is a mail message, with
 * "To" line <to>, "Subject" line <subject> and filename in global <tfname>;
 * if <mailing> is false, this is a generic file whose filename is passed in
 * <to> instead (we don't use <subject> in this case, and it will probably be
 * NULL).  If it's a mail message, we do certain "mail" stuff to it; if not,
 * we just edit it.
 *
 * In order to fire up an editor on the file, we need to close the file
 * stream.  The Demon version of PCElm used to re-open the stream afterwards,
 * and hope that the new stream got opened in the same FILE structure as the
 * old one -- this was really ghastly, and IMHO fairly dangerous, especially
 * when you consider that the file stream had often been closed before being
 * passed to us, so any attempt to use the passed <tfile> would probably end
 * in disaster!  (Observe the way the builtin editor appends a newline to the
 * file before closing it.)  We now use the <reopen> flag to tell us whether
 * the calling function wants us to (re)open the file after we've edited it,
 * and return a pointer to the reopened file stream, or whether we should
 * leave the file alone (and return NULL).
 */
FILE *invoke_editor(FILE *tfile, const char *to, const char *subject,
		    BOOL mailing, BOOL reopen)
{
    int x, y;
    const char *fnamep;
    char *saved_screen, *scp;
    char tstring[LLINELEN];

    if (mailing)		/* Sending mail: do mail-ish things */
    {
	if (dotmenu)
	    cputs("\rInvoking editor\r\n");
	fnamep = tfname;
    }
    else			/* Just editing a file */
	fnamep = to;

    x = wherex();
    y = wherey();
    /* get memory for screen */
    saved_screen = malloc(2 * display.screenheight * display.screenwidth);
    if (saved_screen != NULL)
    {
	if (!gettext(1, 1, display.screenwidth, display.screenheight,
		     saved_screen))
	{
	    cputs(cant_savescr);
	    sleep(1);
	    free(saved_screen);
	    saved_screen = NULL;
	}
    }

    if (stricmp(editor, str_builtin))	/* don't use builtin editor */
    {
	if (tfile)		/* close the file before editing it */
	{
	    if (fclose(tfile) != 0)	/* file had errors */
		write_error(tfname, "closing file for edit");
	}
#if 0			/* Ghastly old manual "%s" substitution */
	/*
	 * ST-FIXME: this "%s substitution" bit is an awful crock, since it
	 * throws away the command string after the %s, which probably isn't
	 * what the user expects!  It should use the buildcmd() system.
	 */
	if ((scp = strstr(editor, "%s")) != NULL)
	{
	    size_t offset = (size_t) (scp - editor);
	    if (offset < sizeof(tstring))   /* may be room for <fnamep> */
	    {
		ustrncpy(tstring, editor, offset);
		AUSTRCAT(tstring, fnamep);
	    }
	    else			    /* no room for <fnamep>! */
		AUSTRCPY(tstring, editor);
	}
	else
	    AUSTRLCPY3(tstring, editor, " ", fnamep);
	/* call editor to enter message text */
	(void) disp_system(tstring, FALSE);
#else			/* Proper command parsing, '$' => filename */
	scp = buildcmd(editor, "", "", fnamep);
	(void) disp_system(scp, FALSE);		/* edit message text */
#endif
    }
    else				/* use builtin editor */
    {
	if (tfile)
	{
	    (void) fputs("\n", tfile);	/* this is a big FIXME, since I have
					   some trouble in my notepad, if a
					   empty file is opened to edit! */
	    (void) fclose(tfile);			/* OK: don't care */
	}
	if (mailing)
	    AUSTRLCPY5(tstring, "To: ", to, " Subject: ", subject, "\n\n");
	else
	    AUSTRCPY(tstring, to);
	notepad(fnamep, tstring);
    }

    if (saved_screen != NULL)
    {		/* now restore */
	puttext(1, 1, display.screenwidth, display.screenheight,
		saved_screen);
	free(saved_screen);
	saved_screen = NULL;
	gotoxy(x, y);
	if (mailing && dotmenu)
	    cputs("Back from editor ... \r\n");
    }

    /*
     * If <reopen> is TRUE, we re-open the file ready to append to it and
     * return the stream pointer.  If not, just return NULL.
     */
    return (reopen) ? fopen(fnamep, "a+") : NULL;
}


/*
 * copy_message()
 *
 * Copy a mail message from global <tfname> to the provided <tfile> stream,
 * adding a Message-ID line immediately before the (first) "To:" line
 * encountered (or as the last header field if there is no "To:" field found);
 * the sequence number to use for the message-id is <sequence>.  Ensure the
 * copy ends with a newline, even if the original didn't.  Return 0 if all
 * goes well, < 0 if we can't open the file to read (after putting up an error
 * message) or > 0 if we have a write error (no error message -- needs to be
 * done by something which knows the written file name).
 */
static int copy_message(FILE *tfile, long sequence)
{
    int done_messageid = FALSE, status = 0;
    FILE *infl;
    char buf[LINELEN];
    char last_eol = 1;

    if ((infl = fopen(tfname, "rt")) == NULL)
    {
	fopen_error(tfname, mailtext_oerr);
	return -1;
    }

    buf[0] = '\0';		/* Ensure this is initialised */
    while (fgets(buf, sizeof(buf), infl) != NULL)
    {
	if (!done_messageid)	/* Waiting to insert msgid */
	{
	    if (last_eol && (htype_match(buf, HT_TO) || ISBLANKLINE(buf)))
	    {
		if (write_messageid(tfile, sequence))	/* write error */
		{
		    status = 1;
		    break;	/* out of the while (fgets) */
		}
		done_messageid = TRUE;
	    }
	    last_eol = eoltest(buf);
	}
	if (fputs(buf, tfile) == EOF)
	{
	    status = 2;
	    break;	/* out of the while (fgets) */
	}
    }
    /* If OK so far, ensure last line ends with newline */
    if (status == 0 && buf[0] && buf[strlen(buf) - 1] != '\n')
	if (fputs("\n", tfile) == EOF)
	    status = 3;
    /* Make sure we've written the message-id */
    if (status == 0 && !done_messageid)		/* Still waiting! */
	if (write_messageid(tfile, sequence))		/* write error */
	    status = 4;
    /* Close the input file */
    if (fclose(infl) != 0 && status == 0)
	status = 5;
    return status;
}


/*
 * write_messageid()
 *
 * Write a "Message-ID:" header line for message sequence number <sequence>
 * to stream <tfile>; return 0 if it appears to go OK, 1 if not.
 */
static int write_messageid(FILE *tfile, long sequence)
{
    if (fprintf(tfile, "%s <%ld@%s>\n", headername[HT_MSGID],
		sequence, hostname) == EOF)
	return 1;
    return 0;
}


/*
 * free_dosmtp_ptrs()
 *
 * Free all the dynamically-allocated SMTP header line buffers and return 1;
 * this is mostly called when things have gone pear-shaped, and we want to
 * bail out without losing too much memory.
 */
static int free_dosmtp_ptrs()
{
    free(reciprest);
    free(expanded_to);
    free(expanded_cc);
    free(expanded_bcc);
    reciprest = expanded_to = expanded_cc = expanded_bcc = NULL;
    reciprest_size = expanded_to_size = expanded_cc_size =
	expanded_bcc_size = 0;
    return 1;
}


/*
 * create_header_file()
 *
 * Try to create the header file, using global array <header_file> for the
 * filename, and with <to>, <cc>, <bcc> and <subject> containing their
 * respective header field bodies; return 0 if all goes well, 1 if we can't
 * open the header file, or 2 if we can't write the headers into it.
 */
static int create_header_file(const char *to, const char *cc, const char *bcc,
			      const char *subject)
{
    FILE *tfile;
    int status = 0;

    AUSTRLCPY2(header_file, cwd, "\\HEADER.TMP");
    if ((tfile = fopen(header_file, "w")) == NULL)
    {
	fopen_error(header_file, "temp. header file");
	return 1;
    }
    if (write_headers(tfile, to, cc, bcc, subject) != 0)
	status = 2;
    if (fclose(tfile) != 0)
	status = 2;
    if (status != 0)
	write_error(header_file, "writing message header");
    return status;
}


/*
 * write_headers()
 *
 * Write RFC [2]822-compatible headers; write all lines (including "Cc" and
 * "Bcc", which may be blank) *except* for "Date" and "Message-ID", which we
 * will fill in later (at the point the user commits the message for sending).
 *
 * Return 0 if all goes well, something else if we couldn't write one of the
 * lines.
 */
static int write_headers(FILE *tfile, const char *to, const char *cc,
			 const char *bcc, const char *subject)
{
    int i;

    if (fprintf(tfile, "%s %s%s<%s@%s>\n", headername[HT_FROM], hdr_fullname,
		(*hdr_fullname) ? " " : "", useraddr, hostname) == EOF)
	return 1;
    if (replyto && split_write(tfile, headername[HT_REPLYTO], replyto) != 0)
	return 2;
    if (split_write(tfile, headername[HT_TO], to) != 0)
	return 3;
    if (split_write(tfile, headername[HT_CC], cc) != 0)
	return 4;
    if (split_write(tfile, headername[HT_BCC], bcc) != 0)
	return 5;
    if (split_write(tfile, headername[HT_SUBJECT], subject) != 0)
	return 6;
    /* include any user defined headers */
    for (i = 0; i < MAXOPTHDR; i++)
	if (opthdr[i] && *opthdr[i] && fprintf(tfile, "%s\n", opthdr[i]) ==
	    EOF)
	    return 7;
    if (fprintf(tfile, "X-Mailer: %s\n", myversion) == EOF)
	return 8;
    return 0;
}


/*
 * conv_base64_ch()
 *
 * Return the appropriate MIME base64 character to use for <ch>
 */
static int conv_base64_ch(char ch)
{
    if (ch < 26)
	return 'A' + ch;
    if (ch < 52)
	return 'a' + (ch - 26);
    if (ch < 62)
	return '0' + (ch - 52);
    if (ch == 62)
	return '+';
    return '/';
}


/*
 * add_headers()
 *
 * Add the headers (from global <header_file>) to the message body (global
 * <tfname>) and put the combined result in <tfname>.  <done_already>
 * indicates whether this has already been done, thus:
 *	0 = Not done
 *	1 = Done
 *     -1 = Force keep existing headers
 * <ismime>, <ct_type> and <ct_enc> indicate what MIME headers we're supposed
 * to write, and what encoding we may be supposed to do to any attachments.
 * Return TRUE if it all goes according to plan, and the headers are
 * successfully added, or FALSE if not.
 *
 * If <ismime> is true, <send_file> will contain the name of any file we are
 * sending; if <send_file> is set, we're sending a file, and if it isn't, we
 * aren't.  This is better than the old system of using the <ct_type> setting
 * to guess whether we were sending a file, which meant that sending a text
 * file didn't get a "name=..." parameter on the Content-Type line.
 */
static int add_headers(int done_already, BOOL ismime, CTTYPE ct_type,
		       CETYPE ct_enc)
{
    char bakfile[PATHLEN];
    char temp[LLINELEN], *p;
    FILE *tfile, *fp;
    const char *errdesc = NULL;

    if (done_already == 1)	/* already done */
	return TRUE;

    /*
     * Headers are in HEADER.TMP (global <header_file>); the existing message
     * is MAILTEXT.TXT (global <tfname>).  Rename current <tfname> to backup,
     * stored in <bakfile>; then rename the current header file to <tfname>
     * and append the stuff from the backup file <bakfile>.
     */
    if (file_backup(tfname, bakfile, sizeof(bakfile)))	/* backup failed */
	return FALSE;
    if (done_already >= 0 && rename(header_file, tfname))
    {
	rename_error(header_file, tfname);
	return FALSE;
    }
    if ((tfile = fopen(tfname, "a+")) == NULL)
    {
	fopen_error(tfname, mailtext_oerr);
	return FALSE;
    }
    /*
     * Open the backup of the body file: this may be a binary file to be
     * encoded, if <ct_enc> is set accordingly
     */
    if (ct_enc == CE_BASE64 || ct_enc == CE_UUENCODE || ct_enc == CE_QUOTED)
	fp = fopen(bakfile, "rb");	/* binary mode if file to be encoded */
    else
	fp = fopen(bakfile, "r");	/* text mode if not to be encoded */
    if (fp == NULL)
    {
	fopen_error(bakfile, backup_oerr);
	(void) fclose(tfile);			/* OK: not used yet */
	return FALSE;
    }

    if (done_already < 0)	/* Copy existing headers from body in <fp> */
    {
	char last_eol = 1;

	while (fgets(temp, sizeof(temp), fp) != NULL)
	{
	    if (last_eol && ISBLANKLINE(temp))
		break;		/* pure LF or CRLF : end of header */
	    last_eol = eoltest(temp);
	    if (fputs(temp, tfile) == EOF)
	    {
		errdesc = "copying headers";
		break;	/* out of the while (fgets) */
	    }
	}
    }

    if (!errdesc && ismime)
    {
	const char *typ, *enc;

	textcolor(col_enhanc);
	cprintf("\r\nEncoding MIME message");
	textcolor(col_normal);

	/* Set <typ> to appropriate Content-Type string */
	switch (ct_type)
	{
	case CT_TEXT_PLNASCII:
	case CT_UNKNOWN:
	case CT_NONE:
	default:
	    typ = "text/plain; charset=\"us-ascii\"";
	    break;
	case CT_TEXT_PLN88591:
	    typ = "text/plain; charset=\"iso-8859-1\"";
	    break;
	case CT_TEXT_RICH:
	    typ = "text/enriched";
	    break;
	case CT_TEXT_HTML:
	    typ = "text/html";
	    break;
	case CT_IMAGE_GIF:
	    typ = "image/gif";
	    break;
	case CT_IMAGE_JPEG:
	    typ = "image/jpeg";
	    break;
	case CT_AUDIO_AU:
	    typ = "audio/au";
	    break;
	case CT_VIDEO_MPEG:
	    typ = "video/mpeg";
	    break;
	case CT_TEXT_MISC:
	case CT_IMAGE_MISC:
	case CT_AUDIO_MISC:
	case CT_VIDEO_MISC:
	case CT_APP_OCTET:
	case CT_APP_MISC:
	case CT_MSG_DELIVSTAT:
	case CT_MSG_MISC:
	    typ = "application/octet-stream";
	    break;
	case CT_MULTI_MIXED:
	    typ = "multipart/mixed";
	    break;
	case CT_MULTI_ALTERN:
	    typ = "multipart/alternative";
	    break;
	case CT_MULTI_DIGEST:
	    typ = "multipart/digest";
	    break;
	case CT_MULTI_MISC:
	    typ = "multipart/unknown";
	    break;
	case CT_MSG_RFC822:
	    typ = "message/rfc822";
	    break;
	}

	/* Set <enc> to appropriate Content-Transfer-Encoding string */
	switch (ct_enc)
	{
	case CE_7BIT:
	case CE_NONE:
	case CE_UNKNOWN:
	default:
	    enc = enc_7bit;
	    break;
	case CE_8BIT:
	    enc = enc_8bit;
	    break;
	case CE_BINARY:
	    enc = enc_binary;
	    break;
	case CE_QUOTED:
	    enc = enc_qp;
	    break;
	case CE_BASE64:
	    enc = enc_base64;
	    break;
	case CE_UUENCODE:
	    enc = "x-uuencode";
	    break;
	}

	/* Write the MIME header fields; check ferror() at the end */
	(void) fprintf(tfile, "%s 1.0\n", headername[HT_MIME_VERSION]);
	(void) fprintf(tfile, "%s %s", headername[HT_CONTENT_TYPE], typ);
	if (send_file[0])	/* It's a file: needs "name=" argument */
	    (void) fprintf(tfile, "; name=\"%.40s\"", leafname(send_file));
	(void) fprintf(tfile, "\n%s %s\n",
		       headername[HT_CONTENT_XFER_ENCODING], enc);
	if (ferror(tfile))
	    errdesc = "writing MIME headers";
    }

    if (!errdesc && fputs("\n", tfile) == EOF)	/* blank line after headers */
	errdesc = "writing separator";

    if (errdesc)		/* It's all gone horribly wrong */
    {
	write_error(tfname, errdesc);
	(void) fclose(tfile);				/* OK: giving up */
	(void) fclose(fp);				/* OK: read only */
	return FALSE;
    }

    if (ct_enc == CE_BASE64)			/* Base64 encoding */
    {
	int n;
	unsigned char ch;

	while ((n = (int) fread(temp, 1, B64_NBYTES, fp)) != 0)
	{
	    for (p = temp; n > 0; n -= 3, p += 3)
	    {
		ch = p[0] >> 2;
		if (fputc(conv_base64_ch(ch), tfile) == EOF)
		    break;
		if (n == 1)		/* Only write p[0] */
		{
		    ch = (p[0] & 0x03) << 4;
		    if (fputc(conv_base64_ch(ch), tfile) == EOF)
			break;
		    if (fputs("==", tfile) == EOF)	/* 2 chars padding */
			break;
		}
		else if (n == 2)	/* Only write p[0] & p[1] */
		{
		    ch = ((p[0] & 0x03) << 4) | (p[1] >> 4);
		    if (fputc(conv_base64_ch(ch), tfile) == EOF)
			break;
		    ch = (p[1] & 0x0f) << 2;
		    if (fputc(conv_base64_ch(ch), tfile) == EOF)
			break;
		    if (fputc('=', tfile) == EOF)	/* 1 char padding */
			break;
		}
		else			/* Write all 3: p[0], p[1], p[2] */
		{
		    ch = ((p[0] & 0x03) << 4) | (p[1] >> 4);
		    if (fputc(conv_base64_ch(ch), tfile) == EOF)
			break;
		    ch = ((p[1] & 0x0f) << 2) | (p[2] >> 6);
		    if (fputc(conv_base64_ch(ch), tfile) == EOF)
			break;
		    ch = p[2] & 0x3f;
		    if (fputc(conv_base64_ch(ch), tfile) == EOF)
			break;
		}
	    }
	    if (n > 0 || fputc('\n', tfile) == EOF)
		break;	/* out of the while (n) */
	}
	if (n != 0)		/* broke out early: error */
	    errdesc = "writing base64 data";
    }
    else if (ct_enc == CE_UUENCODE)		/* UUencoding */
    {
	int n;
	unsigned char ch;
	BOOL err = FALSE;

	if (fputs("table\n"
		  "`!\"#$%&'()*+,-./0123456789:;<=>?\n"
		  "@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_\n", tfile) == EOF ||
	    fprintf(tfile, "begin 644 %s\n", leafname(send_file)) == EOF)
	    err = TRUE;
	else
	{
	    /*
	     * ENC is the basic 1 character encoding function to make a char
	     * printing (although why we have to put it here I don't know -ST)
	     */
#define ENC(c)	((c) ? ((c) & 077) + ' ' : '`')
	    while ((n = (int) fread(temp, 1, UU_NBYTES, fp)) != 0)
	    {
		ch = ENC(n);
		if (fputc(ch, tfile) == EOF)
		    break;
		for (p = temp; n > 0; n -= 3, p += 3)
		{
		    ch = p[0] >> 2;
		    if (fputc(ENC(ch), tfile) == EOF)
			break;
		    ch = ((p[0] & 0x03) << 4) | (p[1] >> 4);
		    if (fputc(ENC(ch), tfile) == EOF)
			break;
		    ch = ((p[1] & 0x0f) << 2) | (p[2] >> 6);
		    if (fputc(ENC(ch), tfile) == EOF)
			break;
		    ch = p[2] & 0x3f;
		    if (fputc(ENC(ch), tfile) == EOF)
			break;
		}
		if (n > 0 || fputc('\n', tfile) == EOF)
		    break;
	    }
	    if (n != 0 || fprintf(tfile, "%c\nend\n", 96) == EOF)
		err = TRUE;
	}
	if (err)
	    errdesc = "writing uuencode data";
    }
    else if (ismime && ct_enc == CE_QUOTED &&
	     (ct_type == CT_TEXT_PLNASCII || ct_type == CT_TEXT_PLN88591))
    {					/* quoted-printable encoding */
	char lb[128];		/* buffer for line (max 76 chars) */
	char *lbp = lb;
	int len;
	unsigned char uc;
	BOOL err = FALSE;

	while (fgets(temp, sizeof(temp), fp) != NULL)
	{
	    for (p = temp; *p; p++)
	    {
		if (strcmp(p, "\r\n") == 0)	/* end of the line */
		{
		    *lbp++ = '\n';	/* write a newline (in text mode) */
		    p++;		/* skip the \r */
		}
		else if (islwsp(*p))		/* linear whitespace */
		{
		    if (strcmp(p + 1, "\r\n") == 0)	/* at end of line */
		    {
			sprintf(lbp, "=%02X\n", *p);			/*OK*/
			lbp += 4;
			p += 2;		/* skip the space & \r */
		    }
		    else		/* OK to use literal char */
			*lbp++ = *p;
		}
		else if (*p < 33 || *p == 61 || *p > 126)   /* needs encode */
		{
		    uc = (unsigned char) *p;
		    /* ST-FIXME: pathetic ISO-8859-1 support */
		    if (ct_type == CT_TEXT_PLN88591 && uc == 156)
			uc = 163;	/* convert pound to u-acute for GBP */
		    sprintf(lbp, "=%02X", uc);				/*OK*/
		    lbp += 3;
		}
		else 			/* OK to use literal */
		    *lbp++ = *p;

		len = (int) (lbp - lb);
		if (lbp[-1] == '\n' && len <= 77)	/* OK to write line */
		{
		    *lbp = '\0';	/* just to be safe */
		    if (fputs(lb, tfile) == EOF)
			break;	/* out of the for (p) */
		    lbp = lb;		/* restart line buffer */
		}
		else if (len > 76)		/* too long: soft break */
		{
		    char *bp;
		    char save[3];
		    int nc;

		    /*
		     * Need to add a '=' and still be <= 76 chars, so break
		     * before lb[75].  If this is part of an =XX sequence,
		     * break before that sequence instead.
		     */
		    bp = &lb[75];
		    if (bp[-1] == '=')
			bp--;
		    else if (bp[-2] == '=')
			bp -= 2;
		    /* Break the line at <bp> */
		    memcpy(save, bp, 3);	/* save the 3 bytes at bp */
		    strcpy(bp, "=\n");		/* overwrite the saved bytes */
		    if (fputs(lb, tfile) == EOF)
			break;	/* out of the for (p) */
		    memcpy(bp, save, 3);	/* replace the saved bytes */
		    /*
		     * Now move any unwritten bytes between <bp> and <lbp> to
		     * the start of the line buffer, ready to append to them
		     */
		    if ((nc = (int) (lbp - bp)) > 0)
		    {
			memmove(lb, bp, (size_t) nc);
			lbp = lb + nc;
		    }
		    else
			lbp = lb;
		}
		/* else carry on building this line; get the next input line */
	    }

	    if (*p)		/* broke out early -- write error */
	    {
		err = TRUE;
		break;	/* out of the while (fgets) */
	    }
	}
	if (!err && lbp > lb)	/* incomplete line waiting in lb[] */
	{
	    /* Force a newline on the end of lb[] and write it out */
	    *lbp++ = '\n';
	    *lbp = '\0';
	    if (fputs(lb, tfile) == EOF)
		err = TRUE;
	}
	if (err)
	    errdesc = "writing quoted-printable data";
    }
    else			/* No encoding -- plain text */
    {
	if (copy_txtfile(tfile, fp, temp, sizeof(temp)) != 0)
	    errdesc = "copying plain text";
    }

    if (fclose(tfile) != 0 && errdesc == NULL)
	errdesc = closing_file;

    (void) fclose(fp);					/* OK: read only */

    if (errdesc)		/* It all went horribly wrong */
    {
	write_error(tfname, errdesc);
	return FALSE;
    }
    return TRUE;
}


/*
 * create_mailbody()
 *
 * Create the mail message body in file <tfname> (MAILTEXT.TXT).  If <mfp> is
 * non-NULL, it's a file pointer to a file which should be included in the
 * body (either being sent to someone, being forwarded to someone, or being
 * quoted for reply purposes).  <to> and <subject> are the list of recipients
 * and the subject respectively; <ismime>, <ct_type> and <ct_enc> indicate
 * whether the message should be a MIME message, and if so, what type and what
 * encoding should be used.
 *
 * On entry, <ismime> will only be true if we're sending a file (whose name is
 * in <send_file>).
 *
 * Return values : -2 : file/disk error
 *		   -1 : aborted by user
 *		    0 : no error; it's not a MIME message
 *		    1 : no error; it's a MIME message
 */
static int create_mailbody(FILE *mfp, const char *to, const char *subject,
			   BOOL ismime, CTTYPE ct_type, CETYPE ct_enc,
			   SENDTYPE action)
{
    FILE *tfile;
    char tstring[LLINELEN];		/* temp for integer conv */
    char *p, ch;
    int header = 0;			/* message header not yet in file */
    BOOL found_mime;
    BOOL writerr = FALSE;

    if (mfp && ismime && ct_enc != CE_NONE)
	tfile = fopen(tfname, "wb+");		/* open binfile for write */
    else
	tfile = fopen(tfname, "w+");		/* open textfile for write */
    if (tfile == NULL)
    {
	fopen_error(tfname, mailtext_oerr);
	return -2;
    }

    if (mfp)		/* Copy the <mfp> file into <tfile>, ready for use */
    {
	if (ismime && ct_enc != CE_NONE)	/* pure binary copy */
	{
	    size_t n;

	    while ((n = fread(tstring, 1, sizeof(tstring), mfp)) != 0)
		if (fwrite(tstring, 1, n, tfile) != n)	/* write error */
		{
		    writerr = TRUE;
		    break;	/* out of the while (n) */
		}
	}
	else					/* line-by-line text copy */
	{
	    if (copy_txtfile(tfile, mfp, tstring, sizeof(tstring)) != 0)
		writerr = TRUE;
	}
	if (!writerr && fflush(tfile) != 0)	/* error flushing writes */
	    writerr = TRUE;
	if (writerr)
	{
	    write_error(tfname, "copying to mailbody");
	    (void) fclose(tfile);			/* OK: giving up */
	    return -2;
	}
    }

    if (mfp == NULL || action == SEND_FWDQUOTE)
    {
	/*
	 * Sending a message, rather than a file; if we're replying to a
	 * previous message and quoting the original, we need to include the
	 * pre-prepared (complete with leading quote strings) file which is
	 * stored in <quote_tmpfile>.
	 */
	if (action == SEND_REPLYQUOTE)
	{
	    FILE *infl;
	    int err = 0;

	    if ((infl = fopen(quote_tmpfile, "rt")) == NULL)
	    {
		fopen_error(quote_tmpfile, "quote file");
		err = 1;
	    }
	    else
	    {
		if (copy_txtfile(tfile, infl, tstring, sizeof(tstring)) != 0)
		{
		    write_error(tfname, "including original message");
		    err = 2;
		}
		/* if copy_txtfile() returned 0, it really is OK */
		(void) fclose(infl);			/* OK: read only */
		unlink(quote_tmpfile);
	    }
	    if (err != 0)
	    {
		(void) fclose(tfile);			/* OK: giving up */
		return -2;
	    }
	}

	if (tty)	/* We've got an active user: create mail normally */
	{
	    if (use_ed)		/* Invoke the editor immediately */
	    {
		/*
		 * <tfile> is open: leave it that way
		 */
		if (append_signature(tfile) != 0)
		{
		    write_error(tfname, sig_werr);
		    (void) fclose(tfile);		/* OK: giving up */
		    return -2;
		}
		tfile = invoke_editor(tfile, to, subject, TRUE, TRUE);
		if (tfile == NULL)
		{
		    fopen_error(tfname, mailtext_oerr2);
		    return -2;
		}
	    }
	}
	else		/* Non-interactive: read message from stdin */
	    if (copy_txtfile(tfile, stdin, tstring, sizeof(tstring)) != 0)
	    {
		write_error(tfname, copying_msg);
		(void) fclose(tfile);			/* OK: giving up */
		return -2;
	    }

	if (tty && dotmenu)
	{
	    /*
	     * ST-FIXME : this whole "dot" menu stuff is a bit silly.  Is
	     * anybody still using it?  If so, why???  It's a relic of the old
	     * BSD days when people used to enter mail without using a proper
	     * editor; I really can't see anyone doing *that* these days.  We
	     * could make things simpler (and smaller) by removing it
	     * altogether.
	     */
	    /* <tfile> is open at this point */

	    textattr(col_enhanc);
	    cprintf("\rType message text.  Enter a '%c' "
		    "on a line by itself to end.\r\n", escape_char);
	    cprintf("Commands: %cp - redisplay msg, %ce - invoke editor, "
		    "%c? - help\r\n\n", escape_char, escape_char, escape_char);
	    textattr(col_normal);

	    ch = '\0';
	    while (TRUE)
	    {
		/* read line from console i.e. stdin */
		/* GBD allow wrapping after display.screenwidth chars */
		tstring[0] = ch;
		tstring[1] = '\0';
		ch = maxgets(tstring, display.screenwidth);
		if (strlen(tstring) < display.screenwidth)
		    cputs("\r\n");
		if (tstring[0] == escape_char && tstring[1] == '\0')
		    break;
		if (*tstring == escape_char)	/* Handle escape command */
		{
		    switch (tstring[1])
		    {
		    case 'p':		/* Print the message so far */
			printf("\r");	/* so the tabs work */
			rewind(tfile);
			while (fgets(tstring, sizeof(tstring), tfile) != NULL)
			    (void) fputs(tstring, stdout);	/*OK*/
			break;

		    case 'e':		/* Drop into editor */
			if ((tfile = reedit_mail(tfile, to, subject, ct_type,
						 ct_enc, header)) == NULL)
			    return -2;		/* it's all gone wrong */
			header = 1;		/* header is now in <tfile> */
			break;

		    case 'q':		/* Quit */
			(void) fclose(tfile);		/* OK: giving up */
			return -1;

		    case 'r':
			{
			    FILE  *ninfl;	/* was 'infl'; clashed -ST */

			    p = skip_lwsp(tstring + 2);
			    if (*p == '\0')
				colputs(col_hilite,
					"No file name specified\r\n");
			    else
			    {
				struct ffblk ffblk;

				if (strpbrk(p, "?*") != NULL)
				{
				    if (findfirst(p, &ffblk,
						  FILENAME | WILDCARDS))
					cprintf("No files %s found\r\n", p);
				    else
				    {
					cprintf("Matching %s:\r\n", p);
					textattr(col_enhanc);
					do
					    cprintf("%-16s", ffblk.ff_name);
					while (!findnext(&ffblk));
					textattr(col_normal);
				    }
				    break;
				}
				if ((ninfl = fopen(p, "rb")) == NULL)
				    colputs(col_hilite, "No such file\r\n");
				else
				{
				    char buff[LINELEN + 1], *ctrlz;

				    cprintf("Reading file %s\r\n", p);
				    while (fgets(buff, sizeof(buff), ninfl)
					   != NULL)
				    {
					(void) rip(buff);
					if ((ctrlz = strchr(buff, '\x1a')) != NULL)
					    *ctrlz = '\0';
					if (fputs(buff, tfile) == EOF ||
					    fputs("\n", tfile) == EOF)
					    break;	/* ferror() at end */
				    }
				    (void) fclose(ninfl);  /* OK: read only */
				}
			    }
			    break;
			}

		    case 'm':
			{
			    extern PHTL currmsg;	/* Current message */
			    PHTL cmsg;
			    int msg = atoi(skip_lwsp(tstring + 2));

			    if (msg == 0)
				cmsg = currmsg->next;
			    else
			    {
				for (cmsg = mbox;
				     cmsg != NULL && cmsg->listno != msg;
				     cmsg = cmsg->next)
				    ;
			    }
			    if (mfile == NULL || cmsg == NULL)
				colputs(col_hilite, "No such message\r\n");
			    else
			    {
				textattr(col_hilite);
				cprintf("Inserting message %d\r\n",
					cmsg->listno);
				textattr(col_normal);
				if (savemsg(cmsg, tfile, HW_NONE, SM_NONE) > 0)
				    write_error(tfname, copying_msg);
			    }
			    break;
			}

		    case '?':
			textattr(col_enhanc);
			cprintf("%ce      - Invoke Editor\r\n",
				escape_char);
			cprintf("%cp      - Display message buffer\r\n",
				escape_char);
			cprintf("%cq      - Abort this message\r\n",
				escape_char);
			cprintf("%cr file - Read file into buffer\r\n",
				escape_char);
			cprintf("%cm msg  - message into buffer\r\n",
				escape_char);
			cprintf("%c%c      - Enter a %c into message\r\n",
				escape_char, escape_char, escape_char);
			textattr(col_normal);
			break;

		    default:
			if (tstring[1] == escape_char)
			{
			    (void) fprintf(tfile, "%s\n", &tstring[1]);
			    /* ferror() check after goto */
			    goto endswitch;
			}

			textcolor(col_enhanc);
			cprintf("Unknown %c escape. %c? for help\r\n",
				escape_char, escape_char);
			textcolor(col_normal);
		    }	/* switch (tstring[1]) */
		    colputs(col_normal, "\r\n(continue)\r\n");
		}
		else		/* Not an escaped command: copy string */
		{
		    (void) rip(tstring);
		    (void) fprintf(tfile, "%s\n", tstring);
		    /* ferror() check below */
		}
	    endswitch:
		if (ferror(tfile))
		{
		    write_error(tfname, "escaped char");
		    break;	/* out of the while (TRUE) */
		}
	    }		/* while (TRUE) */
	    colputs(col_enhanc, "EOF\r\n");
	}
	else if (tty)
	{
	    /* no dot menu: just "send, edit, abort" */
	    int key;
	    BOOL done = FALSE;

	    /*
	     * We need to show the send/edit/abort prompt in a loop, because
	     * it may get overwritten by an errordsp() or similar if something
	     * goes awry during editing; if we only display the prompt once,
	     * and just keep looping, it can get quite confusing if the prompt
	     * gets overwritten!
	     */
	    while (!done)
	    {
		gotoxy(1, display.screenheight - 1);
		clreos();
		cputs("Enter s to send, e to edit or a to abandon the message: ");
		while (TRUE)
		{
		    if ((key = (int) getkey()) == 0)	/* nothing pressed */
			continue;	/* with the while (TRUE) */
		    switch (key)
		    {
		    default:
			beep();
			continue;	/* with the while (TRUE) */

		    case 'a':		/* abort */
		    case 'A':
			if (cnfmsendabort && !y_n("Really abort message?"))
			    break;
			(void) fclose(tfile);		/* OK: giving up */
			return -1;

		    case 'e':		/* re-edit */
		    case 'E':
			if ((tfile = reedit_mail(tfile, to, subject, ct_type,
						 ct_enc, header)) == NULL)
			    return -2;		/* it's all gone wrong */
			header = 1;		/* header is now in <tfile> */
			break;

		    case 's':		/* send */
		    case 'S':
			done = TRUE;
			break;
		    }
		    break;	/* out of the while (TRUE) */
		}
	    }
	    /* We only get here when the user has pressed 's' */
	}

	if (append_signature(tfile) != 0)	/* ensure sig. is appended */
	{
	    write_error(tfname, sig_werr);
	    return -2;
	}

	/*
	 * If the header lines have been written to the file already, look
	 * through them to see if we've already got MIME settings (at present,
	 * just a MIME-Version line, although this will need to be rather more
	 * comprehensive at some stage: ST_FIXME); we'll probably need this
	 * information later.
	 */
	rewind(tfile);
	found_mime = FALSE;
	if (header != 0)	/* We've got header lines to scan in <tfile> */
	{
	    BOOL mimever = FALSE;
	    char last_eol = 1;

	    while (fgets(tstring, sizeof(tstring), tfile) != NULL)
	    {
		if (last_eol && ISBLANKLINE(tstring))	/* End of header */
		    break;	/* out of the while (fgets) */
		last_eol = eoltest(tstring);
		if (!mimever && htype_match(tstring, HT_MIME_VERSION))
		    mimever = TRUE;
	    }
	    if (mimever)	/* Saw a MIME-Version: header field */
		found_mime = TRUE;
	}

	/*
	 * We've scanned to the start of the message body; if it's not already
	 * marked for MIME encoding, examine it to see if we need to encode it
	 * (because of binary/control characters, or overlong lines); if we
	 * do, force quoted-printable ISO 8859-1, which is the best for text
	 * and is OK for anything (not perfect for binaries, but it will
	 * work).
	 */
	if (enablemime && !ismime)		/* Check for binary chars */
	{
	    FILETYPE ftype = get_filetype(tfile, tstring, sizeof(tstring),
					  TRUE);

	    if (ftype == FT_ASCLONG || ftype == FT_ACZLONG)
	    {				/* Long lines, otherwise ASCII */
		/* User may want not to encode long lines, or to be asked */
		if (enclong == 0 ||	/* Never encode FT_ASCLONG */
		    (enclong == 2 &&	/* Prompt user about encoding */
		     y_n("Message has over-long lines. "
			 "Encode with quoted-printable?") == FALSE))
		    ftype = FT_ASCSHRT;		/* Avoid encoding it */
	    }
	    if (ftype != FT_ASCSHRT && ftype != FT_ACZSHRT)
	    {
		*send_file = '\0';	/* MIME sending, but not a file */
		ismime = TRUE;
		ct_type = CT_TEXT_PLN88591;
		ct_enc = CE_QUOTED;
	    }
	}

	/*
	 * If we've already written a non-MIME header to the file and it needs
	 * to be MIMEd, set <header> to -1 to force the addition of MIME stuff
	 */
	if (header && !found_mime && ismime)
	    header = -1;	/* add in header MIME lines if necessary */

	/* else if we haven't put in headers and it will all work */
    }

    if (fclose(tfile) != 0)
    {
	write_error(tfname, closing_file);
	return -2;
    }

    /*
     * A final call to add_headers() to make sure we've got everything set up
     * properly; if <header> is 1, the header has already been added to the
     * file and all is well; but if it's 0, the header hasn't been added at
     * all; and if it's -1, a non-MIME header has been added, but it needs to
     * have MIME bits put in (and the body may need encoding, too).
     */
    if (!add_headers(header, ismime, ct_type, ct_enc))	/* Aargh! */
	return -2;		/* Failed; error message already shown */

    return (ismime) ? 1 : 0;
}


/*
 * reedit_mail()
 *
 * Re-edit the mail message currently available on <tfile>; <to> and <subject>
 * are its "To:" and "Subject:" field bodies respectively, <ct_type> and
 * <ce_type> are its MIME type and encoding, and <header> is an indicator for
 * whether the message header in already part of the file or not (0 if not).
 *
 * Ensuring that (a) the message's signature has been added and (b) the
 * message header is part of the file, close the file stream and call
 * invoke_editor() to (re)edit the file.  If all goes well, return the
 * freshly-reopened stream for the file; if not, put up an error message
 * describing the problem and return NULL.
 */
static FILE *reedit_mail(FILE *tfile, const char *to, const char *subject,
			 CTTYPE ct_type, CETYPE ct_enc, int header)
{
    int err = 0;

    if (append_signature(tfile) != 0)
	err = 1;
    if (fclose(tfile) != 0)
	err = 2;
    tfile = NULL;		/* <tfile> is now closed */
    if (err == 0)		/* OK to run editor */
    {
	if (header == 0 && !add_headers(header, FALSE, ct_type, ct_enc))
	    err = -1;		/* failed to add header */
	else
	    tfile = invoke_editor(NULL, to, subject, TRUE, TRUE);
    }
    if (tfile == NULL)		/* Error of some kind */
    {
	if (err > 0)			/* A write error */
	    write_error(tfname, sig_werr);
	else if (err == 0)		/* fopen() error in invoke_editor() */
	    fopen_error(tfname, mailtext_oerr2);
	/* else err < 0, add_headers() failed: message already shown */
    }
    return tfile;
}


/*
 * build a command line. Supported Macros are
 * $ filename
 * % receiver
 * & subject
 *
 * This needs to be rewritten completely, (a) to remove code duplication, and
 * (b) deal with multiple occurrences of $, % and &.  ST-FIXME!
 */
char *buildcmd(const char *cmdline, const char *percent, const char *amper,
	       const char *dollar)
{
    static char cmd[LINELEN + 1];
    char *s;
    size_t lendollar, lenpercent, lenamper;

    memset(&cmd[0], 0, sizeof(cmd));
    lendollar = strlen(dollar);
    lenamper = strlen(amper);
    lenpercent = strlen(percent);
    if ((strlen(cmdline) + lenamper + lendollar + lenpercent) >= sizeof(cmd))
    {
	/* command too long for DOS commandline */
	return NULL;
    }

    /*
     * Lengths check out OK -- we've got room to add one dollar, one percent
     * and one amper without running out of space in <cmd>.  So we can go for
     * it without worrying about checking for buffer overflows... famous last
     * words 8-}
     */
    strcpy(cmd, cmdline);			/* OK -ST */
    /* insert mailfile name */
    if ((s = strchr(cmd, '$')) != NULL)
    {
	movmem(s, s + lendollar - 1,  (unsigned) (&cmd[0] + strlen(cmd) - s));
	memcpy(s, dollar, lendollar);
    }
    /* insert receiver address */
    if ((s = strchr(cmd, '%')) != NULL)
    {
	movmem(s, s + lenpercent - 1, (unsigned) (&cmd[0] + strlen(cmd) - s));
	memcpy(s, percent, lenpercent);
    }
    /* insert subject */
    if ((s = strchr(cmd, '&')) != NULL)
    {
	movmem(s, s + lenamper - 1, (unsigned) (&cmd[0] + strlen(cmd) - s));
	memcpy(s, amper, lenamper);
    }
    return cmd;
}


/*
 * dosmtpsend()
 *
 * This function need to be ripped apart and completely rewritten, preferably
 * in lots of little bits.  It is a complete monster.  Urgh.  ST-FIXME!
 *
 * <mfp> is an open file which is to be included/quoted in the mail message:
 * it may be NULL, in which case we're just creating a plain message; if not,
 * it is either a file we're intending to send to someone, a message which
 * we're replying to, or a message we're forwarding (if <action> is
 * SEND_FWDQUOTE).  In any case, it just gets passed through to
 * create_mailbody() without any intervention from us.  <to> is the basic "To"
 * list of recipients for the message, <subject> is the subject and <cc> is
 * the "Cc" list.  <ismime> gets passed through to create_mailbody() to
 * indicate whether to create a MIME body or not; <ct_type> and <ct_enc> also
 * get passed to create_mailbody().
 *
 * Return 0 if everything works (ha ha), something else if not.
 */
int dosmtpsend(FILE *mfp, const char *to, const char *subject, const char *cc,
	       BOOL ismime, CTTYPE ct_type, CETYPE ct_enc, SENDTYPE action)
{
    char prefix[LINELEN], smtp_to[LLINELEN + 1], tfilename[LINELEN + 1],
	wfilename[LINELEN + 1], folder_save_name[LINELEN], temp[LLINELEN];
    char *p, *op;
    long sequence = -1L;
    FILE *tfile, *wfile, *ufile;
    BOOL inheader;
    int i, err;
    HEADERTYPE hdrtype;
    int smartsend;
    char *addrp;
    size_t addrlen;
    BOOL writerr = FALSE;
    time_t sendtime;
    size_t l;
    char last_eol;

    /*
     * Allocate storage for our "full" expanded lists of recipients etc.  This
     * whole bit is broken, as there is no guarantee that our lists will fit
     * inside the fixed-size arrays; we need a better solution.  ST-FIXME
     *
     * Make sure all the dynamic arrays are free, set to NULL, and zero
     * length, before we start.
     */
    (void) free_dosmtp_ptrs();		/* ensure all are free and NULL */
    if ((reciprest = malloc(reciprest_size = 4 * LLINELEN + 1)) == NULL)
	return free_dosmtp_ptrs();
    if ((expanded_to = malloc(expanded_to_size = LLINELEN + 1)) == NULL)
	return free_dosmtp_ptrs();
    if ((expanded_cc = malloc(expanded_cc_size = LLINELEN + 1)) == NULL)
	return free_dosmtp_ptrs();
    if ((expanded_bcc = malloc(expanded_bcc_size = LLINELEN + 1)) == NULL)
	return free_dosmtp_ptrs();
    *expanded_bcc = '\0';

    /*
     * Expand the initial "To" list for aliases, then copy it into the full
     * list of recipients in <reciprest>
     */
    PUSTRCPY(expanded_to, expanded_to_size, to);	/* save "To" */
    if (expand_alias(expanded_to, expanded_to_size))
	return free_dosmtp_ptrs();	/* expansion was too long ... */
    PUSTRCPY(reciprest, reciprest_size, expanded_to);

    if (cc && *cc)		/* expand the initial "Cc" list */
    {
	PUSTRCPY(expanded_cc, expanded_cc_size, cc);	/* save "Cc" */
	if (expand_alias(expanded_cc, expanded_cc_size))
	    return free_dosmtp_ptrs();	/* expansion was too long ... */
    }
    else			/* make expanded "Cc" list a null string */
	*expanded_cc = '\0';

    /*
     * We now have the unexpanded "To" and "Cc" names in <to> && <cc>, and the
     * expanded names in <expanded_to> && <expanded_cc>; those two expanded
     * strings are put together to make <reciprest>, which gives a list of
     * *all* recipients - this is done later.
     *
     * We used to take a separate copy of <subject> here, for no readily
     * apparent reason: the resulting buffer was never changed, so why bother?
     */
    if (subject == NULL)
	subject = no_subj_spec;

    /*
     * We know enough to create the headers so we shall do that first
     * into a file called header.tmp which should be in wherever TMP
     * or TEMP points at - tidy this later by adding a tempdir parameter
     * to PCOAK.RC (same way as SNEWS) and storing temporary files
     * throughout the suite in that directory.  FIXME!
     *
     * (Actually, I disagree: putting HEADER.TMP in the current directory,
     * along with MAILTEXT.TXT, makes it easy for the user to get at; and
     * having them both in the same directory makes the ensuing
     * renaming/appending stuff easier. -ST)
     */
    if (create_header_file(expanded_to, expanded_cc, expanded_bcc, subject))
    {
	clear_up();
	return free_dosmtp_ptrs();	/* header creation failed */
    }

    /*
     * Utterly redundant "while (TRUE)" loop (whose last statement was
     * "break"!) removed; it appears to have been part of a PCElm attempt to
     * let the user have another go at editing the message in the event of a
     * disk or file error, but in true PCElm tradition it didn't work -- if an
     * error occurred, PCElm used to wait for 3 seconds and then carry on with
     * the normal contents of the loop as though nothing was amiss, and would
     * eventually hit the "break" at the end of the loop.
     */

    (void) append_signature(NULL);	/* Reset signature writer */
    i = create_mailbody(mfp, to, subject, ismime, ct_type, ct_enc, action);
    if (i < 0)			/* Error, or aborted */
    {
	clear_up();		/* clear up mail files */
	return free_dosmtp_ptrs();
    }
    ismime = (i != 0);

    /*
     * We've now got the (almost) complete mail message, with header, in the
     * global file <tfname>.  We now need to construct the *complete* version,
     * including expanded aliases in the recipient fields and a "Date:" field;
     * this complete file can then be used as the template for creating the
     * *.TXT queue files.
     *
     * Back up the current file <tfname> to <wfilename>, then we can build the
     * new file in <tfname>.  Give up if we can't rename the file.  Once we've
     * done this, open the backup file and the new destination file, then
     * write the Date: header field (showing the time at which the message was
     * committed for sending), and finally copy across the existing message
     * (excluding any blank header lines, expanding recipient fields, and
     * folding any over-long lines where possible).
     *
     * TODO: create a copy of this initial file, if the user wants one
     */
    if (file_backup(tfname, wfilename, sizeof(wfilename)))
	return free_dosmtp_ptrs();
    if ((wfile = fopen(tfname, "w")) == NULL)
    {
	fopen_error(tfname, mailtext_oerr);
	return free_dosmtp_ptrs();
    }
    if ((tfile = fopen(wfilename, "r")) == NULL)
    {
	(void) fclose(wfile);				/* OK: unused */
	fopen_error(wfilename, backup_oerr);
	return free_dosmtp_ptrs();
    }

    *reciprest = '\0';
    *expanded_to = '\0';
    *expanded_cc = '\0';
    *expanded_bcc = '\0';

    inheader = TRUE;
    last_eol = 1;
    (void) fgets_msg(NULL, 0, NULL, FALSE);	/* Reset fgets_msg() store */

    /* Write "Date:" header field, timestamped now, at start of header */
    sendtime = time(NULL);		/* time of sending */
    if (fprintf(wfile, "%s %s\n", headername[HT_DATE], rfcdate(sendtime))
	== EOF)
	writerr = TRUE;

    while (!writerr && fgets_msg(temp, sizeof(temp), tfile, inheader))
    {
	if (inheader && last_eol && ISBLANKLINE(temp))	/* End of header */
	    inheader = FALSE;
	if (inheader && (op = strchr(temp, ':')) != NULL)
	{		/* Header field: should be <name>:<lwspace><body> */
	    last_eol = rip(temp);	/* remove any trailing CRLF */
	    p = skip_lwsp(++op);	/* p = start of header body */
	    if (*p == '\0')
		continue;	/* with while (): discard blank header field */
	    hdrtype = htype(temp);
	    if (hdrtype == HT_TO)		/* "To:" field */
	    {
		PUSTRCPY(expanded_to, expanded_to_size, p);
		if (expand_alias(expanded_to, expanded_to_size))
		    return free_dosmtp_ptrs();  	/* too long ... */
		PUSTRCPY(p, sizeof(temp) - (size_t) (p - temp), expanded_to);
	    }
	    else if (hdrtype == HT_CC)		/* "Cc:" field */
	    {
		PUSTRCPY(expanded_cc, expanded_cc_size, p);
		if (expand_alias(expanded_cc, expanded_cc_size))
		    return free_dosmtp_ptrs();  	/* too long ... */
		PUSTRCPY(p, sizeof(temp) - (size_t) (p - temp), expanded_cc);
	    }
	    else if (hdrtype == HT_BCC)		/* "Bcc:" field */
	    {
		PUSTRCPY(expanded_bcc, expanded_bcc_size, p);
		if (expand_alias(expanded_bcc, expanded_bcc_size))
		    return free_dosmtp_ptrs();  	/* too long ... */
		continue;	/* with while(): don't include in message */
	    }
	    l = (size_t) (op - temp);	/* length of name, inc. ':' */
	    if (l >= sizeof(prefix))	/* name too long for buffer! */
		l = sizeof(prefix) - 1;
	    (void) ustrncpy(prefix, temp, l);
	    if (split_write(wfile, prefix, temp + l))
		writerr = TRUE;
	}
	else
	{
	    if (inheader)
		last_eol = eoltest(temp);
	    if (fputs(temp, wfile) == EOF)
		writerr = TRUE;
	}
    }		/* while (!writerr && fgets_msg) */

    (void) fclose(tfile);			/* OK: read only */
    if (fclose(wfile) != 0)		/* error closing file */
	writerr = TRUE;
    if (writerr)	/* Error while writing file */
    {
	write_error(tfname, "writing mailtext file");
	return free_dosmtp_ptrs();
    }

    /*
     * Build the full recipients list in <reciprest>; this is a concatenation
     * of <expanded_to>, <expanded_cc> and <expanded_bcc>
     */
    if (*expanded_to)
	PUSTRCPY(reciprest, reciprest_size, expanded_to);
    if (*expanded_cc)
	PUSTRLCAT2(reciprest, reciprest_size, " ", expanded_cc);
    if (*expanded_bcc)
	PUSTRLCAT2(reciprest, reciprest_size, " ", expanded_bcc);

    if (tty)
	cprintf("\r");	/* move to the start of the *current* line */

    /*
     * If we have to save a copy of this outgoing message (saveall > 0), work
     * out where to save the message now (since we may destroy reciprest in
     * the next process) and copy the folder to use into <folder_save_name>;
     * if saveall == 1, we use "sent", which is also what we use if we're
     * unable to extract a usable DOS filename from the first recipient's mail
     * address.
     */
    if (saveall > 0)		/* Find folder to save copy into */
    {
	if (saveall == 1 ||
	    get_addr_filename(folder_save_name, sizeof(folder_save_name),
			      reciprest) == 0)
	    strcpy(folder_save_name, str_sent);			/* OK */
    }

    p = reciprest;
    /*
     *	Rather than consume even more stack, we'll borrow the buffers
     *	already defined for 'smtp_to' and 'temp' (both LLINELEN + 1).
     *	They will immediately be rewritten anyway in the next line of
     *	code.  Of course 'check_recipients' functionality could be
     *	inline code here to avoid this mess rather than a sep function
     *	but that would make this over-long function even longer. Both
     *	options are horrid, and the whole lot needs rewriting! -PJD
     */
    smartsend = check_recipients(p, temp, sizeof(temp));
    while ((p = next_address(temp, sizeof(temp), p, &addrp, &addrlen)) != NULL)
    {
	if (addrp == NULL || addrlen == 0)	/* No SMTP address found */
	    continue;		/* skip the processing; on to next recip. */

	/* copy the SMTP address from <temp> into <smtp_to> */
	ustrncpy(smtp_to, addrp, min(addrlen, sizeof(smtp_to) - 1));

	/*
	 * Now send mail to all receivers.  If more than one copy is going to
	 * the same domain, we only create one .WRK file for that domain with
	 * multiple recipient lines.  Do this by working up the receivers
	 * string finding any matches, use them and remove them from the
	 * string. This code is at the end of this function just before the
	 * .WRK file is closed.
	 */
	sequence = get_msgid();		/* get next sequence number */
	if (tty)			/* announce what we are doing */
	{
	    textcolor(col_enhanc);
	    cprintf(sending_fmt, smtp_to);
	    textcolor(col_normal);
	}

	if (used_mailsystem == SYS_UUCP)	/* take it easy, use rmail! */
	{
	    if ((ufile = fopen(mfname, "w+")) == NULL)
	    {
		colputs(col_hilite,
			"OOPS! Can't create file 'UUCPMAIL.$$$'\r\n");
	    }
	    else
	    {
		err = 0;
		/* Do the header fprintf()s blind; ferror() check at end */
		(void) fprintf(ufile, "From %s remote from %s\n", useraddr,
			       hostname);
		(void) fprintf(ufile, "Message-Id: <%ld@%s>\n", sequence,
			       hostname);
		(void) fprintf(ufile, "From: %s@%s", useraddr, hostname);
		if (*fullname)
		    (void) fprintf(ufile, " (%s)", fullname);
		(void) fprintf(ufile, "\n");
		if (uucpreplyto)
		    (void) fprintf(ufile, "Reply-To: %s\n", uucpreplyto);
		else if (replyto)
		    (void) fprintf(ufile, "Reply-To: %s\n", replyto);
		(void) fprintf(ufile, "To: %s\n", smtp_to);
		if (*expanded_cc)
		    (void) fprintf(ufile, "Cc: %s\n", expanded_cc);
		(void) fprintf(ufile, "Subject: %s\n", subject);
		for (i = 0; i < MAXOPTHDR; i++)
		    if (opthdr[i] != NULL && *opthdr[i])
			(void) fprintf(ufile, "%s\n", opthdr[i]);
		(void) fprintf(ufile, "X-Mailer: %s\n\n", myversion);
		if (ferror(ufile))
		    err = 1;
		else
		    err = copy_message(ufile, sequence);
		if (fclose(ufile) != 0 && err == 0)
		    err = 3;
		if (err != 0)		    /* Error: give up now */
		{
		    if (err > 0)	/* write error: need message */
			write_error(mfname, "writing UUCP message file");
		    unlink(mfname);
		}
		else		/* OK so far */
		{
		    char *uucpcall;

		    chdir(cwd);
		    if ((uucpcall = buildcmd(uucp_command, smtp_to,
					     subject, mfname)) != NULL)
		    {
			backslash(uucpcall);
#if 0
			printf(">Calling >%s<\r\n", uucpcall);
			getchar();
			sleep(1);
#endif
			if (disp_system(uucpcall, FALSE))
			{
			    textcolor(col_hilite);
			    cprintf("Error spawning mailer. "
				    "Do a\r\n%s\r\n to send the mail.\r\n",
				    uucpcall);
			    textcolor(col_normal);
			    sleep(4);
			}
			else
			{
			    chdir(cwd);
			    unlink(mfname);
			    sleep(1);
			}
		    }
		    else
		    {
			textcolor(col_system);
			disp_clear();
			(void) fputs("Can't call rmail, aborting ...", stderr);
			(void) fcloseall();
			exit(1);
		    }
		}
	    }
	} /* end-if UUCP */

	else	/* used_mailsystem is SYS_HAM (SMTP) */
	{
	    sprintf(prefix, "%ld", sequence);				/*OK*/
	    if (mlock(mailqdir, prefix))
	    {
		clear_up();
		return free_dosmtp_ptrs();
	    }

	    /*
	     * Create the .TXT file in mqueue directory: the mail file
	     * contains all the headers and the body, but we need to put the
	     * correct message-id in
	     */
	    AUSTRLCPY4(tfilename, mailqdir, "\\", prefix, mboxext);
	    if ((tfile = fopen(tfilename, "wt+")) == NULL)
	    {
		fopen_error(tfilename, "queue TXT file");
		rmlock(mailqdir, prefix);
		return free_dosmtp_ptrs();
	    }
	    err = copy_message(tfile, sequence);
	    if (fclose(tfile) != 0 && err == 0)
		err = 1;
	    if (err != 0)		/* Error: give up now */
	    {
		if (err > 0)		/* Write error: need message */
		    write_error(tfilename, "writing TXT file");
		/* else if (err < 0), error message already shown */
		unlink(tfilename);
		clear_up();
		rmlock(mailqdir, prefix);	/* erase the lock file */
		return free_dosmtp_ptrs();
	    }

	    /*
	     * Now create the .WRK file in the mqueue directory; <err> is
	     * currently 0, so we set it to non-zero on error
	     */
	    AUSTRLCPY4(wfilename, mailqdir, "\\", prefix, wrkfileext);
	    if ((wfile = fopen(wfilename, "w")) == NULL)
	    {
		fopen_error(wfilename, "queue WRK file");
		err = -1;		/* <err> -ve: don't need message */
	    }
	    else
	    {
		const char *tohost, *ntohost;
		char *cp, *ncp, *ap;
		size_t al;

		/*
		 * Find the mail host to send this mail to: may be the local
		 * host, or the smart host, or the domain part of the
		 * addr-spec.  Set <tohost> to this hostname.
		 */
		if ((tohost = strrchr(smtp_to, '@')) == NULL)
		    tohost = hostname;		/* local mail */
		else if (smartsend)
		    tohost = mailhost;		/* smart mail host */
		else
		    tohost++;			/* skip '@'; get the domain */

		/*
		 * Write out the main part of the workfile, including the
		 * first recipient for this domain
		 */
		(void) fprintf(wfile, "%s\n", tohost);
		(void) fprintf(wfile, "%s@%s\n", useraddr, hostname);
		(void) fprintf(wfile, "%s\n", smtp_to);

		if (!ferror(wfile))	/* OK to continue */
		{
		    /*
		     * Now run through all *remaining* recipients in the list,
		     * starting at out current position <p>, examining the
		     * SMTP address for each one; for each one whose host
		     * matches, add the recipient's address to the workfile
		     * and remove the entire recipient entry from the list.
		     */
		    for (cp = p; (ncp = next_address(temp, sizeof(temp), cp,
						     &ap, &al)) != NULL;
			 cp = ncp)
		    {
			if (ap == NULL || al == 0)  /* No SMTP addr: skip it */
			    continue;
			ap[al] = '\0';		/* truncate address */
			if ((ntohost = strrchr(ap, '@')) == NULL)
			    ntohost = hostname;	/* local */
			else
			    ntohost++;		/* skip '@'; get the domain */
			if (smartsend || stricmp(tohost, ntohost) == 0)
			{
			    /*
			     * Same destination host: add the address (ap) to
			     * the .WRK file, and remove the entire entry
			     * (cp -> ncp) from the reciprest string
			     */
			    if (fprintf(wfile, "%s\n", ap) == EOF)
				break;	/* out of the for (cp) */
			    if (tty)
			    {
				textcolor(col_enhanc);
				cprintf(sending_fmt, ap);
				textcolor(col_normal);
			    }
			    memmove(cp, ncp, strlen(ncp) + 1);
			    ncp = cp;		/* move <ncp> back as well */
			}
		    }
		}
		/* done.  <p> was not changed; it should still be correct */
		if (ferror(wfile))	/* Write error */
		    err = 1;		/* +ve <err>: needs message */
		if (fclose(wfile) != 0)
		    err = 1;
		if (err > 0)
		{
		    write_error(wfilename, "writing WRK file");
		    unlink(wfilename);
		}
	    }

	    rmlock(mailqdir, prefix);	/* erase the lock file */

	    if (err != 0)	/* Error occurred (message already shown) */
	    {
		unlink(tfilename);	/* Remove TXT file; WRK already gone */
		clear_up();
		return free_dosmtp_ptrs();
	    }

	    if (tty)
	    {
		textcolor(col_enhanc);
		cprintf("Message %ld done.", sequence);
		textcolor(col_normal);
	    }
	} /* end-if not UUCP */
    }	/* end of while */

    sleep(1);		/* How does this help?  ST-FIXME */

    clear_up();		/* Now delete the mailbody and the header temp file */

    /*
     * If <saveall> is set and we've sent at least one message (sequence >= 0)
     * we save the last copy of the message (helpfully, this one will be
     * <sequence>.TXT) as directed by the user.  We can only do this if we're
     * using SMTP mail; UUCP doesn't produce an appropriate .TXT file.  The
     * file is in the folder directory: ~\SPOOL\MAIL\<username>\<something>
     */
    if (saveall > 0 && sequence >= 0L && used_mailsystem == SYS_HAM)
    {
	BOOL done_status;

	/*
	 * Set the full pathname of the folder to use in <wfilename>, having
	 * first made sure the folder directory exists with a blind call to
	 * mkdir() [Yuck -ST]; if the folder doesn't already exist and
	 * <saveall> is 3, use the SENT folder instead
	 */
	folderpath(wfilename, sizeof(wfilename), NULL);  /* Folder directory */
	(void) mkdir(wfilename);
	AUSTRLCAT2(wfilename, "\\", folder_save_name);	/* Folder full path */
	if (saveall == 3 && access(wfilename, 0))	/* Not found: =sent */
	    folderpath(wfilename, sizeof(wfilename), str_sent);

	if ((wfile = fopen(wfilename, "at+")) == NULL)
	{			/* can't open textfile for append */
	    fopen_error(wfilename, "save file");
	    return free_dosmtp_ptrs();
	}
	/* <tfilename> will already be set to the most recent .TXT file */
	if ((tfile = fopen(tfilename, "r")) == NULL)	/*lint !e645*/ /* OK */
	{
	    fopen_error(tfilename, "mail file");
	    (void) fclose(wfile);			/* OK: unused */
	    return free_dosmtp_ptrs();
	}

	/* Write the message: begin with system Start Of Header line */
	/* Write it blind, and check ferror() below */
	err = 0;
	(void) fprintf(wfile, "%s%s@%s", headername[HT_SOH], useraddr,
		       hostname);
	if (*fullname)
	    (void) fprintf(wfile, " (%s)", fullname);
	(void) fputs("\n", wfile);
	done_status = FALSE;
	last_eol = 1;
	while (fgets(temp, sizeof(temp), tfile) != NULL)
	{
	    if (!done_status)
	    {
		if (last_eol && check_status(temp, sizeof(temp),
					     wfile, FALSE) != 0)
		    done_status = TRUE;
		else
		    last_eol = eoltest(temp);
	    }
	    if (done_status && htype_match(temp, HT_SOH))
		(void) fputs(">", wfile);	/* Protect "From " line */
	    (void) fputs(temp, wfile);
	    if (ferror(wfile))
		break;	/* out of the while (fgets) */
	}
	/* Ensure the Status: header line is written */
	if (!done_status)		/* Nope: message with no body? */
	    (void) write_status_line(wfile, FALSE);
	if (ferror(wfile))
	    err = 1;
	if (fclose(wfile) != 0)
	    err = 1;
	(void) fclose(tfile);				/* OK: read only */
	if (err != 0)
	{
	    write_error(wfilename, "writing save file");
	    return free_dosmtp_ptrs();
	}
	sleep(1);	/* And how does this help?  ST-FIXME */
    }

    (void) free_dosmtp_ptrs();
    return 0;
}


/*
 * backslash()
 *
 * replace all '/' with '\' in the passed string.  OK to pass a NULL pointer!
 */
void backslash(char *s)
{
    if (s)
	while (*s != '\0')
	{
	    if (*s == '/')		/* replace slash with backslash */
		*s = '\\';
	    s++;
	}
}


/*
 * cbackslash()
 *
 * Replace all '/' with '\' in the passed string, stopping at the first
 * whitespace character (so we can backslash a command name, without
 * corrupting any /option strings later in the command string).  OK to pass a
 * NULL pointer!
 */
void cbackslash(char *s)
{
    if (s)
	while (*s != '\0' && !isspace(*s))
	{
	    if (*s == '/')		/* replace slash with backslash */
		*s = '\\';
	    s++;
	}
}


/*
 * fwdslash()
 *
 * replace all '\' with '/' in the passed string.  OK to pass a NULL pointer!
 */
void fwdslash(char *s)
{
    if (s)
	while (*s != '\0')
	{
	    if (*s == '\\')		/* replace backslash with fwdslash */
		*s = '/';
	    s++;
	}
}


/*
 * clear_up()
 *
 * Clear up file debris from a mail message editing session: delete any backup
 * file for <tfname> which may be hanging around, then delete <tfname> and the
 * header file <header_file> (which may not exist, but we do it anyway)
 */
static void clear_up()
{
    char *bakfile = strdup(tfname);

    if (bakfile != NULL)
    {
	backup_name(bakfile);		/* find the backup name for <tfname> */
	unlink(bakfile);		/* delete the backup file */
	free(bakfile);
    }
    unlink(tfname);
    unlink(header_file);
}


/*
 * check_recipients()
 *
 * Return TRUE if we should use the smart mailhost, FALSE if not.  If the
 * user's smarthost level is 2, we need to examine the list of recipients <s>
 * and return FALSE if there is only one destination domain, TRUE if more than
 * 1.
 */
static int check_recipients(const char *s, char *scratch, size_t scratchsize)
{
    char first_to[LINELEN];	/* is this big enough? (should be -ST) */
    char *ap;
    size_t al;

    if (smarthost == 0)		/* Never use smart mailhost */
	return FALSE;

    if (smarthost == 1)		/* Always use smart mailhost */
	return TRUE;

    /*
     *	else smarthost == 2: investigate recip list in 's'.
     *	if only a single recipient OR _all_ the destination
     *	domains are the same, return FALSE.
     *	otherwise return TRUE (more than 1, diff domain).
     */
    while ((s = next_address(scratch, scratchsize, s, &ap, &al)) != NULL &&
	   (ap == NULL || al == 0))
	;

    if (s != NULL)		/* Got the first SMTP address */
    {
	ap[al] = '\0';		/* truncate the SMTP address */
	if ((ap = strrchr(ap, '@')) != NULL)
	    ++ap;
	else
	    ap = hostname;
	AUSTRCPY(first_to, ap);	/* copy first hostname into <temp> */

	while ((s = next_address(scratch, scratchsize, s, &ap, &al)) != NULL)
	{
	    if (ap != NULL && al > 0)
	    {
		ap[al] = '\0';		/* truncate SMTP address */
		if ((ap = strrchr(ap, '@')) != NULL)
		    ++ap;
		else
		    ap = hostname;
		if (stricmp(ap, first_to) != 0)
		    return TRUE;
	    }
	}
    }

    return FALSE;
}


/*
 * backup_name()
 *
 * Convert <origname> into a backup file name (make the last character a 'k')
 */
static void backup_name(char *origname)
{
    origname[strlen(origname) - 1] = 'k';
}


/*
 * file_backup()
 *
 * Generate a backup file name for file <fname> and copy it into <bkname>,
 * then delete any file already existing called <bkname> and rename <fname> to
 * <bkname>.  Return 0 if all this goes well, something else if not (having
 * alerted the user appropriately).
 */
static int file_backup(const char *fname, char *bkname, size_t bksize)
{
    PUSTRCPY(bkname, bksize, fname);	/* copy of main file name */
    backup_name(bkname);		/* make backup file name */
    if (access(bkname, 0) != -1)	/* bkname already exists */
	unlink(bkname);				/* try to unlink it */
    if (rename(fname, bkname))		/* can't rename it */
    {
	rename_error(fname, bkname);
	return 2;
    }
    return 0;
}


/*
 * get_addr_filename()
 *
 * Given a list of addresses (<addresslist>), parse the first SMTP address we
 * can find to produce a filename which may be used to save the mail; copy
 * this filename into <into>, which is of size <intosize>.  Return the length
 * of valid DOS filename we manage to produce (0 if we can't make one).
 *
 * Note that even though '@' is actually a valid DOS filename character, we
 * don't allow it.
 */
int get_addr_filename(char *into, size_t intosize, const char *addr)
{
    char *ap;
    size_t al;

    intosize--;		/* reduce to max. string length possible */

    while ((addr = next_address(NULL, 0, addr, &ap, &al)) != NULL &&
	   (ap == NULL || al == 0))
	;

    if (addr != NULL)		/* Got an address at <ap> */
    {
	ustrncpy(into, ap, min(al, intosize));
	/* truncate <into> at first illegal filename char (inc. '@') */
	into[strcspn(into, " \t?*\\/:=\"+,;<>[]|@")] = '\0';
	if ((ap = strchr(into, '.')) == NULL)	/* no '.' */
	    into[8] = '\0';
	else			/* got a '.' -- must be in valid position */
	{
	    if ((ap - into) > 8)		/* basename too long */
		into[8] = '\0';
	    else
	    {
		if (strlen(ap + 1) > 3)		/* extension > 3 chars */
		    ap[4] = '\0';
		if ((ap = strchr(ap + 1, '.')) != NULL)	/* second '.' */
		    *ap = '\0';
	    }
	}
    }
    else
	*into = '\0';
    return (int) strlen(into);
}


/*
 * get_filetype()
 *
 * Read as much of the file as required to determine its type, and return that
 * type:
 *  FT_ASCSHRT : pure ASCII text, with lines up to MAX_BODYLINE chars
 *  FT_ASCLONG : pure ASCII text, with lines over MAX_BODYLINE chars
 *  FT_ACZSHRT : ASCII text ending Ctrl-Z, with lines up to MAX_BODYLINE chars
 *  FT_ACZLONG : ASCII text ending Ctrl-Z, with lines over MAX_BODYLINE chars
 *  FT_TEXT    : mainly ASCII text, but has some binary (8-bit/control) chars
 *  FT_BINARY  : contains many binary (8-bit/control) chars
 *
 * If <istext> is TRUE, we are looking for a text type: return FT_A??SHRT,
 * FT_A??LONG or FT_TEXT only.
 */
static FILETYPE get_filetype(FILE *fp, char *buff, size_t bufsize, BOOL istext)
{
    FILETYPE ft;
    size_t n, llen;		/* 'bytes read' count, line length */
    unsigned long ntot, cb;	/* total bytes read, count of binary chars */
    unsigned long cz;		/* count of trailing Ctrl-Zs seen */
    const char *p, *e;		/* working buf ptr, ptr to end of buff */

#if 0
    /* this should be safe; we haven't accessed 'fp' yet */
    setvbuf (fp, NULL, _IOFBF, BIGBUF);
#endif

    /* Assume FT_ASCSHRT until proved otherwise */
    ft = FT_ASCSHRT;
    llen = 0;
    ntot = cb = cz = 0L;
    while ((n = fread(buff, 1, bufsize, fp)) > 0)
    {
	ntot += n;
	for (p = buff, e = buff + n; p < e; p++)
	{
	    if (*p == '\032')		/* Ctrl-Z; just count it for now */
	    {
		++cz;
		continue;	/* with the for (p) */
	    }
	    else if (cz > 0L)		/* Seen ^Z, but not at end of file! */
	    {
		if (istext)
		    return FT_TEXT;
		cb += cz;
		cz = 0L;
		ft = FT_TEXT;
	    }

	    if ((*p >= ' ' && *p <= '~') || *p == '\r' ||
		*p == '\n' || *p == '\t' || *p == '\f')     /* ASCII or TEXT */
	    {
		/* Only bother with line length checking if FT_ASCSHRT */
		if (ft == FT_ASCSHRT)
		{
		    if (*p == '\n')
		    {
			if (llen > MAX_BODYLINE)	/* Over-long line */
			    ft = FT_ASCLONG;
			llen = 0;
		    }
		    else if (*p != '\r')
			++llen;
		}
	    }
	    else			/* "Binary" char (8-bit or control) */
	    {
		if (istext)
		    return FT_TEXT;
		++cb;
		ft = FT_TEXT;
	    }
	}

	if (cb > 0L)
	{
	    /*
	     * If more than 1 in BINARY_RATIO chars are "binary" (8-bit or
	     * control) in a representative sample (BINARY_SAMPLE+ bytes) of
	     * the file, assume FT_BINARY
	     */
	    if (ntot >= BINARY_SAMPLE && cb > (ntot / BINARY_RATIO))
		return FT_BINARY;
	}
    }

    if (cb > 0L)		/* Saw some binary chars in the file */
    {
	/*
	 * If more than 1 in BINARY_RATIO chars are "binary" (8-bit or
	 * control) in the whole file, assume FT_BINARY
	 */
	if (cb > (ntot / BINARY_RATIO))
	    return FT_BINARY;
    }
    else if (cz > 0L)		/* ASCII text, ending with 1+ Ctrl-Z(s) */
    {
	if (ft == FT_ASCSHRT)		/* Text + Ctrl-Z, short lines */
	    return FT_ACZSHRT;
	else if (ft == FT_ASCLONG)	/* Text + Ctrl-Z, long lines */
	    return FT_ACZLONG;
    }

    return ft;
}


/*
 * copy_txtfile()
 *
 * Copy a text file, line by line, from <fpsrc> to <fpdst>, using buffer <buf>
 * (whic is of size <bufsize>); ensure the destination file ends with a
 * newline, even if the source file doesn't.  Return 0 if we copy the file OK,
 * 1 or 2 if we have a write error (we ensure that the destination stream is
 * flushed to disk before claiming it's OK).
 */
static int copy_txtfile(FILE *fpdst, FILE *fpsrc, char *buf, size_t bufsize)
{
    *buf = '\0';		/* Just to make sure it's initialised */
    while (fgets(buf, (int) bufsize, fpsrc) != NULL)
    {
#if 0
	(void) rip(buf);
	if (fprintf(fpdst, "%s\n", buf) == EOF)
#else
	if (fputs(buf, fpdst) == EOF)
#endif
	    return 1;
    }
    if (*buf && buf[strlen(buf) - 1] != '\n')	/* Needs newline */
	if (fputs("\n", fpdst) == EOF)
	    return 1;
    return (fflush(fpdst) == 0) ? 0 : 2;
}


/************************************************************************/
/*	Reply stuff							*/
/************************************************************************/

/*
 * reply()
 *
 * Reply to message <cmsg>; reply to the sender, unless <group_reply> is true,
 * in which case reply to all the original recipients.  Return a pointer to a
 * string describing what happened.
 */
const char *reply(PHTL cmsg, BOOL group_reply, BOOL quote_old)
{
    char subject[LINELEN + 1], msgid[LINELEN + 1], intro[LINELEN + 1];
    char *cc, *to, *otherto, *fullto, *tstring;
    size_t cc_size, to_size, otherto_size, fullto_size, tstring_size;
    const char *sep = ", ";
    const char *retstr = NULL;
    const char *p, *author;
    char last_eol;
    long end;
    FILE *fp;

    if (mfile == NULL)
	return no_mail;

    cc = malloc(cc_size = LLINELEN);
    to = malloc(to_size = LLINELEN);
    otherto = malloc(otherto_size = LLINELEN);
    fullto = malloc(fullto_size = LLINELEN);
    tstring = malloc(tstring_size = LLINELEN);
    if (cc == NULL || to == NULL || otherto == NULL || fullto == NULL ||
	tstring == NULL)
    {
	free(cc);
	free(to);
	free(otherto);
	free(fullto);
	free(tstring);
	return no_mem;
    }

    *to = *otherto = *subject = *msgid = *fullto = *cc = '\0';

    fseek(mfile, cmsg->position, SEEK_SET);	/* go to start of msg */
    end = cmsg->position + cmsg->size - 1L;
    last_eol = 1;
    (void) fgets_msg(NULL, 0, NULL, FALSE);	/* Reset fgets_msg() store */
    while (ftell(mfile) < end && fgets_msg(tstring, tstring_size, mfile, TRUE))
    {
	HEADERTYPE hdrtype;

	if (last_eol && ISBLANKLINE(tstring))	/* End of header */
	    break;	/* out of the while (ftell) */

	last_eol = rip(tstring);
	hdrtype = htype(tstring);

	if ((*to == '\0' && hdrtype == HT_FROM) || hdrtype == HT_REPLYTO)
	{
	    /*
	     * Use the author's full address (inc. full name) -- as we do for
	     * the other recipients -- rather than just the mail address as
	     * PCElm used to
	     */
	    PUSTRCPY(to, to_size, hdr_body(tstring, hdrtype));
	}

	if (hdrtype == HT_FROM || (hdrtype == HT_REPLYTO && !*fullto))
	{   /* Find author's full name; use From: in preference to Reply-To: */
	    getname(tstring, fullto, fullto_size);
	}
	else if (hdrtype == HT_SUBJECT)
	{
	    p = hdr_body(tstring, hdrtype);
	    if (strnicmp(p, "Re: ", 4) != 0)	/* don't Re: Re: Re: Re: ... */
		AUSTRLCPY2(subject, "Re: ", p);
	    else
		AUSTRCPY(subject, p);
	}
	else if (group_reply && hdrtype == HT_CC)
	{
	    PUSTRCPY(cc, cc_size, hdr_body(tstring, hdrtype));
	}
	else if (group_reply && hdrtype == HT_TO)
	{
	    if (*otherto)		/* need a separator */
		PUSTRCAT(otherto, otherto_size, sep);
	    PUSTRCAT(otherto, otherto_size, hdr_body(tstring, hdrtype));
	}
	else if (hdrtype == HT_MSGID)
	    AUSTRCPY(msgid, hdr_body(tstring, hdrtype));
    }

    /*
     * Create an "In message <msgid> <author> writes:\n" introductory line in
     * <intro>, but trying to make sure it wraps nicely.  If it won't all fit
     * on one 72-char line, break it after the <msgid> part and have a 3-char
     * indent before the <author> name.  This should all be configurable!
     * ST-FIXME
     */
    author = (fullto[0]) ? fullto : to;
    AUSTRLCPY2(intro, "In message ", (*msgid) ? msgid : "(no message ID)");
    if (strlen(intro) + strlen(author) > 72 - 9)	/* wrap line */
	AUSTRCAT(intro, "\n  ");
    AUSTRLCAT3(intro, " ", author, " writes:\n");

    if (group_reply)	/* take out all copies of our own mail address */
    {
	/*
	 * Scan each address in the <to> and <cc> lines, and if the addr-spec
	 * matches our addr-spec, we remove the whole entry.  First, we need
	 * to create the full list of recipients in <to> by adding any from
	 * <otherto>, then remove any trailing commas/spaces/tabs from both
	 * <to> and <cc>.
	 */
	size_t addrlen, al;
	char *ap, *np, *op;

	if (*otherto)		/* Append <otherto> to <to> list */
	    PUSTRLCAT2(to, to_size, sep, otherto);

	/* Create our addr-spec in <tstring> and note its length */
	PUSTRLCPY3(tstring, tstring_size, useraddr, "@", hostname);
	addrlen = strlen(tstring);

	/*
	 * Remove all addresses in <to> list whose addr-spec matches <tstring>
	 * Now matches case-insensitively
	 */
	for (op = to; (np = next_address(NULL, 0, op, &ap, &al)) != NULL;
	     op = np)
	    if (ap && al == addrlen && strnicmp(ap, tstring, addrlen) == 0)
	    {
		memmove(op, np, strlen(np) + 1);	/* OK - reducing */
		np = op;		/* move <np> back as well */
	    }

	/*
	 * Remove all addresses in <cc> list whose addr-spec matches <tstring>
	 * Now matches case-insensitively
	 */
	for (op = cc; (np = next_address(NULL, 0, op, &ap, &al)) != NULL;
	     op = np)
	    if (ap && al == addrlen && strnicmp(ap, tstring, addrlen) == 0)
	    {
		memmove(op, np, strlen(np) + 1);	/* OK - reducing */
		np = op;		/* move <np> back as well */
	    }

	kill_trail_sep(to);
	kill_trail_sep(cc);
    }

    cputs("\r\nReply To: ");
    maxgets(to, (int) to_size);
    cputs("\r\n");
    if (*to != '\0')
    {
	cputs(cc_prompt);
	clreol();		/* in case "To" spilled over on to this line */
	maxgets(cc, 79);
	if (*cc)
	{
	    char *s1 = skip_lwsp(cc);
	    memmove(cc, s1, strlen(s1) + 1);		/* OK - reducing */
	}
	cputs("\r\n");
	cputs(subj_prompt);
	clreol();		/* in case "To/Cc" spilled over to this line */
	maxgets(subject, sizeof(subject));
	if (*subject == '\0')
	    AUSTRCPY(subject, no_subj_spec);

	if (dotmenu || !use_ed)
	{
	    clrscr();
	    cprintf(to_subj_fmt, to, subject);
	}
	if (quote_old)		/* create temp. file with old msg */
	{
	    if ((fp = fopen(quote_tmpfile, "w")) == NULL)
		retstr = "Error opening tempfile";
	    else
	    {
		int err;
		if (fputs(intro, fp) == EOF)
		    err = 1;
		else
		    err = savemsg(cmsg, fp, HW_NONE, SM_DECODE | SM_QUOTE);
		if (fclose(fp) != 0 && err == 0)
		    err = 2;
		if (err > 0)		/* write error */
		    write_error(quote_tmpfile, copying_msg);
		if (err != 0)		/* failure of some sort */
		{
		    unlink(quote_tmpfile);
		    retstr = "Error writing tempfile";
		}
	    }
	}
	if (retstr != NULL)		/* It's already gone wrong */
	    ;
	else if (dosmtpsend(NULL, to, subject, cc, FALSE, CT_NONE, CE_NONE,
			    (quote_old) ? SEND_REPLYQUOTE : SEND_NEWMSG) == 0)
	    retstr = message_done;
	else
	{
	    if (quote_old && !access(quote_tmpfile, 0))
		unlink(quote_tmpfile);
	    retstr = not_sent;
	}
    }
    else
	retstr = aborted;

    /* Free all our dynamically allocated storage */
    free(cc);
    free(to);
    free(otherto);
    free(fullto);
    free(tstring);

    return retstr;
}


/*
 * getname()
 *
 * Given a "From" address in <from>, find the owner's name amd copy it into
 * <realname>.  This is to be used for the "In message <msgid>, foobar wrote:"
 * intro to a reply which quotes the original.
 *
 * The PCElm way gives different results, depending on whether the address is
 * of type (a) "address (Full Name)" or (b) "Full Name <address>": (a)-style
 * addresses are copied in full to the <realname> buffer, but (b)-style ones
 * copy *only* the Full Name (complete with quotes, if it's a quoted-string)
 * and ignore the <address> completely.  I'm sure this is a bug rather than
 * deliberate, but for the time being (and until we know what the users want)
 * we'll maintain this method.  For the record, many other mail clients seem
 * to use a canonical form like "Full Name <address>" regardless of how the
 * original address was formatted, which seems plausible to me, but I think
 * rain is wet, so who am I to say?
 */
static void getname(const char *from, char *realname, size_t realnamesize)
{
    const char *cp, *ap;

    *realname = '\0';
    if ((cp = strchr(from, ':')) != NULL)
    {
	cp = skip_lwsp(cp + 1);		/* Skip LWSP before the address */
	if ((ap = strchr(cp, '<')) != NULL && ap > cp)
	{		/* Found start of <address>, with a Full Name before */
	    size_t n = (size_t) (ap - cp);

	    if (n >= realnamesize)
		n = realnamesize - 1;
	    ustrncpy(realname, cp, n);
	}
	else		/* Use complete thing, address and all */
	    PUSTRCPY(realname, realnamesize, cp);
	kill_trail_lwsp(realname);
    }
}
