/*
  This source is part of PCOak, an electronic mailer for DOS based on PCElm.

  PCElm is Copyright (c) 1988-1993 Martin Freiss and Wolfgang Siebeck
           Copyright (c) 1992-1999 Demon Internet
  PCOak is Copyright (c) 2000-2003 Simon Turner, Pete Disdale and dispc members

  Thanks to an agreement between the original PCElm authors and Demon Internet
  made in late 1999:

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License, version 1, as
	published by the Free Software Foundation.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	See the file COPYING, which contains a copy of the GNU General
	Public License.
*/

#include <stdio.h>
#include <alloc.h>
#include <conio.h>
#include <stdlib.h>
#include <dir.h>
#include <dos.h>
#include <fcntl.h>
#include <io.h>
#include <ctype.h>
#include <bios.h>
#include <string.h>
#include <time.h>
#include "pcoak.h"
#include "display.h"
#include "notepad.h"
#include "keyboard.h"
#include "ustring.h"
#include "commands.h"
#include "macros.h"

unsigned int _stklen = 16384;	/* Force a 16KB stack size at compile time */

/************************************************************************/
/* Global configurable variables defined here, somewhat categorised	*/
/************************************************************************/

/*
 * (I'm not opposed to judicious use of globals -- far from it -- but some of
 * this is perilously close to being silly. -ST)
 *
 * A point of interest: generally, a dynamically-allocated string will take up
 * less memory than a deliberately over-large fixed buffer.  This is NOT TRUE
 * for short strings (buffers less than 16 bytes or so): a fixed length buffer
 * takes up precisely that number of bytes, but a malloc'd buffer takes up 4
 * bytes for the pointer, PLUS the required size of the buffer, added to 4
 * bytes of heap management data, all rounded up to to a 16-byte paragraph
 * (BC++ allocates all memory by paragraphs); therefore the smallest possible
 * memory usage of a dynamically-allocated string is 20 bytes.  For these
 * known-to-be-short strings, a short fixed length buffer actually takes up
 * less memory!
 *
 * However, the extra processing required to accomodate the small number of
 * such strings would outweigh the memory savings to be had, so we don't
 * bother.  (There were only 7 of them: quotestr[10], mboxext_store[2][5],
 * sequencename[13], printdevice[10], startofhdr_store[2][20]: total old size
 * 83 bytes, minimum new size 140 bytes; only 57 bytes difference!)
 *
 * We DO use fixed-length buffers for strings which are going to be changing
 * a lot (e.g. name of current mailbox), to avoid memory fragmentation caused
 * by frequent strdup()/free() calls as the string is changed.
 *
 * A final note: we don't bother explicitly initialising external pointer
 * variables to NULL, since they will be initialised to NULL anyway, and
 * explicit initialisation (a) clutters the source, and (b) increases the EXE
 * file size (although it makes no difference to memory usage, since the BSS
 * segment will contain all the zeros required for uninitialised pointers).
 * What we *do* do below is to set the proper default values for some
 * user-configurable non-pointer variables.
 */

/************************************************************************/
/* Global vars defined once at startup/config, and thereafter unchanged	*/
/************************************************************************/

/* Directly configurable by the user */
BOOL enablemime = FALSE;	/* Only try to handle MIME if this is set */
BOOL nosigdashes = FALSE;	/* Supress the --_ before .signatures */
BOOL slash_ed = FALSE;		/* Use / on editor filename instead of \ */
BOOL use_ed = FALSE;		/* Drop into editor when creating messages */
BOOL asktoggle = FALSE;		/* Ask for new host/fullname on file change */
BOOL dosify = FALSE;		/* Change files to DOS EOL */
int saveall = 0;		/* Save all outgoing messages to =sent (0-3) */
BOOL saverecd = FALSE;		/* Default save messages to =received */
BOOL autosave = FALSE;		/* Don't ask whether to save changes */
BOOL dotmenu = FALSE;		/* If true then use . .q .e .? etc. */
BOOL autodel = TRUE;		/* Auto flag as Deleted after save? */
BOOL betterread = TRUE;		/* Use Better Read it First msg? (eughh!) */
BOOL askdelqueue = FALSE;	/* Ask before deleting from mail queue */
BOOL exitconfirm = FALSE;	/* Ask on Exit? Default to FALSE */
BOOL sortdescend = FALSE;	/* TRUE for descending, FALSE for ascending */
SORTBYTYPE sortby = SORT_INCOMING; /* Sort order (SORT_INCOMING / SORT_DATE) */
char escape_char = '.';		/* "Dot menu" escape character (default '.') */
char movedirection = 'A';	/* Current move direction ('D', 'A' or 'U') */
char *quotestr;			/* for quoting include msgs in R)eply */
char *username;			/* name of this user from rc file */
char *useraddr;			/* local address of this user from rc file */
char *replyto;			/* address for Reply-To: header field */
char *uucpreplyto;		/* addr. for Reply-To: header (uucp mode) */
char *maildir;			/* defined mail directory */
char *editor;			/* user's favourite text editor */
char *pager;			/* user's favourite pager */
char *gifview;			/* user's favourite gif viewer */
char *jpegview;			/* user's favourite jpeg viewer */
char *mpegview;			/* user's favourite mpeg viewer */
char *audioview;		/* user's favourite audio player */
char *miscview;			/* user's favourite miscellaneous viewer */
char *multimedia;		/* user's favourite multimedia program */
char *mboxext_store[NUM_SYSTEMS]; /* mailbox file extensions */
char *uucp_command;		/* cmd to use when spawning rmail */
char *sequencename;		/* name of sequence file */
char *mailqdir;			/* dir for spooling outgoing mail */
char *printdevice;		/* Used print device, if not spooling */
char **weedlist1;		/* Headers to weedout for weedlevel 1 */
char **weedlist2;		/* Headers to display for weedlevel 2 */
char *opthdr[MAXOPTHDR];	/* List of optional header lines */
int smarthost = 0;		/* 0 = never, 1 = always, 2 = you_know */
char *mailhost;			/* name of our smart host */
int weedstyle = 0;		/* Style of weedout to use (0, 1 or 2) */
int printweed = HW_FULL;	/* Level of weedout for printing (0-3) */
BOOL cnfmsendabort = FALSE;	/* Confirm aborting message to be sent? */
BOOL cnfmfoldcreate = FALSE;	/* Confirm creating new folder when saving? */
BOOL viewdelread = TRUE;	/* Auto-read next msg when 'd' from viewer? */
BOOL viewnextunread = FALSE;	/* Auto-read next msg when 'n' from viewer? */
BOOL retendquit = FALSE;	/* Quit the pager with <Return> at end? */
int enclong = 1;		/* 0 = never, 1 = always, 2 = prompt */

/* Not directly configurable by the user */
char *cwd;			/* working directory (directory at startup) */
char *homedir;			/* user's home directory */
char *firstname;		/* name of FIRST (main) user; used by "c!" */
char *mailqueue;		/* workjob filename wildcard, outgoing mail */
char *tfname;			/* file for mailtext */
char *mfname;			/* file for mailtext w/ header */
int printernumber = 0;		/* printer port to use */
time_t yeartime = 0L;		/* UTC time before which we display year */

/************************************************************************/
/* Global vars defined at startup/config, but may be changed later	*/
/************************************************************************/

/* Directly configurable by the user */
int used_mailsystem = SYS_HAM;  /* 1: use smtp (ham radio); 0: spawn 'rmail'*/
int col_normal;			/* Normal colour attribute */
int col_enhanc;			/* Enhanced, errors etc. */
int col_select;			/* Selected message in menu */
int col_system;			/* Attribute at startup */
int col_hilite;			/* Highlighted Letters etc. */
int col_heading;		/* Heading colour attribute */
int extpager = EP_EXTMOSTLY;	/* External pager usage (0-2) (Changeable) */
int weedlevel = HW_WEEDED;	/* Level of weedout to apply (0-3) (Chgable) */
BOOL insertmode = FALSE;	/* Insert or overwrite in maxgets()? */
BOOL show_help = TRUE;		/* Do we show the main screen help? */
char *fullname;			/* full name of this user from rc file */
char *hostname_store[NUM_SYSTEMS]; /* names of this host */
char *startofhdr_store[NUM_SYSTEMS]; /* SOH strings for all systems*/
struct notepad_parms npsetup = { /* Builtin "notepad" editor settings */
    1,		/* edit_lira */
    65,		/* edit_rera */
    4,		/* edit_tabsize */
    TRUE,	/* wrap */
    FALSE,	/* edit_insrt */
    "",		/* key_end */
    TRUE	/* confirm_abort */
};

/* Not directly configurable by the user */
BOOL mmdf_store[NUM_SYSTEMS];	/* <mmdf> storage for all systems */
char notename[SLINELEN];	/* Name of current mailbox */
char *hdr_fullname;		/* User's full name, for use in header */
BOOL color_card;		/* Colour display (TRUE) or mono (FALSE)? */

/*****************************************************************************/

/*
 * Other global variables defined and used here, which don't get initialised
 * by the call to setup_globals()
 */
BOOL tty;				/* tells if stdin is a tty */
static struct text_info origscr_ti;	/* Original screen settings */
#ifdef SAVE_SCREEN
static char origscr_buf[132 * 60 * 2];	/* Original screen contents buffer */
#endif

char *mboxext;			/* filename extension for mailboxes */
char *hostname;			/* Name of this host for current system */
BOOL mmdf;			/* flag: mmdf-type mailbox if set */
PHTL mbox;			/* Mailbox: linked list of PHTLs */
FILE *mfile = NULL;		/* File pointer to current mailbox file */
int nmsgs = 0;			/* The number of messages in this mailbox */
PHTL currmsg = NULL;		/* The current message pointer */
static int numtagged = 0;	/* No. of tagged messages in mailbox */
BOOL mbchanged = FALSE;		/* Have we changed the current mailbox? */
char notefile[PATHLEN];		/* Full path name of the mailbox text file */
static char foldername[FOLDERLEN]; /* Name of current folder e.g. =grahame */
char keep_savefile[PATHLEN];	/* Name of last file saved (recall using F3) */
BOOL index_onscreen = FALSE;	/* Is the index screen displayed? */

/*****************************************************************************/

/*
 * Fixed strings defined and used here
 */
static const char change_prompt1[] =
  "\rEnter =<name> to change folder or ! to change back to incoming\r\n";
static const char change_prompt2[] =
  "Enter < to change to your SENT folder or > for your RECEIVED folder\r\n";
static const char change_prompt3[] =
  "Enter * to see available Mailboxes or =* to see available Folders";

static const char changes_saved[]  = "Changes saved to mailbox";
static const char mbox_unchanged[] = "Mailbox unchanged";
static const char printing_err[]   = "Printing mail";
static const char printer_err[]    = " Printer error: ";
static const char command_prompt[] = "Command: ";
static const char no_more[]        = "No more unread mail";
static const char better_read_it[] = "Better read it first!";
static const char outofmem[]       = "Out of memory: sorry!";

/*
 * Description strings for the various <extpager> and <weedlevel> settings
 */
static const char *extpager_msg[NUM_EXTPAGER] = {
    "Use internal pager only",				/* EP_INTONLY */
    "Use external pager; offer internal sometimes",	/* EP_EXTMOSTLY */
    "Use external pager only",				/* EP_EXTONLY */
};
static const char *weedlevel_msg[NUM_WEEDLEVEL] = {
    "0: full headers",					/* HW_FULL */
    "1: weeded list",					/* HW_WEEDED */
    "2: summary list",					/* HW_SUMMARY */
    "3: no headers",					/* HW_NONE */
};

/*
 * Text for the four-line quick help display on the index screen
 */
static const char *help_lines[] = {
"send \2m\2ail, \2R\2eply inc msg, \2r\2eply, \2Gg\2roup reply, \2f\2orward, \2s\2ave, \2w\2rite w/o header",
"\2t\2ransmit from file, \2l\2ist unsent, \2d\2elete, \2u\2ndelete, \2c\2hange m'box, \2n\2ext unread",
"\2F\2reshen, \2C\2olour, \2p\2aint, \2P\2rint, \2-+\2un/tag, \2ESC/Q\2uit, e\2X\2it, [\2F1\2] help, \2h\2elp on/off",
"[\2cr/space\2] read msg, \2\030\031\2, \2PgUp\2, \2PgDn\2, \2Home\2, \2End\2 to browse, \2Del\2 del, \2Ins\2 undel ",
};

#define HELPLINES	(int) NELEM(help_lines)		/* No. of help lines */

/*
 * External global variables used in this module
 */
extern const char cant_open[];		/* "Can't open " */
extern const char tempfile_err[];	/* "Error opening tempfile" */
extern const char no_tmpfile[];	   /* "tmpfile: could not create temp file." */
extern const char no_mem[];		/* "Not enough memory" */
extern const char search_prompt[];	/* "Search for what? " */
extern const char aborted[];		/* "Aborted" */
extern const char message_done[];	/* "Message done" */
extern const char str_sent[];		/* "sent" */
extern const char str_recd[];		/* "received" */
extern const char wrkfileext[];		/* ".wrk" */
extern const char lckfileext[];		/* ".lck" */
extern const char version_fmt[];	/* printf() fmt string: 1 string arg */

/*
 * Functions defined and used in this module
 */
static void change_mail_file(PHTL *currptr, BOOL toggling);
static void get_newdata(void);
static char *get_string(char *prompt, char *oldvalue);
static char *new_dynstr(const char *src, char *oldstr);
static void move_current(void);
static void printmsgs(void);
void set_system(void);
char *protect_name(const char *name);
void shellcmd(void);
static int dosave(char *buf, size_t bufsize, BOOL withhead);
char rip(char *s);
static void tag_msg(PHTL cmsg);
void untag_msg(PHTL cmsg);
static void listqueue(const char *path);
static int killjob(const char *j);
void get_cr(const char *msg);
BOOL y_n(const char *msg);
static int cr_prompt(const char *msg, const int *keylist);
static int bottom_prompt(const char *msg, const int *keylist);
static int bottom_text(int attr, const char *msg);

/*
 * Functions defined elsewhere and used in this module
 */
char maxgets(char *s, int maxlen);
int getwin(int x, int y, int maxlen, int winattr, char *input,
	   const char *prompt, ...);
void help(void);
void about(void);
void show_files(char *pattern, size_t patternsize);
int viewmsg(PHTL cmsg);
int savemsg(PHTL cmsg, FILE *tfile, int weed, unsigned options);
int delmsg(PHTL cmsg);
const char *reply(PHTL cmsg, BOOL group_reply, BOOL quote_old);
void closenotes(BOOL savechanges);
int get_save_file(char *into, size_t intosize, PCHTL cmsg);
int mailto(char *buf, size_t bufsize, SENDTYPE action);
int dosmtpsend(FILE *mfp, const char *to, const char *subject, const char *cc,
	       BOOL ismime, CTTYPE ct_type, CETYPE ct_enc, SENDTYPE action);
void backslash(char *s);
void fwdslash(char *s);
FILE *invoke_editor(FILE *tfile, const char *to, const char *subject,
		    BOOL mailing, BOOL reopen);
void colputs(int attr, const char *str);
void catputs(int attr, const char *str);
BOOL init_colors(BOOL have_color);
void writef(int x, int y, int attr, const char *line);
char *seqstr(size_t n, char ch);
void quickcenter(int line, int attr, const char *txt);
void errorbox(const char *title, ...);
void errordsp(const char *txt);
void fopen_error(const char *filename, const char *descrip);
void write_error(const char *filename, const char *descrip);
void unlink_error(const char *filename, const char *descrip);
void clreos(void);
void show_index(PCHTL current);
void cleanup(void);
void bell(void);
void edit_aliases(char *errstr, size_t errsize);
int copymail(BOOL isbiffed, BOOL dosort);
PHTL last_mbox(void);
int check_for_new_mail(void);
unsigned int getkey(void);
int wildmat(const char *text, const char *p, int ignore_case);
int harderr_handler(int errval, int ax, int bp, int si);
void harderr_reset(void);
int print_loaded(void);
int print_file(char far *fname);
void set_soh(const char *soh);
void setup_globals(void);
void weedconfig(void);
BOOL weedlevelok(int lvl);
void beep(void);
void folderpath(char *path, size_t pathsize, const char *foldername);
void mailboxpath(char *path, size_t pathsize, const char *mailboxname);


/*
 * change_mail_file()
 *
 * Allow the user to change the current mail file.  <currptr> is a pointer to
 * the current message number, which is to be reset if we change mailbox OK;
 * <toggling> is true if we're changing mail systems.
 */
static void change_mail_file(PHTL *currptr, BOOL toggling)
{
    char name[81];
    char save_notefile[sizeof(notefile)], save_notename[sizeof(notename)],
	save_foldername[sizeof(foldername)];
    int save_currlistno = (*currptr)->listno;
    int pos_x, pos_y;

    pos_x = wherex();
    pos_y = wherey();
    /* Display our prompts on the bottom 3 lines of the screen */
    gotoxy(1, display.screenheight - 2);
    cputs(change_prompt1);
    cputs(change_prompt2);
    cputs(change_prompt3);

    /*
     * Get the name of mailbox/folder to change to; give up if null name
     */
    *name = '\0';
    for (;;)
    {
	gotoxy(pos_x, pos_y);
	clreol();
	cputs("Change to new mailbox: ");
	maxgets(name, 64);
	if (name[0] == '\0')
	{
	    gotoxy(1, display.screenheight);
	    clreol();
	    return;
	}
	if (strchr(name, '*') == NULL && strchr(name, '?') == NULL)
	    break;	/* out of the for (;;) -- no wildcards */
	show_files(name, sizeof(name));
	if (*name)		/* show_files() had an error: display it */
	{
	    gotoxy(1, display.screenheight - 2);
	    clreol();
	    cputs(change_prompt1);
	    clreol();
	    cputs(change_prompt2);
	    quickcenter(display.screenheight, col_hilite, name);
	}
	*name = '\0';
    }

    /*
     * Once we get to here, the user has entered the name of a mailbox or
     * folder to change to; we may need to apply changes to the old one before
     * we can close it
     */
    if (mbchanged)		/* Old mailbox was changed */
    {
	int ch;

	gotoxy(1, display.screenheight - 2);
	clreol();
	gotoxy(1, display.screenheight - 1);
	clreol();
	if (!autosave)
	{
	    /* ST-FIXME: surely this should use y_n()?  Looks dangerous! */
	    gotoxy(10, display.screenheight - 2);
	    colputs(col_enhanc, "Save changed mailbox? ");
	    do
		ch = (int) getkey();
	    while (ch == 0);
	}
	else
	    ch = 'y';
	if (toupper(ch) == 'Y')
	{
	    if (!autosave)
		cputs("Yes\r\n");
	    closenotes(TRUE);		/* Close file, saving changes */
	    cputs(changes_saved);
	}
	else
	{
	    cputs("No\r\n");
	    cputs(mbox_unchanged);
	}
    }

    /*
     * Any changes made have either been applied to the old file or aborted;
     * take copies of the old mail file details, in case we need to reopen it
     */
    AUSTRCPY(save_notefile, notefile);
    AUSTRCPY(save_notename, notename);
    AUSTRCPY(save_foldername, foldername);

    if (asktoggle)		/* Ask for new user details */
	get_newdata();

    /*
     * Work out what notename (if we're changing user) or foldername the user
     * gave us
     */
    switch (*name)
    {
    case '<':		/* SENT folder */
	strcpy(foldername, str_sent);				/* OK */
	break;

    case '>':		/* RECEIVED folder */
	strcpy(foldername, str_recd);				/* OK */
	break;

    case '=':		/* Must be =<name> */
	AUSTRCPY(foldername, name + 1);
	break;

    case '!':		/* Change to original mailbox */
	AUSTRCPY(notename, firstname);
	*foldername = '\0';		/* this is a mailbox, not a folder */
	break;

    default:		/* Must be changing user */
	ustrncpy(notename, name, 8);
	*foldername = '\0';		/* this is a mailbox, not a folder */
	break;
    }

    if (toggling)		/* Toggle type of mailsystem */
    {
	used_mailsystem = (used_mailsystem == SYS_UUCP) ? SYS_HAM : SYS_UUCP;
	set_system();
    }

    if (*foldername)		/* Changing to a folder (same user) */
	folderpath(notefile, sizeof(notefile), foldername);
    else			/* Changing user */
	mailboxpath(notefile, sizeof(notefile), notename);

    if (mfile)	/* No changes, or changes were aborted - close file now */
	closenotes(FALSE);	/* Close file, don't save changes */

    gotoxy(1, display.screenheight - 2);
    clreol();
    gotoxy(1, display.screenheight - 1);
    clreol();
    pos_x = copymail(FALSE, TRUE);
    index_onscreen = FALSE;		/* Force index screen redraw */

    if (pos_x < 0)	/* couldn't open new file: reopen old one */
    {
	AUSTRCPY(notefile, save_notefile);
	AUSTRCPY(notename, save_notename);
	AUSTRCPY(foldername, save_foldername);
	/* now reopen current mail box */
	if (copymail(FALSE, TRUE) < 0)
	    exit(1);	/* FATAL could not re read Mail file */
	/* now reset <*currptr> to look at the same message it used to */
	for (*currptr = mbox; *currptr != NULL; *currptr = (*currptr)->next)
	    if ((*currptr)->listno == save_currlistno)
		break;	/* out of the for (*currptr) */
	if (*currptr == NULL)	/* Couldn't find same message: restart */
	    *currptr = mbox;
    }
    else		/* new file opened OK */
	*currptr = mbox;	/* set <*currptr> to first message in file */
}


/*
 * [[ Wenn der note-name geaendert wird, werden auch
 *    die anderen Klamotten abgefragt, sodass mehrere
 *    User rum-matschen koennen ... ]]
 *
 * If the note-name is changed, we must also find the other properties, so
 * that several users are able to mud around (can be rum-squash (?!))
 * ST-FIXME : need a better translation!
 */
static void get_newdata(void)
{
    char oldname[LINELEN];

    AUSTRCPY(oldname, fullname);
    fullname = get_string("fullname", fullname);
    hostname_store[used_mailsystem] =
	get_string("hostname", hostname_store[used_mailsystem]);
    if (strcmp(fullname, oldname))	/* Need new <hdr_fullname> */
    {
	char *newname = protect_name(fullname);
	if (newname == NULL)
	{
	    colputs(col_hilite, outofmem);
	    sleep(2);
	}
	else
	{
	    free(hdr_fullname);
	    hdr_fullname = newname;
	}
    }
    set_system();		/* ensure <hostname> pointer is correct */
    cputs("\n");
}


/*
 * get_string()
 *
 * Prompt user for a new version of <oldvalue>, which is a dynamically
 * allocated string, and return the string to be used from now on.  This may
 * be the same as the old string, if (a) the user didn't change it, or (b)
 * it's the same length (in which case we re-use the buffer); or it may be a
 * freshly-allocated new string, in which case the old one will have been
 * freed before returning.  (PC-lint should have fun with this!)  If we try to
 * allocate space for a new string but fail, we complain and return the old
 * string (so at least we haven't lost anything).
 *
 * Only used by get_newdata() above.
 */
static char *get_string(char *prompt, char *oldstr)
{
    char tmp[80], answer[80], *evp;

    textattr(col_normal);
    AUSTRLCPY2(tmp, "New value for ", prompt);
    AUSTRCPY(answer, oldstr);
    if (getwin(5, 17, 40, col_select, answer, tmp, NULL) <= 0)	/* No change */
    {
	cputs("Kept old value   ");
	return oldstr;
    }

    if (answer[0] == '$')	/* Appears to be an env. variable name */
    {
	(void) strupr(answer);
	if ((evp = getenv(&answer[1])) == NULL)	/* Var. not found */
	{
	    colputs(col_hilite, "No such environment variable!");
	    sleep(2);
	    return oldstr;
	}
	else		/* Got the env. variable OK */
	{
	    AUSTRCPY(answer, evp);
	    cprintf("Took \"%s\"    ", answer);
	}
    }

    return new_dynstr(answer, oldstr);
}


/*
 * new_dynstr()
 *
 * Given an existing dynamically-allocated string <oldstr> and a new setting
 * <src>, return a pointer to a new string containing this setting.  If the
 * new string is the same length as the old one, we simply re-use the storage;
 * if not, we try to allocate a new string first, and if successful we free
 * the old string and return the new one; if the new malloc fails, we complain
 * and return the old string.
 */
static char *new_dynstr(const char *src, char *oldstr)
{
    if (strlen(src) != strlen(oldstr))		/* Allocate new string */
    {
	char *newstr;

	if ((newstr = strdup(src)) == NULL)
	{
	    colputs(col_hilite, outofmem);
	    sleep(2);
	    return oldstr;
	}
	free(oldstr);
	return newstr;
    }
    else if (strcmp(oldstr, src) != 0)		/* Strings are different */
	strcpy(oldstr, src);		/* OK: we know they are same length */

    return oldstr;
}


/*
 * Move the current message pointer in the correct direction
 */
static void move_current(void)
{
    if (currmsg != NULL)
    {
	if ((movedirection == 'D' || (!sortdescend && movedirection == 'A')) &&
	    currmsg->next != NULL)
	    currmsg = currmsg->next;
	else if (currmsg->prev != NULL)
	    currmsg = currmsg->prev;
    }
}


/*
 * printmsgs()
 *
 * Queue all tagged messages for printing
 */
static void printmsgs(void)
{
    int spool_ok, err;
    char prnname[LINELEN + 1], errtxt[SLINELEN + 1];
    FILE *prnfile;
    PHTL cmsg;

    spool_ok = print_loaded();
    AUSTRLCPY2(prnname, mailqdir, "\\PRNXXXXXX");	/* was / -ST */
    if (mktemp(prnname) == NULL ||
	(prnfile = fopen(prnname, dosify ? "wt+" : "w+")) == NULL)
    {
	errorbox(tempfile_err, prnname, no_tmpfile, NULL);
	harderr_reset();	/* Since we abort printing anyway */
	return;
    }

    err = 0;
    for (cmsg = mbox; cmsg != NULL; cmsg = cmsg->next)
	if (MSGISTAGGED(cmsg))
	{
	    untag_msg(cmsg);
	    sprintf(errtxt, "Queueing message %d", cmsg->listno);  	/*OK*/
	    errordsp(errtxt);
	    if ((err = savemsg(cmsg, prnfile, printweed, SM_DECODE)) != 0)
		break;	/* out of the for (cmsg) */
	}

    if (err == 0 && fflush(prnfile) != 0)	/* fflush() failed */
	err = 1;

    if (err != 0)		/* Write error on file */
    {
	(void) fclose(prnfile);		/* OK: giving up */
	unlink(prnname);
	errorbox("File error", prnname,
		 (err > 0) ? "Write error" : "Internal error",
		 "Aborting print operation", NULL);
	return;
    }
    /* else we've written and flushed the file OK; it should be fine */

    if (spool_ok)		/* OK to use print spooler */
    {
	(void) fclose(prnfile);			/* OK: it's been fflush()ed */
	/*
	 * Submit to PRINT, if spooling
	 *
	 * Note: Since I don't get a useful error information on
	 * Compaq DOS 3.31, I ignore the result. It's up to
	 * the user to listen, if his printer is working.
	 */
	(void) print_file((char far *) prnname);
	/* Don't unlink the temp file; the spool is still using it! */
    }
    else			/* do it the hard way */
    {
	int error = 0;
	int i, c, last_c;

	errordsp(printing_err);
	rewind(prnfile);
	AUSTRCPY(errtxt, printer_err);
	last_c = 0;		/* something harmless */
	while ((c = getc(prnfile)) != EOF)
	{
	    /* CMS: Even in binary mode, we miss the CR */
	    if (c == 0x0a && last_c != 0x0d)	/* LF without preceding CR */
	    {
		(void) ungetc(c, prnfile);
		c = 0x0d;
	    }
	    i = biosprint(0, c, printernumber);
	    if (i == 0xC8)
	    {
		AUSTRCAT(errtxt, "turned off");
		error++;
	    }
	    else if (i & 0x20)
	    {
		AUSTRCAT(errtxt, "out of paper");
		error++;
	    }
	    else if (i & 0x08)
	    {
		AUSTRCAT(errtxt, "offline");
		error++;
	    }
	    if (error)
	    {
		/* TODO: improve this prompt */
		AUSTRCAT(errtxt, ", Esc to abort, any key to retry");
		errordsp(errtxt);
		AUSTRCPY(errtxt, printer_err);
		if (getch() == ESC)
		{
		    errordsp(aborted);
		    return;
		}
		error = 0;
		errordsp(printing_err);
	    }
	    last_c = c;
	}
	(void) fclose(prnfile);		/* Should be OK; don't care any more */
	(void) biosprint(0, 0x0c, printernumber);	/* CMS: Send a FF */
	unlink(prnname);
    }
}


/*
 * set_system()
 *
 * Set the current system (one of SYS_*); this involves setting pointers to
 * various <*_store>[] strings (mboxext, hostname, mmdf) and setting the SOH
 * string correctly; we also need to get the default file mode correct
 */
void set_system(void)
{
    mboxext = mboxext_store[used_mailsystem];
    hostname = hostname_store[used_mailsystem];
    mmdf = mmdf_store[used_mailsystem];
    set_soh(startofhdr_store[used_mailsystem]);
    _fmode = (int) ((used_mailsystem == SYS_UUCP) ? O_BINARY : O_TEXT);
}


/*
 * protect_name()
 *
 * Given a name string <name>, return a dynamically-allocated version of it
 * which is safe for use in RFC 2822 header bodies: this means converting it
 * to an RFC 2822 quoted-string (enclosing it in double quotes, and making any
 * " or \ characters in it into quoted-pairs \" and \\ respectively) if it
 * contains any RFC 2822 special characters.  We do a simple check that the
 * string isn't already protected; if it begins and ends with ", we assume
 * it's already a valid quoted-string, and just return a copy of it.
 */
char *protect_name(const char *name)
{
    size_t namelen = strlen(name);

    if (name[strcspn(name, "()<>[]:;@\\,.\"")] == '\0' ||   /* No specials */
	(name[0] == '\"' && name[namelen - 1] == '\"'))     /* Quoted-string */
	return strdup(name);
    else			/* Contains specials: return a quoted-string */
    {
	const char *cp;
	size_t nqp = 0;
	char *newname;

	/* Count the number of illegal quoted-string characters */
	for (cp = name; *cp; cp++)
	    if (*cp == '\"' || *cp == '\\')	/* Needs a quoted-pair */
		nqp++;
	if ((newname = malloc(namelen + 3 + nqp)) != NULL)
	{
	    char *np = newname;

	    *np++ = '\"';	/* Start with a DQUOTE */
	    for (cp = name; *cp; cp++, np++)
	    {
		if (*cp == '\"' || *cp == '\\')	/* Char needs to be quoted */
		    *np++ = '\\';
		*np = *cp;
	    }
	    *np++ = '\"';	/* End with a DQUOTE */
	    *np = '\0';		/* Terminate the string! */
	}
	return newname;
    }
}


/************************************************************************/
/*	main()								*/
/************************************************************************/

/*
 * main()
 *
 * command parser, handle at least read, send, headers, delete
 *
 * Developer note: since the whole program runs under this function, we try to
 * minimise function-wide stack usage here; this means that temporary buffers
 * used only in one bit of code should be scoped for that bit of code etc.
 */
void main(int argc, char *argv[])
{
    char statmsg[132];	/* Buffer for status messages (used to be "command") */
    const char *valid_commands = DEF_COMMANDS;
    int i, noreprint, n, oextpager;
    int commandchar;
    BOOL from_view = FALSE;
    struct time now;
    long loopcounter;
    PHTL cmsg, nextnewmsg;

    tzset();

    if (isatty(fileno(stdin)))			/* Save the screen. */
    {
	gettextinfo(&origscr_ti);
#ifdef SAVE_SCREEN
	gettext(1, 1, origscr_ti.screenwidth, origscr_ti.screenheight,
		(void *) origscr_buf);
#endif
	clrscr();			/* clear the screen */
	printf(version_fmt, myversion);	/* announce ourselves */
	tty = TRUE;
	if (atexit(cleanup))
	{
	    cputs("Can't install exit procedure, aborting ...\n");
	    _exit(1);
	}
    }
    else
	tty = FALSE;

    setup_globals();
    weedconfig();

    currmsg = NULL;

    /*
     * Having done the program setup, did we have a command line?  If so, send
     * a single message and stop immediately
     *
     * ST-FIXME: use the command line better (don't just send msg from stdin)
     */
    if (argc > 1)
    {
	char *notel;

	if (argc > 2)
	    notel = argv[2];
	else
	    notel = NULL;
	if (dosmtpsend(NULL, argv[1], notel, NULL, FALSE, CT_NONE, CE_NONE,
		       SEND_NEWMSG) == 0)
	    puts(message_done);
	else
	    puts("Message send failed!");
	exit(0);
    }

    mailboxpath(notefile, sizeof(notefile), notename);
    disp_init();
    disp_helplines((show_help) ? HELPLINES : 0);

    if (copymail(FALSE, TRUE) < 0)  /* FATAL couldn't read primary mail file */
    {
	/* Move cursor to near bottom of screen before wimping out */
	disp_gotoxy(1, display.screenheight - 2);
	exit(1);
    }
    currmsg = mbox;

    clrscr();
    index_onscreen = FALSE;		/* Force index screen redraw */
    statmsg[0] = '\0';			/* No status messages yet */

    harderr(harderr_handler);		/* Install "Abort, Retry" handler */

    oextpager = extpager;		/* Make sure this is in sync */

    noreprint = FALSE;	/* set to TRUE whilst cursoring around to
			   stop the flickering Command prompt */
    /* command parsing loop */
    while (TRUE)
    {
	if (from_view)		/* We've just come from the viewer */
	{
	    index_onscreen = FALSE;	/* Force index screen redraw */
	    from_view = FALSE;
	}
	if (!noreprint)
	{
	    char filename[max(sizeof(notename), sizeof(foldername)) + 1];
	    char title[100];

	    if (*foldername == '\0')
		AUSTRCPY(filename, notename);		/* max. 63 chars */
	    else
		AUSTRLCPY2(filename, "=", foldername);	/* max. 64 chars */
	    /*
	     * <title> is 100 bytes long: sprintf() must write <= 99 chars!
	     * Currently it's a maximum of 29 + 30 + 5 + 1 + 10 + 16 + 8 = 99.
	     */
	    sprintf(title,						/*OK*/
		    " Mailbox %.30s, %d message%s%s %.16s System: %s",	/*29*/
		    filename,						/*30*/
		    nmsgs,						 /*5*/
		    (nmsgs > 1) ? "s" : "",				 /*1*/
		    (mbchanged) ? " - changed" : "",			/*10*/
		    myversion,						/*16*/
		    (used_mailsystem == SYS_HAM) ? "Internet" : "UUCP"); /*8*/
	    if (!index_onscreen)	/* Need to redraw index screen */
	    {
		textattr(col_normal);
		clrscr();
		show_index(currmsg);
		if (show_help)		/* Show quick help */
		{
		    int l, row;

		    for (l = 0, row = display.screenheight -
			     MAIN_BOT_LINES - HELPLINES;
			 l < HELPLINES; l++, row++)
			quickcenter(row, col_heading, help_lines[l]);
		}
		index_onscreen = TRUE;	/* FIX for 3.1 */
	    }
	    /* quickcenter patched to wipe out entire line if using
	       heading colour so we get a band of the same colour at the
	       top - this makes it look similar to SNEWS and DIS.EXE */
	    quickcenter(1, col_heading, title);
	    /* blank line follows printed in the heading background colour */
	    writef(1, 2, col_heading, seqstr(display.screenwidth, ' '));
	}

	/* Clear the bottom MAIN_BOT_LINES of the screen, put up the prompt */
	gotoxy(1, display.screenheight - (MAIN_BOT_LINES - 1));
	clreos();
	cputs(command_prompt);
	noreprint = FALSE;

	if (statmsg[0] != '\0')		/* Display status message */
	{
	    backslash(statmsg);		/* ensure any paths have \ not / */
	    errordsp(statmsg);
	}

	/*
	 * We reset the harderr flag whenever we're waiting for a keypress, so
	 * that any disk harderr interrupts which happen from now on are
	 * treated as new ones, and cause re-prompting of the user
	 */
	harderr_reset();	

	/*
	 * Hammer round this loop until we have a keypress; keep track of the
	 * time, and check for new mail in the currently-viewed mailbox or
	 * folder every 3 seconds.  If we find any, refresh our copy of the
	 * mailbox/folder to be up to date.
	 */
	gettime(&now);
	loopcounter = now.ti_hour * 3600L + now.ti_min * 60L + now.ti_sec;
	do
	{
	    commandchar = (int) getkey();	/* Range 0x0000 - 0x1fff */
	    if (commandchar == 0)
	    {					/* we are idle so */
		long currsecs;

		gettime(&now);
		currsecs = now.ti_hour * 3600L + now.ti_min * 60L + now.ti_sec;
		if (currsecs - loopcounter > 3L)
		{
		    /*
		     * Clear the bottom two screen lines (there may be
		     * temporary messages displayed there at the moment), and
		     * put the cursor back after the "Command" prompt; then
		     * check for new mail, and if any has arrived, advise the
		     * user then re-read the mailbox/folder to get our copy up
		     * to date (doing this no longer sets <mbchanged> to TRUE;
		     * why should it?)
		     */
		    gotoxy(1, display.screenheight - 1);
		    clreos();
		    gotoxy(10, display.screenheight - 3);
		    if (check_for_new_mail())
		    {
			/* mbchanged = TRUE; */		/* Removed -ST */
			bell();				/* ding-dong ! */
			i = copymail(TRUE, TRUE);	/* biff user */
			if (i == -1)
			    exit(1);	/* FATAL could not re read Mail file */
			/* This should be less vicious! ST-FIXME */
			if (currmsg == NULL)	/* mailbox used to be empty */
			    currmsg = mbox;
			index_onscreen = FALSE;		/* force redraw */
		    }
		    loopcounter = currsecs;
		    if (!index_onscreen)
			break;	/* out of the do .. while (commandchar == 0) */
		}
	    }
	    else		/* Got a keypress: handle it */
	    {
		/*
		 * Since we allow the user to change the commands, we
		 * have to fiddle a bit with these keys.
		 */
	    gotcommandchar:
		if (commandchar == '$')		/* map $ to F (Freshen) */
		    commandchar = 'F';
		else if (commandchar == KTAB)	/* map TAB key to ASCII 9 */
		    commandchar = '\t';
		else if (commandchar == KCR)	/* Hard-wire Ret -> Read */
		    commandchar = CMD_READMSG;
		else if (commandchar == KESC)	/* Hard-wire Esc -> Quit */
		    commandchar = CMD_QUIT;
		if (commandchar < 255)		/* Check list of commands */
		{
		    const char *cp;
		    if ((cp = strchr(valid_commands, commandchar)) != NULL)
			commandchar = CMD_BASE + (int)(cp - valid_commands) +1;
		}
		/* else we try <commandchar> in the switch () below anyway */
	    }
	}
	while (commandchar == 0);

	if (!index_onscreen)	/* biffed: new mail was detected */
	    continue;	/* with the main while (TRUE) -- force redraw */

	/*
	 * ST-FIXME : Holy Mother of Pearl, this switch is ghastly.  It needs
	 * to be completely ripped out and done from scratch; the flying
	 * gotos alone are enough to make it quite impenetrable.  This brings
	 * back all my nightmares about BASIC!
	 */
	clreos();
	statmsg[0] = '\0';
	switch (commandchar)
	{
	case '?':
	case KF1:
	    help();
	    index_onscreen = FALSE;	/* Force index screen redraw */
	    break;

	case CMD_ABOUT:
	    about();
	    index_onscreen = FALSE;	/* Force index screen redraw */
	    break;

	case KF2:
	    directvideo = !directvideo;
	    AUSTRLCPY2(statmsg, "Direct video is now ",
		       (directvideo) ? "on" : "off");
	    break;

	case CMD_TAG:	/* tag message */
	    tag_msg(currmsg);
	    move_current();
	    show_index(currmsg);
	    break;

	case CMD_UNTAG:	/* untag message */
	    untag_msg(currmsg);
	    move_current();
	    show_index(currmsg);
	    break;

	case KKP8:
	case KUP:
	    if (currmsg && currmsg->prev != NULL)
		show_index(currmsg = currmsg->prev);
	    noreprint = TRUE;
	    break;

	case KKP2:
	case KDN:
	    if (currmsg && currmsg->next != NULL)
		show_index(currmsg = currmsg->next);
	    noreprint = TRUE;
	    break;

	case KKP9:
	case KPGUP:
	    if (currmsg)
		for (i = 0; i < display.main_lines && currmsg->prev != NULL;
		     i++)
		    currmsg = currmsg->prev;
	    show_index(currmsg);
	    noreprint = TRUE;
	    break;

	case KKP3:
	case KPGDN:
	    if (currmsg)
		for (i = 0; i < display.main_lines && currmsg->next != NULL;
		     i++)
		    currmsg = currmsg->next;
	    show_index(currmsg);
	    noreprint = TRUE;
	    break;

	case KKP7:
	case KHOME:
	    show_index(currmsg = mbox);
	    break;

	case KKP1:
	case KEND:
	    show_index(currmsg = last_mbox());
	    break;

	case CMD_REPAINT:	/* rebuild screen */
	    index_onscreen = FALSE;	/* Force index screen redraw */
	    break;

	case CMD_TOGGLE:	/* toggle mailsystem */
	    change_mail_file(&currmsg, TRUE);
	    break;

	case CMD_READMSG:	/* read current msg */
	read_it:
	    if (mbox == NULL)
		break;
	    cputs("Read message\r\n");
	    commandchar = viewmsg(currmsg);
	    if (extpager != oextpager)	/* Quietly replace old pager setting */
		extpager = oextpager;
	    if (commandchar)
	    {
		gotoxy(1, display.screenheight - (MAIN_BOT_LINES - 1));
		clreos();
		from_view = TRUE;
		if (commandchar == 'e')	/* Map to 'E' -- ghastly fudge! */
		    commandchar = 'E';
		goto gotcommandchar;
	    }
	    index_onscreen = FALSE;	/* Force index screen redraw */
	    break;

	case CMD_SHELL:		/* execute shellcmd */
	    shellcmd();
	    index_onscreen = FALSE;	/* Force index screen redraw */
	    break;

	case CMD_NEXTUNREAD:	/* go to next unread msg */
	    nextnewmsg = currmsg;
	    while (nextnewmsg != NULL)
	    {
		if (movedirection == 'D' ||
		    (!sortdescend && movedirection == 'A'))
		    nextnewmsg = nextnewmsg->next;
		else
		    nextnewmsg = nextnewmsg->prev;
		if (nextnewmsg != NULL && !(nextnewmsg->status & S_READ))
		    break;
	    }
	    if (nextnewmsg != NULL)
	    {
		currmsg = nextnewmsg;
		if (!from_view)
		    show_index(currmsg);
		else if (viewnextunread)	/* read this one now */
		    goto read_it;
	    }
	    else
		AUSTRCPY(statmsg, no_more);
	    break;

	case CMD_TRAWL:		/* read next unread msg */
	    nextnewmsg = currmsg;
	    while (nextnewmsg != NULL && (nextnewmsg->status & S_READ))
		nextnewmsg = nextnewmsg->next;
	    if (nextnewmsg != NULL)
	    {
		currmsg = nextnewmsg;
		goto read_it;
	    }
	    AUSTRCPY(statmsg, no_more);
	    break;

	case CMD_MARKUNREAD:	/* Mark message unread */
	    /* Why doesn't this operate on all tagged messages? -ST */
	    if (currmsg && currmsg->status & S_READ)	/* has been read */
	    {	/* not deleted or read: can't delete an unread msg */
		currmsg->status &= ~(S_READ | S_DELETE);
		mbchanged = TRUE;
		show_index(currmsg);
	    }
	    break;

	case KKP0:
	case KINS:
	case CMD_UNDELETE:	/* undelete a msg */
	    if (mbox == NULL)
		break;
	    cputs("Undelete message");
	    n = 0;
	    if (numtagged == 0)
	    {
		tag_msg(currmsg);
		move_current();
	    }
	    else if (numtagged == 1 && !MSGISTAGGED(currmsg))
		n = 2;		/* Fudge to ensure screen gets redrawn */
	    for (cmsg = mbox; cmsg != NULL; cmsg = cmsg->next)
		if (MSGISTAGGED(cmsg))
		{
		    cmsg->status &= ~S_DELETE;
		    untag_msg(cmsg);
		    n++;
		}
	    if (n > 1)
		index_onscreen = FALSE;		/* Force index screen redraw */
	    else
		show_index(currmsg);
	    break;

	case CMD_LIST:	/* display unsent messages */
	    listqueue(mailqueue);
	    index_onscreen = FALSE;	/* Force index screen redraw */
	    break;

	case CMD_EXIT:	/* eXit */
	    cputs("eXit\r\n");
	    closenotes(FALSE);		/* Close file, don't save changes */
	    cputs(mbox_unchanged);
	    exit(0);
	    break;

	case CMD_COLOR:	/* Color-toggle */
	    color_card = !color_card;
	    (void) init_colors(color_card);
	    clrscr();
	    index_onscreen = FALSE;	/* Force index screen redraw */
	    break;

	case CMD_QUIT:	/* quit, save changes */
	    if (exitconfirm)
	    {
		if (mbchanged)
		    cprintf("\rEnd Session and Update? Press Y or N");
		else
		    cprintf("\rEnd Session? Press Y or N");
		clreol();
		if (toupper(getch()) != 'Y')
		    break;
	    }
	    cputs("Quitting... ");
	    closenotes(TRUE);		/* Close file, saving changes */
	    if (mbchanged)
		colputs(col_enhanc, changes_saved);

	    /* Restore the screen. */
	    /* funny - we don't restore the screen if we X out! */

	    textattr((int) origscr_ti.attribute);
#ifdef SAVE_SCREEN
	    (void) puttext(1, 1, origscr_ti.screenwidth,
			   origscr_ti.screenheight, (void *) origscr_buf);
#else
	    clrscr();
#endif
	    gotoxy(origscr_ti.curx, origscr_ti.cury);
	    exit(0);
	    break; /* NOT REACHED */

	case KKPCOMMA:
	case KDEL:
	case CMD_DELETE:	/* delete a message */
	    if (mbox == NULL)
		break;
	    cputs("Delete message");
	    n = 0;
	    if (numtagged == 0)
	    {
		PCHTL oldcurrmsg = currmsg;
		tag_msg(currmsg);
		move_current();
		if (from_view && currmsg == oldcurrmsg)
		{
		    /* Deleting last message while viewing so return to menu */
		    from_view = FALSE;
		    index_onscreen = FALSE;	/* Force index screen redraw */
		    n = 2;	/* cheat; force headers to be reshown at end */
		}
	    }
	    else if (numtagged == 1 && !MSGISTAGGED(currmsg))
		n = 2;		/* Fudge to ensure screen gets redrawn */
	    for (cmsg = mbox; cmsg != NULL; cmsg = cmsg->next)
	    {
		if (MSGISTAGGED(cmsg))
		{
		    if (!betterread)	/* make it look read to allow delete */
			cmsg->status |= S_READ;
		    if (!delmsg(cmsg) && betterread)
			if (!(cmsg->status & S_READ))	/* unread */
			    AUSTRCPY(statmsg, better_read_it);
		    untag_msg(cmsg);
		    n++;
		}
	    }
	    if (from_view && viewdelread)	/* Read next msg now */
		goto read_it;
	    if (n > 1)
		index_onscreen = FALSE;		/* Force index screen redraw */
	    else
		show_index(currmsg);
	    break;

	case CMD_NEWSOH:    /* change Start of Header, enables notes reading */
	    {
		char buf[81], newsoh[32];

		AUSTRLCPY3(buf, "new Start of header (now \"",
			   startofhdr_store[used_mailsystem], "\")");
		if (getwin(15, 17, 19, col_select, newsoh, buf, NULL) > 0)
		{
		    startofhdr_store[used_mailsystem] =
			new_dynstr(newsoh, startofhdr_store[used_mailsystem]);
		    mmdf_store[used_mailsystem] =
			(stricmp(newsoh, "mmdf") == 0);
		    set_system();		/* Reset system settings */
		    AUSTRLCPY3(statmsg, "SOH changed to \"",
			       startofhdr_store[used_mailsystem], "\"");
		}
		else
		    AUSTRCPY(statmsg, "No change");
	    }
	    break;

	case CMD_SAVE:		/* save current msg to file with header */
	case CMD_WRITE:		/* save current msg to file w/o header */
	    if (mbox == NULL)
		break;
	    n = dosave(statmsg, sizeof(statmsg),
		       (commandchar == CMD_SAVE) ? TRUE : FALSE);
	    if (n > 1 || from_view)
		index_onscreen = FALSE;		/* Force index screen redraw */
	    else
		show_index(currmsg);
	    break;

	case CMD_PRINTMSG:	/* Print (tagged) messages */
	    if (mbox == NULL)
		break;	/* Nothing to do */
	    errordsp("Queueing messages for print");
	    if (numtagged == 0)
		tag_msg(currmsg);
	    printmsgs();
	    index_onscreen = FALSE;		/* Force index screen redraw */
	    break;

	case CMD_MIME:
	    mailto(statmsg, sizeof(statmsg), SEND_NEWMSG);
	    index_onscreen = FALSE;	/* Force index screen redraw */
	    break;

	case CMD_TRANSMIT:		/* transmit file*/
	    mailto(statmsg, sizeof(statmsg), SEND_FILE);
	    index_onscreen = FALSE;	/* Force index screen redraw */
	    break;

	case CMD_MAILTO:
	    mailto(statmsg, sizeof(statmsg), SEND_NEWMSG);
	    index_onscreen = FALSE;	/* Force index screen redraw */
	    break;

	case CMD_REPLY:		/* reply to author */
	case CMD_REPLY_INCL:	/* reply to author, including old message */
	case CMD_GROUP:		/* group reply */
	case CMD_GROUP_INCL:	/* group reply, including old message */
	    if (mbox == NULL)
		break;
	    if (betterread && !(currmsg->status & S_READ))
	    {
		AUSTRCPY(statmsg, better_read_it);
		break;
	    }
	    AUSTRCPY(statmsg,
		     reply(currmsg,
			   (commandchar == CMD_GROUP ||
			    commandchar == CMD_GROUP_INCL) ? TRUE : FALSE,
			   (commandchar == CMD_REPLY_INCL ||
			    commandchar == CMD_GROUP_INCL) ? TRUE : FALSE));
	    index_onscreen = FALSE;	/* Force index screen redraw */
	    break;

	case CMD_FRESHEN:	/* freshen mailbox */
	    if (!mbchanged)
		AUSTRCPY(statmsg, "No need to do so");
	    else
	    {
		cputs("Freshen... ");
		closenotes(TRUE);	/* Close file, saving changes */
		if (copymail(FALSE, TRUE) < 0)
		    exit(1);	/* FATAL could not re read Mail file */
		currmsg = mbox;
		index_onscreen = FALSE;	/* Force index screen redraw */
	    }
	    break;

	case CMD_FORWARD:		/* forward a message */
	    if (mbox == NULL)
		break;
	    if (numtagged == 0)
		tag_msg(currmsg);
	    mailto(statmsg, sizeof(statmsg), SEND_FWDQUOTE);
	    untag_msg(currmsg);
	    index_onscreen = FALSE;	/* Force index screen redraw */
	    break;

	case CMD_CHANGEMBOX:	/* display or change notefiles */
	    change_mail_file(&currmsg, FALSE);
	    break;

	case CMD_ALIASES:
	    edit_aliases(statmsg, sizeof(statmsg));
	    index_onscreen = FALSE;	/* Force index screen redraw */
	    break;

	case CMD_SEARCHSUBJ:
	    {
		char pattern[81];	/* Big enough for 70 chars + stars */

		pattern[0] = '*';	/* Pattern must start with '*' */
		pattern[1] = '\0';	/* ... and be terminated! */
		if (getwin(5, 17, 70, col_select, pattern + 1, search_prompt,
			   NULL) > 0)
		{
		    AUSTRCAT(pattern, "*");	/* Pattern must end with '*' */
		    for (cmsg = currmsg; cmsg != NULL; cmsg = cmsg->next)
		    {
			if (wildmat(cmsg->subject, pattern, 1))
			{
			    currmsg = cmsg;	      /* Set current message */
			    index_onscreen = FALSE;   /* Force index redraw */
			    break;	/* out of the for (cmsg) */
			}
		    }
		}
	    }
	    break;

	case CMD_TOGGLEHELP:		/* Toggle 4-line help on & off */
	    show_help = !show_help;
	    disp_helplines((show_help) ? HELPLINES : 0);
	    index_onscreen = FALSE;	/* Force index screen redraw */
	    break;

	case CMD_TOGGLEWEED:		/* Cycle the heading weeding level */
	    do
		weedlevel = (weedlevel + 1) % NUM_WEEDLEVEL;
	    while (!weedlevelok(weedlevel));
	    AUSTRLCPY2(statmsg, "Header weeding level ",
		       weedlevel_msg[weedlevel]);
	    break;

	case CMD_EXTPAGER:		/* Cycle external pager setting */
	case CMD_EXTPAGREAD:		/* Read with external pager */
	    if (!pager)		/* no external pager */
		AUSTRCPY(statmsg, "No external pager defined");
	    else if (commandchar == CMD_EXTPAGREAD)
	    {
		oextpager = extpager;	/* Keep old setting; replace after */
		extpager = EP_EXTONLY;	/* Temporarily set "external only" */
		goto read_it;	/* Read the message */
	    }
	    else
	    {
		oextpager = extpager = (extpager + 1) % NUM_EXTPAGER;
		AUSTRCPY(statmsg, extpager_msg[extpager]);
	    }
	    break;

	default:
	    AUSTRCPY(statmsg, "Invalid key");
	    break;
	}		/* End of switch (commandchar) */
    }		/* End of command-parser */
}


/*
 * shellcmd()
 *
 * Prompt the user for a shell command to run and run it; prompt to continue
 * before returning, unless (a) it was a real shell or (b) the user's command
 * started with the special character NPCMD, in which case we just return
 * directly.
 */
void shellcmd(void)
{
    char cmd[128];

    cmd[0] = '\0';
    if (getwin(5, 17, 70, col_select, cmd,
	       "Enter command or \021\304\331 for sh", NULL) == 0)
    {
	const char *evp = getenv("COMSPEC");

	cmd[0] = NPCMD;		/* Prepend NPCMD to stop prompt afterwards */
	PUSTRCPY(cmd + 1, sizeof(cmd) - 1, (evp) ? evp : "command");
#if 0	    /* putenv has no effect with the swapping exec() -ST */
	putenv("PROMPT=(PCOak active)$_$p$g ");
#endif
    }
    (void) disp_system(cmd, TRUE);
    if (cmd[0] != NPCMD)	/* Prompt before continuing */
	get_cr(NULL);
}


/*
 * dosave()
 *
 * Try to save all the tagged messages (current message only if none tagged)
 * to a file or folder of the user's choosing; if <withhead> is true, we save
 * the headers as well, otherwise it's just the bodies.  Return the number of
 * messages saved.
 *
 * Offer a default name of "=received" (if <saverecd> is true), or a name
 * found from the address of the sender of the current message <currmsg>.
 * Once we've agreed about a filename to use, try to open it (in an
 * appropriate mode) and do the saving if we can.
 *
 * Once we're finished, copy a message about what we've done into <buf> (which
 * is <bufsize> bytes long).
 */
static int dosave(char *buf, size_t bufsize, BOOL withhead)
{
    char savefile[PATHLEN], prompt[81];
    const char *mode = (dosify) ? "at" : "a";	/* File mode for fopen() */
    const char *action = NULL;
    FILE *fp;
    int n;
    PHTL cmsg;

    /*
     * Copy the default folder name to use into <savefile> to start off
     */
    strcpy(savefile, "=");
    if (saverecd)	/* Default to RECEIVED folder */
	AUSTRCAT(savefile, str_recd);
    else		/* Find foldername from sender's address */
    {
	/* TODO: use recipient address if sent by this user */
	(void) get_save_file(savefile + 1, sizeof(savefile) - 1, currmsg);
	strlwr(savefile);
    }

    AUSTRLCPY3(prompt, "Add message ", (withhead) ? "with" : "without",
	       " header to (=<name> saves to folder. F3 recall)");

    /*
     * Keep asking for a name until (a) a filename is chosen and accepted, or
     * (b) the user aborts the prompt.
     */ 
    while (action == NULL &&
	   getwin(5, 17, 69, col_select, savefile, prompt, NULL) != 0)
    {
	AUSTRCPY(keep_savefile, savefile);	/* Keep a copy for F3 use */
	if (strchr("=<>!", *savefile) != NULL)	/* It's a folder or mailbox */
	{
	    char namebuf[PATHLEN];	/* For a possible copy of <savefile> */
	    const char *name;		/* Pointer to name of folder */
	    char type = *savefile;	/* Type: '=', '<', '>' or '!' */
	    char *scp;
	    BOOL isnew;

	    /*
	     * Get the full file path to use into <savefile>; this may be a
	     * folder or the current mailbox
	     */
	    if ((scp = strchr(savefile, ' ')) != NULL)
		*scp = '\0';	/* Truncate name at first space */
	    if (type == '!')	/* Mailbox */
	    {
		mailboxpath(savefile, sizeof(savefile), notename);
		name = NULL;
	    }
	    else		/* Folder (< , > or =... */
	    {
		if (type == '<')	/* SENT folder */
		    name = str_sent;
		else if (type == '>')	/* RECEIVED folder */
		    name = str_recd;
		else			/* User-specified folder */
		    name = AUSTRCPY(namebuf, savefile + 1);
		/*
		 * Set the full pathname of the folder to use in <savefile>,
		 * having first made sure the folder directory exists with a
		 * blind call to mkdir() [Yuck -ST]
		 */
		folderpath(savefile, sizeof(savefile), NULL);	/* Directory */
		(void) mkdir(savefile);
		AUSTRLCAT2(savefile, "\\", name);		/* File path */
	    }
	    isnew = (access(savefile, 0)) ? TRUE : FALSE;
	    if (cnfmfoldcreate && isnew && name)  /* Confirm folder creation */
	    {
		char txt[81];

		sprintf(txt, "Folder =%.44s does not exist. Create it?", /*OK*/
			name);
		if (!y_n(txt))
		{
		    AUSTRCPY(savefile, keep_savefile);	/* restore entry */
		    continue;	/* with the while (getwin) */
		}
	    }
	    action = (isnew) ? " created" : " appended";
	}
	else					/* It's a straight file */
	{
	    if (!access(savefile, 0))	/* File already exists */
	    {
		static int keylist[] = { 'a', 'r', 't', 'e',
					 'A', 'R', 'T', 'E', 0 };
		int ch;

		/* Blank bottom 4 lines of screen for our prompting */
		gotoxy(1, display.screenheight - 3);
		clreos();
		gotoxy(1, display.screenheight - 2);
		catputs(col_select, savefile);
		ch = bottom_prompt("File exists.  (A)ppend, (R)eplace, "
				   "(T)ype again or (E)xit?", keylist);
		gotoxy(1, display.screenheight - 2);
		clreol();
		ch = toupper(ch);
		if (ch == 'A')
		    action = " appended";
		else if (ch == 'R')
		{
		    mode = (dosify) ? "wt" : "w";
		    action = " replaced";
		}
		else if (ch == 'T')
		{
		    AUSTRCPY(savefile, keep_savefile);	/* restore entry */
		    continue;	/* with the while (getwin) */
		}
		else if (ch == 'E')
		    break;	/* out of the while (getwin) */
	    }
	    else			/* File doesn't exist */
	    {
		/* ST-FIXME?  Ask about whether to create the file? */
		action = " created";
	    }
	}
    }

    /*
     * If <action> is NULL, the user decided aborted the save process; return
     * now, having done nothing.
     */
    if (action == NULL)
    {
	PUSTRCPY(buf, bufsize, aborted);
	return 0;
    }

    /*
     * Right!  We've got a full file path in <savefile> and the user is happy
     * with the choice.  Try to open it; if we fail, fill in an error message
     * and return 0.
     */
    if ((fp = fopen(savefile, mode)) == NULL)		/* Rats! */
    {
	PUSTRLCPY2(buf, bufsize, cant_open, savefile);
	return 0;
    }

    /*
     * We've got the file open; start saving.  If there are no messages
     * tagged, tag the current one; then run through all the tagged messages
     * in displayed order, saving each one.  Once we're done, close the file
     * (displaying an error message if this fails, and everything had worked
     * OK up to that point), fill in a message saying what we did, and return
     * the number of messages successfully saved.
     */
    if (numtagged == 0)		/* Tag the current message */
    {
	tag_msg(currmsg);
	if (autodel)
	    move_current();
    }

    for (cmsg = mbox, n = 0; cmsg != NULL; cmsg = cmsg->next)
	if (MSGISTAGGED(cmsg))
	{
	    int err = savemsg(cmsg, fp, (withhead) ? HW_FULL : HW_NONE,
			      SM_NONE);
	    untag_msg(cmsg);
	    if (err != 0)
	    {
		if (err > 0)    /* write error during savemsg() */
		    write_error(savefile, "saving message");
		break;	/* out of the for (cmsg) */
	    }
	    if (autodel)
	    {
		cmsg->status |= S_DELETE;	/* flag as deleted */
		mbchanged = TRUE;
	    }
	    n++;
	}

    if (fclose(fp) != 0 && cmsg == NULL)	/* Need error msg */
	write_error(savefile, "saving message");

    /*
     * Fill in a useful message about what we've done and return the number of
     * messages saved
     */
    PUSTRLCPY2(buf, bufsize, savefile, action);
    return n;
}


/*
 * rip()
 *
 * Replace terminating end of line marker(s) with null, and return the kind of
 * EOL marker that caused the termination; this may be '\r', '\n' or 0 if the
 * line wasn't terminated.
 */
char rip(char *s)
{
    char *cp;
    char ret = '\0';

    if ((cp = strchr(s, '\r')) != NULL)
    {
	ret = *cp;
	*cp = '\0';
    }
    if ((cp = strchr(s, '\n')) != NULL)
    {
	ret = *cp;
	*cp = '\0';
    }
    return ret;
}


/*
 * eoltest()
 *
 * Return '\n' if <line> ends with LF, 0 if not
 */
char eoltest(const char *s)
{
    size_t l = strlen(s);
    return (l > 0 && s[l - 1] == '\n') ? '\n' : '\0';
}


/*
 * tag_msg()
 *
 * Tag message <cmsg>
 */
static void tag_msg(PHTL cmsg)
{
    if (cmsg && !MSGISTAGGED(cmsg))
    {
	cmsg->status |= S_ISTAGGED;
	numtagged++;
    }
}


/*
 * untag_msg()
 *
 * Un-tag message <cmsg>
 */
void untag_msg(PHTL cmsg)
{
    if (cmsg && MSGISTAGGED(cmsg))
    {
	cmsg->status &= ~S_ISTAGGED;
	if (numtagged > 0)
	    numtagged--;
    }
}


/************************************************************************/

/*
 * List mailqueue
 */
static void listqueue(const char *path)
{
    struct ffblk buffer;
    char tstring[85];
    char host[SLINELEN];
    char to[SLINELEN];
    char line[SLINELEN];
    char from[SLINELEN];
    char errtxt[LINELEN];
    char *p, ch, *scp;
    int  done, maxlines, nmsg, endidx, pos, justdeleted, i;
    FILE *fp;
#define MAXLINES 60
    char *storelines[MAXLINES];

    maxlines = display.screenheight - 5;	/* max. lines we can show */
    if (maxlines > MAXLINES)
	maxlines = MAXLINES;
    pos = 0;
    do
    {
	clrscr();
	AUSTRLCPY2(tstring, "Unsent messages in ", mailqdir);
	quickcenter(1, col_heading, tstring);
	writef(1, 2, col_heading, seqstr(display.screenwidth, ' '));
	writef(1, 2, col_heading,
	       "S   Job-ID Host                 To                        From                 ");
	gotoxy(1, 3);
	nmsg = 0;
	done = findfirst(path, &buffer, 0);
	while (!done)
	{
	    AUSTRCPY(line, buffer.ff_name);
	    AUSTRLCPY3(tstring, mailqdir, "\\", line);
	    if ((fp = fopen(tstring, "r")) == NULL)
		fopen_error(tstring, "queued WRK file");
	    else
	    {
		if (nmsg == maxlines)	/* already full, with more to come */
		{
		    sprintf(errtxt, "More than %d msgs unsent",		/*OK*/
			    maxlines);
		    errordsp(errtxt);
		    (void) fclose(fp);			/* OK: read only */
		    sleep(2);
		    break;	/* out of the while (!done) */
		}
		if ((p = strrchr(line, '.')) != NULL)
		    *p = '\0';
		AUSTRLCPY4(tstring, mailqdir, "\\", line, lckfileext);
		if (access(tstring, 0))
		    ch = ' ';
		else
		    ch = 'L';
		if (fgets(host, sizeof(host), fp) == NULL)
		    host[0] = '\0';
		else
		    (void) rip(host);
		if (fgets(from, sizeof(from), fp) == NULL)
		    from[0] = '\0';
		else
		    (void) rip(from);
		if (fgets(to, sizeof(to), fp) == NULL)
		    to[0] = '\0';
		else
		    (void) rip(to);
		(void) fclose(fp);			/* OK: read only */
		sprintf(tstring, "%c %8s %-20.20s %-25.25s %-22.22s",	/*OK*/
			ch, line, host, to, from);
		gotoxy(1, 3 + nmsg);
		cputs(tstring);
		if ((storelines[nmsg] = strdup(tstring)) == NULL)
		{
		    errordsp(no_mem);
		    sleep(3);
		    break;    /* just do the ones we have got in memory then */
		}
		nmsg++;
	    }
	    done = (findnext(&buffer));
	}

	if (nmsg < 1)
	{
	    colputs(col_enhanc, "No mail awaiting transmission.");
	    get_cr(NULL);
	    return;
	}
	endidx = nmsg - 1;	/* end index for this list: endidx >= 0 */

	if (pos > endidx)	/* <pos> is over the end of the new list */
	    pos = endidx;	/* so that if we delete the last one we
				   will move up on the screen */
	justdeleted = done = FALSE;
	while (!done)
	{
	    gotoxy(62, display.screenheight - 1);
	    quickcenter(display.screenheight - 1, col_heading,
			"Cursor \2up/down\2, \2d\2elete, "
			"\2space\2/\2return\2/\2e\2dit text, "
			"\2w\2ork file, \2l\2ock, \2u\2nlock or \2ESC\2");
	    gotoxy(1, 3 + pos);
	    textattr(col_hilite);
	    cputs(storelines[pos]);		/*lint !e644 */	/* OK */
	    textattr(col_normal);
	    p = storelines[pos] + 2;
	    while (*p == ' ')
		p++;	/* move to the job id */
	    ustrncpy(line, p, 8);
	    if ((scp = strchr(line, ' ')) != NULL)
		*scp = '\0';
	    /* line now contains the string format of the job id */
	    gotoxy(1, display.screenheight);	/* move the cursor to the bottom */
	    cputs(command_prompt);
	    while ((i = (int) getkey()) == '\0')
		;
	    switch (i)
	    {
	    case ESC:
		done = TRUE;
		break;

	    case KKP8:
	    case KUP:
		if (pos > 0)		/* OK to decrement <pos> */
		{
		    gotoxy(1, 3 + pos);
		    cputs(storelines[pos]);
		    pos--;
		}
		else
		    beep();
		break;

	    case KKP2:
	    case KDN:
		if (pos < endidx)	/* OK to increment <pos> */
		{
		    gotoxy(1, 3 + pos);
		    cputs(storelines[pos]);	/*lint !e676 */ /* OK */
		    pos++;
		}
		else
		    beep();
		break;

	    case KKP7:
	    case KHOME:
		if (pos == 0)		/* Already at top of list */
		    beep();
		else
		{
		    gotoxy(1, 3 + pos);
		    cputs(storelines[pos]);
		    pos = 0;
		}
		break;

	    case KKP1:
	    case KEND:
		if (pos == endidx)	/* Already at end of list */
		    beep();
		else
		{
		    gotoxy(1, 3 + pos);
		    cputs(storelines[pos]);
		    pos = endidx;
		}
		break;

	    case 'e':
	    case ' ':
	    case KCR:
		AUSTRLCPY4(tstring, mailqdir, "\\", line,
			   mboxext_store[SYS_HAM]);
		if (slash_ed)		/* editor needs /, not \ */
		    fwdslash(tstring);
		(void) invoke_editor(NULL, tstring, NULL, FALSE, FALSE);
		break;

	    case 'w':
		AUSTRLCPY4(tstring, mailqdir, "\\", line, wrkfileext);
		if (slash_ed)		/* editor needs /, not \ */
		    fwdslash(tstring);
		(void) invoke_editor(NULL, tstring, NULL, FALSE, FALSE);
		break;

	    case 'u':
		if (*storelines[pos] != 'L')
		    beep();		/* it's not locked */
		else
		{
		    AUSTRLCPY4(tstring, mailqdir, "\\", line, lckfileext);
		    if (unlink(tstring))
			unlink_error(tstring, "lock file");
		    else
			*storelines[pos] = ' ';
		}
		break;

	    case 'l':
		if (*storelines[pos] == 'L')
		    beep();	/* it's already locked */
		else
		{
		    AUSTRLCPY4(tstring, mailqdir, "\\", line, lckfileext);
		    if ((fp = fopen(tstring, "w+")) != NULL)
			(void) fclose(fp);		/* OK: unused */
		    else
			fopen_error(tstring, "lock file");
		    *storelines[pos] = 'L';
		}
		break;

	    case 'd':
		if (askdelqueue == FALSE ||
		    y_n("Delete from the mail queue?"))
		{
		    if (killjob(line))
			while (getkey() == '\0')
			    ;
		    justdeleted = TRUE;
		    done = TRUE;
		}
		break;

	    default:
		beep();
		break;
	    }
	}

	/* Free the stored strings in <storelines> */
	for (i = 0; i < nmsg; i++)
	    free(storelines[i]);
    }
    while (justdeleted);	/* deleted one so re-read the whole screen */
}


/*
 * killjob()
 *
 * Delete the job with filename <j> from the queue directory; <j> is "1234" or
 * similar
 */
static int killjob(const char *j)
{
    char s[SLINELEN], errtxt[SLINELEN];
    char *p;
    int res, i;

    res = 0;
    AUSTRLCPY4(s, mailqdir, "\\", j, lckfileext);
    p = s + strlen(s) - strlen(lckfileext);	/* end of file basename */
    if (!access(s, 0) && unlink(s) != 0)
    {
	AUSTRLCPY2(errtxt, "Can't unlock job ", j);
	errorbox(NULL, errtxt);
	return 1;
    }
    for (i = 2; i < 4; i++)
    {
	strcpy(p, (i == 2) ? wrkfileext : mboxext_store[SYS_HAM]);
	if (unlink(s) != 0)
	{
	    AUSTRLCPY3(errtxt, "Job id ", j, " not found");
	    errorbox(NULL, errtxt);
	    res = i;
	}
    }
    return res;
}


/************************************************************************/

/*
 * get_cr()
 *
 * Prompt the user to press return to continue, with an (optional) message
 * string <msg>
 */
void get_cr(const char *msg)
{
    static int keylist[] = { KCR, ' ', ESC, 0 };

    (void) cr_prompt(msg, keylist);
}


/*
 * y_n()
 *
 * Prompt the user for a Y/N response to prompt <msg>; return TRUE if yes,
 * FALSE if no.
 */
BOOL y_n(const char *msg)
{
    static int keylist[] = { 'y', 'Y', 'n', 'N', 0 };
    int ch;

    ch = bottom_prompt(msg, keylist);
    return (toupper(ch) == 'Y') ? TRUE : FALSE;
}


/*
 * abort_retry()
 *
 * Prompt the user for an A/R response to prompt <msg>; return TRUE if abort,
 * FALSE if retry
 */
BOOL abort_retry(const char *msg)
{
    static int keylist[] = { 'a', 'A', 'r', 'R', 0 };
    int ch;

    ch = bottom_prompt(msg, keylist);
    return (toupper(ch) == 'A') ? TRUE : FALSE;
}


/*
 * view_cr()
 *
 * Prompt the user to press return to continue, with an (optional) message
 * string <msg>; this is a viewer prompt, so we respond to some viewer command
 * keys as well as the normal get_cr() Ret/Space/Esc ones.  Return 0 if it's a
 * normal get_cr() key that gets pressed, or the actual key if it's an
 * allowable viewer command key.
 */
int view_cr(const char *msg)
{
    static int keylist[] = { KCR, ' ', ESC,	/* Normal get_cr() keys */
			     'i', 'q', KKP8, KUP, KKP9, KPGUP,
			     'h', 'a', '8', '4', '!',
			     'r', 'R', 'g', 'G', 'f', 'm', 't',
			     'p', 'd', 'u', 'n', 's', 'w', 'e',
			     0 };
    int ch;

    ch = cr_prompt(msg, keylist);
    if (ch == KCR || ch == ' ' || ch == ESC)
	return 0;
    return ch;
}


/*
 * ignore_alt()
 *
 * Prompt the user whether to ignore an alternative block of text, using
 * prompt string <msg>; this is a viewer prompt, so we respond to some viewer
 * command keys as well as the normal Ret/Space/Esc/v ones.  Return 0 if it
 * should be viewed (v), 1 if it should be ignored (Ret/Space/Esc), or the
 * actual key (> 1) if it's an allowable viewer command key.
 */
int ignore_alt(const char *msg)
{
    static int keylist[] = { KCR, ' ', ESC, 'v',    /* Normal prompt keys */
			     'i', 'q', KKP8, KUP, KKP9, KPGUP,
			     'h', 'a', '8', '4', '!',
			     'r', 'R', 'g', 'G', 'f', 'm', 't',
			     'p', 'd', 'u', 'n', 's', 'w', 'e',
			     0 };
    int ch;

    ch = bottom_prompt(msg, keylist);
    if (ch == 'v')				/* View the alt. text */
	return 0;
    if (ch == KCR || ch == ' ' || ch == ESC)	/* Ignore the alt. text */
	return 1;
    return ch;					/* A viewer key */
}


/*
 * cr_prompt()
 *
 * Prompt the user to press return to continue, with an (optional) message
 * string <msg>; return when one of the keys in <keylist> is pressed
 */
static int cr_prompt(const char *msg, const int *keylist)
{
    const char *hit_return = "Hit return or ESC to continue";
//  const char *hit_return = "Hit Ret/ESC/space to continue";
    char t[81];
    int ch;

    if (msg)
	AUSTRLCPY3(t, msg, ". ", hit_return);
    else
	AUSTRCPY(t, hit_return);
    ch = bottom_prompt(t, keylist);
    if (msg == NULL)
	clrscr();
    return ch;
}


/*
 * bottom_prompt()
 *
 * Display prompt string <msg> on the bottom line of the screen, with the
 * cursor just past the end of the text (or at the end, if it takes up the
 * entire screen).  Wait for a key to be pressed that matches one of the ones
 * in <keylist> (which is ended by 0), then blank the line and return the
 * key pressed.  Restore the original cursor position once we're done.
 */
static int bottom_prompt(const char *msg, const int *keylist)
{
    const int *kp;
    int ox, oy, i, ch;

    ox = wherex();
    oy = wherey();
    i = bottom_text(col_select, msg);
    disp_gotoxy(i, display.screenheight);    /* move cursor to end of line */
    for (;;)
	if ((ch = (int) getkey()) > 0)
	{
	    for (kp = keylist; *kp > 0 && *kp != ch; kp++)
		;
	    if (*kp > 0)	/* Got a match! */
		break;	/* out of the for (;;) */
	    /* ST-FIXME: should we beep here because of the incorrect key? */
	}
    (void) bottom_text(col_normal, NULL);
    gotoxy(ox, oy);
    return ch;
}


/*
 * bottom_text()
 *
 * Show text <msg> on the bottom line of the screen, using attribute <attr>,
 * with an extra space on the end if there's room.  Blank the line with
 * attribute <col_normal> first; if <msg> is NULL or a null string, return
 * after the blanking.  Return the column position of the last character on
 * the line.
 */
static int bottom_text(int attr, const char *msg)
{
    writef(1, display.screenheight, col_normal,
	   seqstr(display.screenwidth, ' '));
    if (msg && *msg)
    {
	int l;

	writef(1, display.screenheight, attr, msg);
	if ((l = (int) strlen(msg)) < display.screenwidth)    /* add space */
	    writef(++l, display.screenheight, attr, " ");
	return l;
    }
    return 1;
}
