/*
  This source is part of PCOak, an electronic mailer for DOS based on PCElm.

  PCElm is Copyright (c) 1988-1993 Martin Freiss and Wolfgang Siebeck
           Copyright (c) 1992-1999 Demon Internet
  PCOak is Copyright (c) 2000-2002 Simon Turner, Pete Disdale and dispc members

  Thanks to an agreement between the original PCElm authors and Demon Internet
  made in late 1999:

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License, version 1, as
	published by the Free Software Foundation.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	See the file COPYING, which contains a copy of the GNU General
	Public License.
*/

#include <stdio.h>
#include <dos.h>
#include <conio.h>
#include <string.h>
#include <time.h>
#include "tcheader.h"
#include "pcoak.h"
#include "header.h"
#include "ustring.h"
#include "chars.h"
#include "display.h"
#include "macros.h"

#define FIRSTLINE	4	/* First line to use on index page */

/*
 * External global variables used in this module
 */
extern const char file_err[];		/* "File error" */
extern const char cant_open[];		/* "Can't open " */
extern const char no_mail[];		/* "No mail" */
extern const char somebody[];		/* "Somebody" */
extern BOOL index_onscreen;

/*
 * Functions defined and used in this module
 */
void colputs(int attr, const char *str);
void catputs(int attr, const char *str);
void writef(int x, int y, int attr, const char *line);
char *seqstr(size_t n, char ch);
void quickcenter(int line, int attr, const char *txt);
void errorbox(const char *title, ...);
void errordsp(const char *txt);
int get_hdrfield_body(HEADERTYPE hdrtype, FILE *fp, long pos, char *buf,
		      size_t bufsize, BOOL unfold);

/*
 * Functions defined elsewhere and used in this module
 */
char rip(char *s);
BOOL htype_match(const char *p, HEADERTYPE type);
void free_mbox(void);
void free_globals(void);
char *dispdate(time_t utc, long offset);


/*
 * colputs()
 *
 * "puts() in colour" -- set colour to <attr>, call cputs() to display text
 * <str>, then reset the attribute to the default <col_normal>.
 *
 * I don't think this does what the programmer wanted: he talks about
 * "attributes" above, but textcolor() only changed the TEXT FOREGROUND
 * colour, not the attribute.  This is why we get weird things like dark blue
 * on black for things that should be "highlighted".  Nonetheless, this is
 * what the users are used to, so we'll leave it like this for the time being.
 * Use catputs() below if you want to set the complete fore/back attribute.
 */
void colputs(int attr, const char *str)
{
    textcolor(attr);
    cputs(str);
    textcolor(col_normal);
}


/*
 * catputs()
 *
 * cputs() with the given attribute <attr>; the default attribute <col_normal>
 * is restored afterwards.  I *think* this is what colputs() is really
 * supposed to do!
 */
void catputs(int attr, const char *str)
{
    textattr(attr);
    cputs(str);
    textattr(col_normal);
}


/*
 * Now some screen-primitives ...
 */
BOOL init_colors(BOOL have_color)
{
    if (have_color)
    {
	col_normal = CO_NORMAL;
	col_enhanc = CO_ENHANC;
	col_hilite = CO_HILITE;
	col_select = CO_SELECT;
    }
    else
    {
	col_normal = MONO_NORMAL;
	col_enhanc = MONO_ENHANC;
	col_hilite = MONO_HILITE;
	col_select = MONO_SELECT;
    }
    return have_color;
}


/*
 * pokewritef()
 *
 * Direct-to-memory screen writing: write text <line>, starting at screen
 * position (<x>, <y>), using attribute <attr>: if we come across ASCII 2 (^B)
 * in the text, toggle "bold" on/off ("bold" characters are drawn with
 * attribute <col_enhanc> instead of the given <attr>).  Because this writes
 * direct to the video memory, it's not affected by things like the current
 * window, and it doesn't change the cursor position or current text
 * attribute.
 */
static void pokewritef(int x, int y, int attr, const char *line)
{
    char far *lp = (char far *) 0x00449L;  /* video mode byte */
    unsigned scrseg, scrofs, screen_width = *(unsigned far *) 0x44a;
    BOOL boldon;
    int store_attr = attr;
    register const char *s;

    scrseg = 0xb000;
    if (*lp != 7)
	scrseg += 0x800;

    boldon = FALSE;
    scrofs = ((((unsigned) y * (screen_width << 1))	/* offset from (0,0) */
	       - (screen_width << 1)) + ((unsigned) x << 1)) - 2;
    lp = MK_FP(scrseg, scrofs);
    for (s = line; *s != '\0'; s++)
    {
	switch (*s)
	{
	case '\2':			/* Bold on toggle on ^B */
	    boldon = !boldon;
	    if (boldon)
		attr = col_enhanc;	/* Set full intensity */
	    else
		attr = store_attr;	/* Reset user intensity */
	    break;

	default:
	    *lp++ = *s;
	    *lp++ = (char) attr;
#if 0
	    pokeb(scrseg, scrofs++, line[i]);	/* poke character ...*/
	    pokeb(scrseg, scrofs++, attr);		/* ... and attribute */
#endif
	}
    }
}


/**************************************************************************/

/*
 * writef()
 *
 * Screen writing: write text <line>, starting at screen position (<x>, <y>),
 * using attribute <attr>: if we come across ASCII 2 (^B) in the text, toggle
 * "bold" on/off ("bold" characters are drawn with attribute <col_enhanc>
 * instead of the given <attr>).  This calls pokewritef() if we're using
 * direct-to-memory video access, otherwise it uses the BIOS.  This is not
 * affected by the current window, text attribute or cursor position: if we
 * have to do it using the BIOS, we set the window to full-screen before we
 * start, and then restore the original window, text attribute and cursor
 * position after we've finished.
 */
void writef(int x, int y, int attr, const char *line)
{
    struct text_info ti;
    int i;
    BOOL boldon;

    if (directvideo)
    {
	pokewritef(x, y, attr, line);
	return;
    }
    /*
     * If using bios, we have to do it the hard way. Save current
     * window and cursor position to restore afterwards...
     */
    gettextinfo(&ti);
    window(1, 1, display.screenwidth, display.screenheight);
    gotoxy(x, y);
    boldon = FALSE;
    textattr(attr);
    for (i = 0; line[i] != '\0'; i++)
    {
	switch (line[i])
	{
	case '\2':			/* Bold on toggle on ^B */
	    boldon = !boldon;
	    if (boldon)
		textattr(col_enhanc);	/* Set full intensity */
	    else
		textattr(attr);		/* Reset user attribute */
	    break;

	default:
	    cprintf("%c", line[i]);
	    break;
	}
    }
    /* Restore previous window, attribute and cursor position */
    window(ti.winleft, ti.wintop, ti.winright, ti.winbottom);
    textattr(ti.attribute);
    gotoxy(ti.curx, ti.cury);
}


/**************************************************************************/

/*
 * Return a pointer to a static array containing <n> instances of char <ch>
 */
char *seqstr(size_t n, char ch)
{
    static char tempst[LINELEN];
    size_t i;

    if (n >= sizeof(tempst))	/* Too many chars: reduce requested length */
	n = sizeof(tempst) - 1;
    for (i = 0; i < n; i++)
	tempst[i] = ch;
    tempst[i] = '\0';
    return tempst;
}


/**************************************************************************/

/*
 * quickcenter()
 *
 * Write text <txt> on screen line <line> using attribute <attr>, centred on
 * an otherwise blank line.  If <attr> is not <col_heading>, blank the line
 * using the <col_normal> attribute; if it's <col_heading>, use that attribute
 * for blanking as well as the text (this allows us to have a full line of the
 * heading colour, similar to Snews heading block).
 */
void quickcenter(int line, int attr, const char *txt)
{
    int x, i, j;

    if (attr == col_heading)
	x = attr;
    else
	x = col_normal;
    writef(1, line, x, seqstr(display.screenwidth, ' '));
    if (*txt == '\0')
	return;
    x = (int) (display.screenwidth / 2 - (strlen(txt) / 2));
    /*
     * Count control characters in <j> before centering
     */
    for (i = 0, j = 0; txt[i] != '\0'; i++)
	if (txt[i] < ' ')
	    j++;
    j /= 2;
    x += j;
    if (x < 1)
	x = 1;
    writef(x, line, attr, txt);
}


/*
 * errordsp()
 *
 * Display error message <txt> centred on the next-to-last line of the screen,
 * blanking the list first and using the <col_hilite> attribute for the text.
 *
 * Note that quickcenter() uses writef(), which ignores the current window
 * setting, and leaves the text attribute and cursor position unchanged after
 * it's finished; since we get it to use <col_hilite> under all circumstances,
 * there is little point in setting the text attribute before and after
 * calling errordsp(), as seems to have been done in various places!
 */
void errordsp(const char *txt)
{
    quickcenter(display.screenheight - 1, col_hilite, txt);
}


/**************************************************************************/

/*
 * clreos()
 *
 * Clear from the current cursor position to the end of the screen
 */
void clreos(void)
{
    int oldx, oldy, y;

    clreol();				/* clear current line */
    oldx = wherex();
    y = oldy = wherey();
    while (++y <= display.screenheight)	/* and down the window */
    {
	gotoxy(1, y);
	clreol();
    }
    gotoxy(oldx, oldy);			/* return to old curpos */
}


/**************************************************************************/

/*
 * get_current_attribute()
 *
 * Return the current screen attribute; only called from config.c, which uses
 * it to store the original system attribute <col_system> at program startup
 */
int get_current_attribute(void)
{
    union REGS REG;

    REG.h.ah = 0x08;			/* read char/attr */
    REG.x.bx = 0;			/* page number */
    int86(0x10, &REG, &REG);		/* execute intr */
    return REG.h.ah;			/* return attr @ cursor */
}


/**************************************************************************/

/*
 * get_hdrfield_body()
 *
 * Jump to position <pos> in file <fp>, which is expected to be at the start
 * of a line in the header of the message, and search for a header field of
 * type <hrdtype>; once found, copy the field body (without leading and
 * trailing whitespace) into <buf>.  If <unfold> is true, any folded
 * continuation lines should be unfolded into <buf> as well, without trailing
 * and leading whitespace.  Return -1 if we couldn't seek to the file
 * position, 0 if we actually copied the body contents properly, or 1 if we
 * got to the end of the header without finding it.
 */
int get_hdrfield_body(HEADERTYPE hdrtype, FILE *fp, long pos, char *buf,
		      size_t bufsize, BOOL unfold)
{
    char tstring[LINELEN];
    BOOL got_start = FALSE;
    char got_eol, last_eol = 1;

    if (fseek(fp, pos, SEEK_SET) != 0)
	return -1;

    while (fgets(tstring, sizeof(tstring), fp) != NULL)
    {
	if (last_eol && ISBLANKLINE(tstring))	/* end of header */
	    break;	/* out of the while (fgets) */
	got_eol = rip(tstring);
	kill_trail_lwsp(tstring);
	if (got_start)		/* Look for folded continuation lines */
	{
	    if (!last_eol)		/* direct continuation of last line */
		PUSTRCAT(buf, bufsize, tstring);
	    else
	    {
		if (!islwsp(*tstring))		/* Not a folded continuation */
		    break;	/* out of the while (fgets) */
		PUSTRLCAT2(buf, bufsize, " ", skip_lwsp(tstring));
	    }
	}
	else if (last_eol && htype_match(tstring, hdrtype))
	{				/* Found start of header field */
	    PUSTRCPY(buf, bufsize,
		     skip_lwsp(tstring + strlen(headername[hdrtype])));
	    got_start = TRUE;
	    if (!unfold)
		break;	/* out of the while (fgets) */
	}
	last_eol = got_eol;
    }

    return (got_start) ? 0 : 1;
}


/*
 * copy_name()
 *
 * Extract the human name from an RFC [2]822 address in <str> and copy it into
 * <name> (which has size <namesize>); return a pointer to <name>.  If we
 * can't find a full name, copy the mail address instead; if the line is
 * blank, just make <name> a blank string.
 *
 * Assumes there is only one address in <str>, and expects it to have one of
 * these forms:
 *	user@a.b.c (John Doe)
 *	Jane Doe <user@a.b.c>
 *	"Jane Doe" <user@a.b.c>
 * Within these limitations, works reasonably well.
 */
static void copy_name(char *name, size_t namesize, const char *str)
{
    const char *p, *q;
    BOOL quote = FALSE, gotit = FALSE;
    size_t len;

    /* Find the last non-LWSP character in <str> */
    for (p = str + strlen(str) - 1; p > str && islwsp(*p); p--)
	;

    if (p > str && *p == ')')		/* user@a.b.c (John Doe) */
    {
	q = p;				/* Note end of bracketed comment */
	while (p > str && !gotit)
	{
	    switch (*--p)
	    {
	    case '"':
		quote = !quote;
		break;

	    case '(':			/* Found start of full name comment */
		if (!quote)
		{
		    p++;		/* move to start of name */
		    gotit = TRUE;	/* full name from <p> to <q-1> */
		}
		break;

	    default:
		break;
	    }
	}
    }
    else if (p > str && *p == '>') 	/* Jane Doe <user@a.b.c> */
    {
	while (p > str && !gotit)
	{
	    switch (*--p)
	    {
	    case '"':
		quote = !quote;
		break;

	    case '<':			/* Found start of mail address <...> */
		if (!quote)
		{
		    if (p > str)		/* There's a long name */
		    {
			q = p;
			p = str;
			gotit = TRUE;	/* full name from <p> to <q-1> */
		    }
		    /* else p == str: will fall out of the while (p) */
		}
		break;

	    default:
		break;
	    }
	}
    }

    if (!gotit)		/* couldn't find the full name: use the whole thing */
    {
	p = str;
	q = str + strlen(str);		/* use name from <p> to <q-1> */
    }
    
    /*
     * We've now got a name (full or otherwise) running from <p> to <q-1>
     * inclusive.  Skip any leading or trailing whitespace, quotes and <...>
     * to give us the "plain text" version of the name, then copy that into
     * <name>.
     */
    p = skip_lwsp(p);
    while (q > p && islwsp(q[-1]))		/*lint !e644 */ /* OK */
	q--;
    if (*p == '"' || *p == '<')		/* Look for trailing '"' or '>' */
    {
	if (q > p + 1 && q[-1] == '"' || q[-1] == '>')	/* Kill "" or <> */
	{
	    q--;
	    p++;
	}
    }

    /* We've got the final name to use, running from <p> to <q-1> inclusive */
    len = (size_t) (q - p);		/* length of name */
    if (len >= namesize)		/* too long for buffer */
	len = namesize - 1;
    ustrncpy(name, p, len);
}


/*
 * get_display_name()
 *
 * Extract the human name (or, failing that, the SMTP address) from the "From"
 * line of <cmsg> and return it.  If the name is the same as global
 * <fullname>, the mail was sent by the user, so use the "To" address instead
 * (if, and only if, there is actually a "To" address to use!).
 */
static const char *get_display_name(PCHTL cmsg)
{
    static char ret[LINELEN];

    copy_name(ret, sizeof(ret), cmsg->from);
    if (*fullname && strcmp(ret, fullname) == 0)
    {				/* It's from the user: use the "To:" line */
	char tstring[LINELEN];

	/* We don't have the "To:" line in memory, so get it from the file */
	if (get_hdrfield_body(HT_TO, mfile, cmsg->position, tstring,
			      sizeof(tstring), FALSE) == 0)
	{
	    char *p;
	    if ((p = strchr(tstring, ',')) != NULL)	/* Stop at 1st ',' */
		*p = '\0';
	    strcpy(ret, "To: ");
	    copy_name(ret + 4, sizeof(ret) - 4, tstring);
	}
    }
    return ret;
}


/*
 * showaline()
 *
 * Display a line describing message number <msg> at the current cursor
 * position.  This uses the entire 80-column width of the screen, and takes
 * the following basic form:
 *
 *   N  2 Pete Disdale         3 Sep 17:26   3638 Re: PCOak save 
 *  D  27 To: Pete Disdale    22 Mar 08:15   1472 Re: PCOak date display
 * M  459+Fred                17 Jan 23:59 107326 Security update
 * M  1002 Someone            14 Dec 12:21 2108372 Large attachment, sorry
 *
 * Note: the separate columns for (D)eleted and (N)ew, which is unlike Elm,
 * but which I prefer (it's easier to see at a glance which messages have been
 * dealt with and deleted, and which ones are new); the old 'T' column to
 * indicate Tagged status has become a '+' between the message number and the
 * author field -- the author field reduces in length from 20 to 19 if the
 * message number is 1000 or more; the 6-character field for message length,
 * followed by a 30-character field for subject (which now runs up to the last
 * column to maximise screen use) -- the subject field width reduces to 29 or
 * 28 characters if the message length is 7 or 8(!) digits respectively.
 * This assumes that we will never have more than 9,999 messages in a folder,
 * or messages larger than 99,999,999 bytes (which I think is reasonable).
 *
 * This is still a ghastly kludge, but it's a bit better than before IMHO.
 * What it needs is proper user control over field widths and formatting.
 * ST-FIXME!
 */
static void showaline(PCHTL cmsg)
{
    int addrwidth = (cmsg->listno < 1000) ? 22 : 21;
    int subjwidth = (cmsg->size < 1000000L) ? 30 :
	(cmsg->size < 10000000L) ? 29 : 28;

    clreol();
    cprintf("%c%c%c%3d%c%-*.*s %-12.12s %6ld %.*s",
	    (cmsg->status & S_MIME)     ? 'M' : ' ',
	    (cmsg->status & S_DELETE)   ? 'D' : ' ',
	    (cmsg->status & S_READ)     ? ' ' : 'N',
	    cmsg->listno,
	    (cmsg->status & S_ISTAGGED) ? '+' : ' ',
	    addrwidth, addrwidth,
	    (cmsg->headers & H_FROM) ? get_display_name(cmsg) : somebody,
	    (cmsg->headers & H_DATE) ?
	    dispdate((time_t) cmsg->utc, (long) cmsg->offmins * 60L) : "",
	    cmsg->size,
	    subjwidth,
	    (cmsg->headers & H_SUBJECT) ? cmsg->subject : "");
}


/*
 * show_index()
 *
 * Show the index screen, arranging things so that <current> is somewhere on
 * the screen and is highlit.  This may involve showing a different part of
 * the mailbox than we were previously showing; if this is the case, try to
 * move by a screenful each time.
 */
void show_index(PCHTL current)
{
    static int from = 1;		/* Message no. for top index line */
    static int oldfrom = -999;		/* Last top line message no. */
    static PCHTL oldcurrent = NULL;	/* Last current msg pointer */
    int maxfrom = nmsgs - display.main_lines + 1;

    if (mbox == NULL || current == NULL)
    {
	errordsp(no_mail);
	return;
    }
    while (current->listno >= from + display.main_lines)
	from += display.main_lines;
    while (current->listno < from)
	from -= display.main_lines;
    if (from > maxfrom)
	from = maxfrom;
    if (from < 1)
	from = 1;
    /*
     * If we're currently displaying the index page and the top of page hasn't
     * changed, we only need to redraw the last highlit line to get the
     * display back to a "nothing is highlit" state; our old pointer
     * <oldcurrent> should be guaranteed to be valid!
     */
    if (oldfrom == from && index_onscreen && oldcurrent != NULL)
    {				/* Redraw last current line */
	gotoxy(1, FIRSTLINE + oldcurrent->listno - from);
	showaline(oldcurrent);
    }
    else			/* Redraw entire display */
    {
	PCHTL cmsg = current;
	int i;

	while (cmsg->listno < from && cmsg->next != NULL)
	    cmsg = cmsg->next;
	while (cmsg->listno > from && cmsg->prev != NULL)
	    cmsg = cmsg->prev;
	for (i = 0; i < display.main_lines; i++)
	{
	    gotoxy(1, FIRSTLINE + i);
	    if (cmsg != NULL)
	    {
		showaline(cmsg);
		textattr(col_normal);
		cmsg = cmsg->next;
	    }
	    else
		clreol();
	}
    }
    textattr(col_hilite);
    gotoxy(1, FIRSTLINE + current->listno - from);
    showaline(current);
    oldcurrent = current;
    oldfrom = from;
    index_onscreen = TRUE;
    textattr(col_normal);
}


/*
 * cleanup()
 *
 * This is called by atexit(); use it to free all allocated storage, close
 * open files etc.
 */
void cleanup(void)
{
    cputs("\r\n");
    textattr(col_system);	/* prepare to clear screen */
    free_mbox();
    free_globals();
}


/*
 * select_max()
 *
 * Return a pointer to a static string, of form
 *	"Select your choice (1 - %d):  "
 * which can be used in getwin() prompt boxes; this form is used in several
 * places
 */
const char *select_max(int maxnum)
{
    static char str[] = "Select your choice (1 - 99):    ";

    if (maxnum < 1 || maxnum > 99)
	return "";
    sprintf(str + 24, "%d):  ", maxnum);				/*OK*/
    return str;
}


/*
 * fopen_error()
 *
 * Show an error box warning the user about a file open error
 */
void fopen_error(const char *filename, const char *descrip)
{
    char buf[81];

    AUSTRLCPY2(buf, cant_open, descrip);
    errorbox(file_err, filename, buf, NULL);
}


/*
 * write_error()
 *
 * Show an error box warning the user about a write error, which we presume
 * is caused by running out of disk space
 */
void write_error(const char *filename, const char *descrip)
{
    char buf[81];

    AUSTRLCPY2(buf, "Write error while ", descrip);
    errorbox(file_err, filename, buf, "Out of disk space?", NULL);
}


/*
 * unlink_error()
 *
 * Show an error box warning the user about a file unlink failure
 */
void unlink_error(const char *filename, const char *descrip)
{
    char buf[81];

    AUSTRLCPY2(buf, "Can't remove ", descrip);
    errorbox(file_err, filename, buf, NULL);
}


/*
 * rename_error()
 *
 * Show an error box warning the user about a file rename failure
 */
void rename_error(const char *fromname, const char *toname)
{
    errorbox(file_err, "Error renaming", fromname, "to", toname, NULL);
}
