/*
  This source is part of PCOak, an electronic mailer for DOS based on PCElm.

  PCElm is Copyright (c) 1988-1993 Martin Freiss and Wolfgang Siebeck
           Copyright (c) 1992-1999 Demon Internet
  PCOak is Copyright (c) 2000 Simon Turner, Pete Disdale and dispc members

  Thanks to an agreement between the original PCElm authors and Demon Internet
  made in late 1999:

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License, version 1, as
	published by the Free Software Foundation.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	See the file COPYING, which contains a copy of the GNU General
	Public License.
*/

/*
 * ustring.c: useful string functions for PCOak
 */

#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include "ustring.h"

/*
 * Functions defined and used here
 */
static int vustrlcpy(char *dest, size_t destsize, const char *src, va_list ap);


/************************************************************************/
/* "Useful" functions (which never do any overrun reporting)		*/
/************************************************************************/

/*
 * ustrncpy()
 *
 * "Useful" replacement for strncpy(), which ensures that the result is
 * null-terminated.  Yes, I know that this can be more efficiently done using
 * a macro, thus:
 *
 *   #define ustrncpy(d,s,n)    strncpy((d), (s), (n))[(n)] = '\0'
 *
 * but it's safer as a function, especially since the macro form uses the <n>
 * argument twice, which could produce unexpected side-effects.  Making it a
 * function also allows us to return the <dest> pointer, like the normal
 * strncpy().
 */
char *ustrncpy(char *dest, const char *src, size_t maxlen)
{
    strncpy(dest, src, maxlen)[maxlen] = '\0';
    return dest;
}


/************************************************************************/
/* "Safe" functions (which report if REPORT_OVERRUNS is defined)	*/
/************************************************************************/

/*
 * ustrcpy()
 *
 * "Safe" replacement for strcpy(), which takes the size of the <dest> buffer
 * and copies at most <destsize> - 1 bytes into it, before ensuring that the
 * result is null terminated.  If the <src> string is longer than this and
 * REPORT_OVERRUNS is defined, we complain.
 */
char *ustrcpy(char *dest, size_t destsize,
#ifdef REPORT_OVERRUNS
	      const char *filename, int line, const char *destname,
#endif
	      const char *src)
{
    size_t len = strlen(src);

    if (len >= destsize)		/* would be an overrun */
    {
#ifdef REPORT_OVERRUNS
	warn_overrun(dest, destsize, len, filename, line, destname);
#endif
	len = (destsize == 0) ? 0 : (destsize - 1);
    }
    memcpy(dest, src, len);
    dest[len] = '\0';
    return dest;
}


/*
 * ustrcat()
 *
 * "Safe" replacement for strcat() -- append as many characters as possible
 * from <src> to the end of the string in <dest>, subject to never exceeding
 * <destsize> - 1 characters in <dest> (leaving room for the null terminator,
 * which is always appended to <dest>).
 */
char *ustrcat(char *dest, size_t destsize,
#ifdef REPORT_OVERRUNS
	      const char *filename, int line, const char *destname,
#endif
	      const char *src)
{
    size_t dlen = strlen(dest);
    size_t slen = strlen(src);

    if (dlen + slen >= destsize)	/* would be an overrun */
    {
#ifdef REPORT_OVERRUNS
	warn_overrun(dest, destsize, dlen + slen, filename, line, destname);
#endif
	if (dlen >= destsize)		/* dest already too long: truncate */
	{
	    dest[destsize - 1] = '\0';
	    return dest;
	}
	slen = destsize - dlen - 1;
    }
    memcpy(dest + dlen, src, slen);
    dest[dlen + slen] = '\0';
    return dest;
}


/*
 * ustrlcat()
 *
 * Somewhat like ustrcat() above, but taking a list of <src> strings to
 * append to the destination string, rather than just one.  The list of
 * strings to append is ended by a NULL pointer or a null string.
 */
char *ustrlcat(char *dest, size_t destsize,
#ifdef REPORT_OVERRUNS
	       const char *filename, int line, const char *destname,
#endif
	       const char *src, ...)
{
    size_t dlen = strlen(dest);
    va_list ap;

    if (dlen >= destsize)		/* dest already too long: truncate */
    {
#ifdef REPORT_OVERRUNS
	size_t slen = strlen(src);	/* find total length of src strings */
	va_start(ap, src);
	while ((src = va_arg(ap, const char *)) != NULL && *src)
	    slen += strlen(src);
	va_end(ap);
	warn_overrun(dest, destsize, dlen + slen, filename, line, destname);
#endif
	dest[destsize - 1] = '\0';
	return dest;
    }

    va_start(ap, src);
#ifdef REPORT_OVERRUNS
    {
	int n;
	if ((n = vustrlcpy(dest + dlen, destsize - dlen, src, ap)) != 0)
	    warn_overrun(dest, destsize, dlen + (size_t) n, filename, line,
			 destname);
    }
#else
    (void) vustrlcpy(dest + dlen, destsize - dlen, src, ap);
#endif
    va_end(ap);
    return dest;
}


/*
 * ustrlcpy()
 *
 * Somewhat like ustrlcat() above, but we're setting the entire contents
 * of the array, not just adding stuff to the end; we take a list of <src>
 * strings which should be concatenated to form the destination string.  The
 * list of strings to append is ended by a NULL pointer or a null string.
 */
char *ustrlcpy(char *dest, size_t destsize,
#ifdef REPORT_OVERRUNS
	       const char *filename, int line, const char *destname,
#endif
	       const char *src, ...)
{
    va_list ap;

    va_start(ap, src);
#ifdef REPORT_OVERRUNS
    {
	int n;
	if ((n = vustrlcpy(dest, destsize, src, ap)) != 0)
	    warn_overrun(dest, destsize, (size_t) n, filename, line, destname);
    }
#else
    (void) vustrlcpy(dest, destsize, src, ap);
#endif
    va_end(ap);
    return dest;
}


/*
 * vustrlcpy()
 *
 * Take a list of source strings (const char*) comprising <src> and the
 * varargs list <ap>, and concatenate them into the <destsize> bytes of
 * destination buffer <dest>, ensuring that we don't overrun the buffer, and
 * that the final result is null terminated (we copy at most <destsize> - 1
 * bytes before terminating it).  The list of strings to append is ended by a
 * NULL pointer or a null string.
 *
 * Return 0 if we had enough room for all the strings; non-zero if not (if
 * REPORT_OVERRUNS is defined, we actually return the total size of all the
 * string we were asked to concatenate so that it can be used in a report).
 */
static int vustrlcpy(char *dest, size_t destsize, const char *src, va_list ap)
{
    size_t len;
#ifdef REPORT_OVERRUNS
    size_t srcbytes = 0;
#endif

    do
    {
	if ((len = strlen(src)) < destsize)	/* there's enough room */
	{
	    memcpy(dest, src, len);
	    dest += len;		/* point to start of next string */
	    destsize -= len;		/* destsize will be > 0 */
#ifdef REPORT_OVERRUNS
	    srcbytes += len;
#endif
	}
	else					/* will overrun buffer */
	{
	    memcpy(dest, src, --destsize);
	    dest[destsize] = '\0';	/* ensure it's terminated */
#ifdef REPORT_OVERRUNS
	    srcbytes += len;
	    while ((src = va_arg(ap, const char *)) != NULL && *src)
		srcbytes += strlen(src);
	    return (int) srcbytes;	/* return total size of all src */
#else
	    return 1;			/* just return 1 to indicate error */
#endif
	}
    }
    while ((src = va_arg(ap, const char *)) != NULL && *src);

    *dest = '\0';		/* ensure <dest> is terminated */
    return 0;
}


#ifdef REPORT_OVERRUNS

#include <dos.h>

typedef int BOOL;

int dispwin(int x, int y, int winattr, const char *title, BOOL squeeze_blanks,
	    ...);

extern const char* pressakey;		/* from maxgets.c */

void warn_overrun(const char *buf, size_t bufsize, size_t srclen,
		  const char *filename, int line, const char *bufname)
{
    char errlines[3][128];

    sound(880);
    delay(100);
    nosound();
    delay(50);
    sound(650);
    delay(100);
    nosound();

    sprintf(errlines[0], "File %.12s, line %d, buffer \"%.16s\"",
	    filename, line, bufname);
    sprintf(errlines[1], "Buffer size %u, source string length %u",
	    bufsize, srclen);
    sprintf(errlines[2], "\"%.72s\"", buf);
    (void) dispwin(0, 0, 0x4e, "*** BUFFER OVERRUN ***", 0,
		   errlines[0], errlines[1], errlines[2],
		   "", pressakey, NULL);
}

#endif	/* REPORT_OVERRUNS */
