/*
  This source is part of PCOak, an electronic mailer for DOS based on PCElm.

  PCElm is Copyright (c) 1988-1993 Martin Freiss and Wolfgang Siebeck
           Copyright (c) 1992-1999 Demon Internet
  PCOak is Copyright (c) 2000 Simon Turner, Pete Disdale and dispc members

  Thanks to an agreement between the original PCElm authors and Demon Internet
  made in late 1999:

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License, version 1, as
	published by the Free Software Foundation.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	See the file COPYING, which contains a copy of the GNU General
	Public License.
*/

/*
 * User string functions which ensure (a) no buffer overruns, and (b) all
 * strings are always null terminated.  There are two distinct flavours:
 *
 * "Safe" functions are used to replace blind copying (like strcpy(), strcat()
 * etc.): they take buffer size as an argument, and will only copy <size> - 1
 * bytes to allow room for the terminator, which is ALWAYS added; if the
 * string to be copied is longer than this, they complain about the overrun.
 *
 * "Useful" functions are used to replace careful copying (strncpy() only, at
 * the moment): they take the maximum number of characters to be copied
 * EXCLUDING the null terminator; this number is expected to be at least one
 * byte less than the size of the buffer, so that there will always be room
 * for the terminator -- which, unlike strncpy(), is ALWAYS added (even if the
 * string being copied is longer than the no. of character specified).  No
 * complaint will be raised if the string being copied is too long, as this is
 * the main reason to use strncpy() in the first place!
 *
 * ustrcpy	"Safe" replacement for strcpy()
 * ustrcat	"Safe" replacement for strcat()
 * ustrlcpy	"Safe" function to copy multiple strings into the destination
 *		  buffer (takes a varargs list of strings to copy)
 * ustrlcat	"Safe" function to add multiple strings on to the end of the
 *		  destination buffer (takes a varargs list of strings to add)
 *
 * ustrncpy	"Useful" replacement for strncpy()
 *
 * All functions return their first argument (the destination string).  Note
 * that the "safe" functions take the buffer size as the SECOND argument --
 * immediately after the buffer pointer -- but ustrncpy() takes the no. of
 * characters to copy as the LAST argument, like ordinary strncpy().
 *
 * Note that ustrcat() differs from the standard library function strncat() in
 * that we're specifying the size of the buffer, rather than the maximum
 * number of characters to append; strncat() is all about copying only some of
 * the source string, rather than protecting the destination buffer, and is
 * therefore of little interest to us.
 *
 * The functions provided here have the following prototypes:
 *
 *    char *ustrcpy(char *dest, size_t destsize, const char *src);
 *    char *ustrcat(char *dest, size_t destsize, const char *src);
 *    char *ustrlcat(char *dest, size_t destsize, const char *src, ...);
 *    char *ustrlcpy(char *dest, size_t destsize, const char *src, ...);
 *    char *ustrncpy(char *dest, const char *src, size_t maxlen);
 */

/*
 * We define quick time-saver macros for calling the ustr*() functions with
 * arrays as the destination (AU*): the size of the destination buffer is
 * found from sizeof().  Using sizeof() for the size of an array removes the
 * chance of typing in the wrong size, which is alarmingly common!  We can't
 * define general multi-argument macros for the ustrl*() functions; so we
 * provide 2-, 3-, 4- and 5-argument versions which should cover most
 * eventualities.
 *
 * For symmetry, and to prevent people missing off the NULL at the end for the
 * strl*() varargs-list functions, we also provide PU* functions for use with
 * pointers to buffers of a known size (the size cannot be determined from
 * sizeof(), because they are pointers rather than arrays).
 */

/*
 * If defined, REPORT_OVERRUNS enables overrun reporting: the "safe" macros
 * AUSTR*() and PUSTR*() pass extra information to the ustr*() functions, and
 * this information is used to produce detailed buffer overrun reports which
 * are displayed to the user (and, with luck, fed back to the developers!).
 *
 * There is also a CHKSTR() macro (which can be used anywhere, and is
 * typically used to check on buffers after a potentially dangerous sprintf()
 * operation): if REPORT_OVERRUNS is defined, is checks the buffer's contents
 * and produces a report if there is a problem; if not, the macro reduces to a
 * simple "do {} while (0)" which should be optimised away.
 */

#define REPORT_OVERRUNS		/* Check for and report on string overruns */


#ifdef REPORT_OVERRUNS

/************************************************************************/
/*	Provide detailed reports on overruns				*/
/************************************************************************/

char *ustrcpy(char *dest, size_t destsize, const char *filename, int line,
	      const char *destname, const char *src);
char *ustrcat(char *dest, size_t destsize, const char *filename, int line,
	      const char *destname, const char *src);
char *ustrlcat(char *dest, size_t destsize, const char *filename, int line,
	       const char *destname, const char *src, ...);
char *ustrlcpy(char *dest, size_t destsize, const char *filename, int line,
	       const char *destname, const char *src, ...);

char *ustrncpy(char *dest, const char *src, size_t maxlen);

void warn_overrun(const char *buf, size_t bufsize, size_t srclen,
		  const char *filename, int line, const char *bufname);

#define CHKSTR(s,l)							\
    do {								\
	size_t sl;							\
	if ((sl = strlen(s)) >= (l))					\
	    warn_overrun((s), (l), sl, __FILE__, __LINE__, #s);		\
    } while (0)

/* Array macros */

#define AUSTRCPY(d,s)							\
    ustrcpy((d),sizeof(d),__FILE__,__LINE__,#d,(s))
#define AUSTRCAT(d,s)							\
    ustrcat((d),sizeof(d),__FILE__,__LINE__,#d,(s))

#define AUSTRLCAT2(d,s1,s2)						\
    ustrlcat((d),sizeof(d),__FILE__,__LINE__,#d,(s1),(s2),NULL)
#define AUSTRLCAT3(d,s1,s2,s3)						\
    ustrlcat((d),sizeof(d),__FILE__,__LINE__,#d,(s1),(s2),(s3),NULL)
#define AUSTRLCAT4(d,s1,s2,s3,s4)					\
    ustrlcat((d),sizeof(d),__FILE__,__LINE__,#d,(s1),(s2),(s3),(s4),NULL)
#define AUSTRLCAT5(d,s1,s2,s3,s4,s5)					\
    ustrlcat((d),sizeof(d),__FILE__,__LINE__,#d,(s1),(s2),(s3),(s4),(s5),NULL)

#define AUSTRLCPY2(d,s1,s2)						\
    ustrlcpy((d),sizeof(d),__FILE__,__LINE__,#d,(s1),(s2),NULL)
#define AUSTRLCPY3(d,s1,s2,s3)						\
    ustrlcpy((d),sizeof(d),__FILE__,__LINE__,#d,(s1),(s2),(s3),NULL)
#define AUSTRLCPY4(d,s1,s2,s3,s4)					\
    ustrlcpy((d),sizeof(d),__FILE__,__LINE__,#d,(s1),(s2),(s3),(s4),NULL)
#define AUSTRLCPY5(d,s1,s2,s3,s4,s5)					\
    ustrlcpy((d),sizeof(d),__FILE__,__LINE__,#d,(s1),(s2),(s3),(s4),(s5),NULL)

/* Pointer + size macros */

#define PUSTRCPY(d,n,s)							\
    ustrcpy((d),(n),__FILE__,__LINE__,#d,(s))
#define PUSTRCAT(d,n,s)							\
    ustrcat((d),(n),__FILE__,__LINE__,#d,(s))

#define PUSTRLCAT2(d,n,s1,s2)						\
    ustrlcat((d),(n),__FILE__,__LINE__,#d,(s1),(s2),NULL)
#define PUSTRLCAT3(d,n,s1,s2,s3)					\
    ustrlcat((d),(n),__FILE__,__LINE__,#d,(s1),(s2),(s3),NULL)
#define PUSTRLCAT4(d,n,s1,s2,s3,s4)					\
    ustrlcat((d),(n),__FILE__,__LINE__,#d,(s1),(s2),(s3),(s4),NULL)
#define PUSTRLCAT5(d,n,s1,s2,s3,s4,s5)					\
    ustrlcat((d),(n),__FILE__,__LINE__,#d,(s1),(s2),(s3),(s4),(s5),NULL)

#define PUSTRLCPY2(d,n,s1,s2)						\
    ustrlcpy((d),(n),__FILE__,__LINE__,#d,(s1),(s2),NULL)
#define PUSTRLCPY3(d,n,s1,s2,s3)					\
    ustrlcpy((d),(n),__FILE__,__LINE__,#d,(s1),(s2),(s3),NULL)
#define PUSTRLCPY4(d,n,s1,s2,s3,s4)					\
    ustrlcpy((d),(n),__FILE__,__LINE__,#d,(s1),(s2),(s3),(s4),NULL)
#define PUSTRLCPY5(d,n,s1,s2,s3,s4,s5)					\
    ustrlcpy((d),(n),__FILE__,__LINE__,#d,(s1),(s2),(s3),(s4),(s5),NULL)


#else	/* !REPORT_OVERRUNS */

/************************************************************************/
/*	Don't provide reports on overruns				*/
/************************************************************************/

char *ustrcpy(char *dest, size_t destsize, const char *src);
char *ustrcat(char *dest, size_t destsize, const char *src);
char *ustrlcat(char *dest, size_t destsize, const char *src, ...);
char *ustrlcpy(char *dest, size_t destsize, const char *src, ...);
char *ustrncpy(char *dest, const char *src, size_t maxlen);

#define CHKSTR(s,l)	do {} while (0)

/* Array macros */

#define AUSTRCPY(d,s)		ustrcpy((d), sizeof(d), (s))
#define AUSTRCAT(d,s)		ustrcat((d), sizeof(d), (s))

#define AUSTRLCAT2(d,s1,s2)					\
    ustrlcat((d), sizeof(d), (s1), (s2), NULL)
#define AUSTRLCAT3(d,s1,s2,s3)					\
    ustrlcat((d), sizeof(d), (s1), (s2), (s3), NULL)
#define AUSTRLCAT4(d,s1,s2,s3,s4)				\
    ustrlcat((d), sizeof(d), (s1), (s2), (s3), (s4), NULL)
#define AUSTRLCAT5(d,s1,s2,s3,s4,s5)				\
    ustrlcat((d), sizeof(d), (s1), (s2), (s3), (s4), (s5), NULL)

#define AUSTRLCPY2(d,s1,s2)					\
    ustrlcpy((d), sizeof(d), (s1), (s2), NULL)
#define AUSTRLCPY3(d,s1,s2,s3)					\
    ustrlcpy((d), sizeof(d), (s1), (s2), (s3), NULL)
#define AUSTRLCPY4(d,s1,s2,s3,s4)				\
    ustrlcpy((d), sizeof(d), (s1), (s2), (s3), (s4), NULL)
#define AUSTRLCPY5(d,s1,s2,s3,s4,s5)				\
    ustrlcpy((d), sizeof(d), (s1), (s2), (s3), (s4), (s5), NULL)

/* Pointer + size macros */

#define PUSTRCPY(d,n,s)		ustrcpy((d), (n), (s))
#define PUSTRCAT(d,n,s)		ustrcat((d), (n), (s))

#define PUSTRLCAT2(d,n,s1,s2)					\
    ustrlcat((d), (n), (s1), (s2), NULL)
#define PUSTRLCAT3(d,n,s1,s2,s3)				\
    ustrlcat((d), (n), (s1), (s2), (s3), NULL)
#define PUSTRLCAT4(d,n,s1,s2,s3,s4)				\
    ustrlcat((d), (n), (s1), (s2), (s3), (s4), NULL)
#define PUSTRLCAT5(d,n,s1,s2,s3,s4,s5)				\
    ustrlcat((d), (n), (s1), (s2), (s3), (s4), (s5), NULL)

#define PUSTRLCPY2(d,n,s1,s2)					\
    ustrlcpy((d), (n), (s1), (s2), NULL)
#define PUSTRLCPY3(d,n,s1,s2,s3)				\
    ustrlcpy((d), (n), (s1), (s2), (s3), NULL)
#define PUSTRLCPY4(d,n,s1,s2,s3,s4)				\
    ustrlcpy((d), (n), (s1), (s2), (s3), (s4), NULL)
#define PUSTRLCPY5(d,n,s1,s2,s3,s4,s5)				\
    ustrlcpy((d), (n), (s1), (s2), (s3), (s4), (s5), NULL)


#endif	/* REPORT_OVERRUNS */
